// Copyright (c) 2020-2022  David Sorokin <david.sorokin@gmail.com>, based in Yoshkar-Ola, Russia
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use crate::simulation::event::*;
use crate::simulation::branch::*;

/// Forecast the average value of computation by binary branching nested simulations
/// provided the specified branching depth is defined.
pub fn binary_forecast_by_depth<M>(comp: M, branching_depth: usize) -> impl Event<Item = f64>
    where M: Event<Item = f64> + Clone
{
    cons_event(move |p| {
        Result::Ok((p.run.specs.stop_time - p.time) / (branching_depth as f64))
    }).flat_map(move |time_dt| {
        binary_forecast_by_dt(comp, time_dt)
    })
}

/// Forecast the average value of computation by binary branching nested simulations
/// provided the specified time shift (delta) is defined.
pub fn binary_forecast_by_dt<M>(comp: M, time_dt: f64) -> impl Event<Item = f64>
    where M: Event<Item = f64> + Clone
{
    cons_event(move |p| {
        if p.time >= p.run.specs.stop_time {
            comp.call_event(p)
        } else {
            let comp = {
                future_event(p.time + time_dt, binary_forecast_by_dt(comp.clone(), time_dt), true)
                    .flat_map(move |x1| {
                        future_event(p.time + time_dt, binary_forecast_by_dt(comp, time_dt), true)
                            .map(move |x2| {
                                (x1 + x2) / 2.0
                            })
                    })
            };
            comp.call_event(p)
        }
    })
}
