//! This module provides an implementation of the $\mathbb{G}_2$ group of BLS12-381.

use crate::fp::Fp;
use crate::fp2::Fp2;
use crate::BlsScalar;

use core::borrow::Borrow;
use core::iter::Sum;
use core::ops::{Add, AddAssign, Mul, MulAssign, Neg, Sub, SubAssign};
use dusk_bytes::{Error as BytesError, HexDebug, Serializable};
use subtle::{Choice, ConditionallySelectable, ConstantTimeEq, CtOption};

#[cfg(feature = "canon")]
use canonical::{Canon, CanonError, Sink, Source};
#[cfg(feature = "canon")]
use canonical_derive::Canon;
#[cfg(feature = "serde_req")]
use serde::{de::Visitor, Deserialize, Deserializer, Serialize, Serializer};

/// This is an element of $\mathbb{G}_2$ represented in the affine coordinate space.
/// It is ideal to keep elements in this representation to reduce memory usage and
/// improve performance through the use of mixed curve model arithmetic.
///
/// Values of `G2Affine` are guaranteed to be in the $q$-order subgroup unless an
/// "unchecked" API was misused.
#[derive(Copy, Clone, HexDebug)]
pub struct G2Affine {
    pub(crate) x: Fp2,
    pub(crate) y: Fp2,
    infinity: Choice,
}

#[cfg(feature = "canon")]
impl Canon for G2Affine {
    fn encode(&self, sink: &mut Sink) {
        sink.copy_bytes(&self.to_bytes());
    }

    fn decode(source: &mut Source) -> Result<Self, CanonError> {
        let mut bytes = [0u8; Self::SIZE];

        bytes.copy_from_slice(source.read_bytes(Self::SIZE));

        Self::from_bytes(&bytes).map_err(|_| CanonError::InvalidEncoding)
    }

    fn encoded_len(&self) -> usize {
        Self::SIZE
    }
}

impl Default for G2Affine {
    fn default() -> G2Affine {
        G2Affine::identity()
    }
}

impl<'a> From<&'a G2Projective> for G2Affine {
    fn from(p: &'a G2Projective) -> G2Affine {
        let zinv = p.z.invert().unwrap_or(Fp2::zero());
        let zinv2 = zinv.square();
        let x = p.x * zinv2;
        let zinv3 = zinv2 * zinv;
        let y = p.y * zinv3;

        let tmp = G2Affine {
            x,
            y,
            infinity: Choice::from(0u8),
        };

        G2Affine::conditional_select(&tmp, &G2Affine::identity(), zinv.is_zero())
    }
}

impl From<G2Projective> for G2Affine {
    fn from(p: G2Projective) -> G2Affine {
        G2Affine::from(&p)
    }
}

impl ConstantTimeEq for G2Affine {
    fn ct_eq(&self, other: &Self) -> Choice {
        // The only cases in which two points are equal are
        // 1. infinity is set on both
        // 2. infinity is not set on both, and their coordinates are equal

        (self.infinity & other.infinity)
            | ((!self.infinity)
                & (!other.infinity)
                & self.x.ct_eq(&other.x)
                & self.y.ct_eq(&other.y))
    }
}

impl ConditionallySelectable for G2Affine {
    fn conditional_select(a: &Self, b: &Self, choice: Choice) -> Self {
        G2Affine {
            x: Fp2::conditional_select(&a.x, &b.x, choice),
            y: Fp2::conditional_select(&a.y, &b.y, choice),
            infinity: Choice::conditional_select(&a.infinity, &b.infinity, choice),
        }
    }
}

impl Eq for G2Affine {}
impl PartialEq for G2Affine {
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        bool::from(self.ct_eq(other))
    }
}

impl Serializable<96> for G2Affine {
    type Error = BytesError;

    /// Serializes this element into compressed form. See [`notes::serialization`](crate::notes::serialization)
    /// for details about how group elements are serialized.
    fn to_bytes(&self) -> [u8; Self::SIZE] {
        // Strictly speaking, self.x is zero already when self.infinity is true, but
        // to guard against implementation mistakes we do not assume this.
        let x = Fp2::conditional_select(&self.x, &Fp2::zero(), self.infinity);

        let mut res = [0; Self::SIZE];

        (&mut res[0..48]).copy_from_slice(&x.c1.to_bytes()[..]);
        (&mut res[48..96]).copy_from_slice(&x.c0.to_bytes()[..]);

        // This point is in compressed form, so we set the most significant bit.
        res[0] |= 1u8 << 7;

        // Is this point at infinity? If so, set the second-most significant bit.
        res[0] |= u8::conditional_select(&0u8, &(1u8 << 6), self.infinity);

        // Is the y-coordinate the lexicographically largest of the two associated with the
        // x-coordinate? If so, set the third-most significant bit so long as this is not
        // the point at infinity.
        res[0] |= u8::conditional_select(
            &0u8,
            &(1u8 << 5),
            (!self.infinity) & self.y.lexicographically_largest(),
        );

        res
    }

    /// Attempts to deserialize a compressed element. See [`notes::serialization`](crate::notes::serialization)
    /// for details about how group elements are serialized.
    fn from_bytes(buf: &[u8; Self::SIZE]) -> Result<Self, Self::Error> {
        // We already know the point is on the curve because this is established
        // by the y-coordinate recovery procedure in from_compressed_unchecked().

        // Obtain the three flags from the start of the byte sequence
        let compression_flag_set = Choice::from((buf[0] >> 7) & 1);
        let infinity_flag_set = Choice::from((buf[0] >> 6) & 1);
        let sort_flag_set = Choice::from((buf[0] >> 5) & 1);

        // Attempt to obtain the x-coordinate
        let xc1 = {
            let mut tmp = [0; 48];
            tmp.copy_from_slice(&buf[0..48]);

            // Mask away the flag bits
            tmp[0] &= 0b0001_1111;

            Fp::from_bytes(&tmp)
        };
        let xc0 = {
            let mut tmp = [0; 48];
            tmp.copy_from_slice(&buf[48..96]);

            Fp::from_bytes(&tmp)
        };

        let x: Option<Self> = xc1
            .and_then(|xc1| {
                xc0.and_then(|xc0| {
                    let x = Fp2 { c0: xc0, c1: xc1 };

                    // If the infinity flag is set, return the value assuming
                    // the x-coordinate is zero and the sort bit is not set.
                    //
                    // Otherwise, return a recovered point (assuming the correct
                    // y-coordinate can be found) so long as the infinity flag
                    // was not set.
                    CtOption::new(
                        G2Affine::identity(),
                        infinity_flag_set & // Infinity flag should be set
                    compression_flag_set & // Compression flag should be set
                    (!sort_flag_set) & // Sort flag should not be set
                    x.is_zero(), // The x-coordinate should be zero
                    )
                    .or_else(|| {
                        // Recover a y-coordinate given x by y = sqrt(x^3 + 4)
                        ((x.square() * x) + B).sqrt().and_then(|y| {
                            // Switch to the correct y-coordinate if necessary.
                            let y = Fp2::conditional_select(
                                &y,
                                &-y,
                                y.lexicographically_largest() ^ sort_flag_set,
                            );

                            CtOption::new(
                                G2Affine {
                                    x,
                                    y,
                                    infinity: infinity_flag_set,
                                },
                                (!infinity_flag_set) & // Infinity flag should not be set
                            compression_flag_set, // Compression flag should be set
                            )
                        })
                    })
                })
            })
            .into();

        match x {
            Some(x) if x.is_torsion_free().unwrap_u8() == 1 => Ok(x),
            _ => Err(BytesError::InvalidData),
        }
    }
}

#[cfg(feature = "serde_req")]
impl Serialize for G2Affine {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        use serde::ser::SerializeTuple;
        let mut tup = serializer.serialize_tuple(Self::SIZE)?;
        for byte in self.to_bytes().iter() {
            tup.serialize_element(byte)?;
        }
        tup.end()
    }
}

#[cfg(feature = "serde_req")]
impl<'de> Deserialize<'de> for G2Affine {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct G2AffineVisitor;

        impl<'de> Visitor<'de> for G2AffineVisitor {
            type Value = G2Affine;

            fn expecting(&self, formatter: &mut ::core::fmt::Formatter) -> ::core::fmt::Result {
                formatter.write_str("a 48-byte cannonical compressed G2Affine point from Bls12_381")
            }

            fn visit_seq<A>(self, mut seq: A) -> Result<G2Affine, A::Error>
            where
                A: serde::de::SeqAccess<'de>,
            {
                let mut bytes = [0u8; G2Affine::SIZE];
                for i in 0..G2Affine::SIZE {
                    bytes[i] = seq
                        .next_element()?
                        .ok_or(serde::de::Error::invalid_length(i, &"expected 48 bytes"))?;
                }

                G2Affine::from_bytes(&bytes).map_err(|_| {
                    serde::de::Error::custom(&"compressed G2Affine was not canonically encoded")
                })
            }
        }

        deserializer.deserialize_tuple(Self::SIZE, G2AffineVisitor)
    }
}

impl<'a> Neg for &'a G2Affine {
    type Output = G2Affine;

    #[inline]
    fn neg(self) -> G2Affine {
        G2Affine {
            x: self.x,
            y: Fp2::conditional_select(&-self.y, &Fp2::one(), self.infinity),
            infinity: self.infinity,
        }
    }
}

impl Neg for G2Affine {
    type Output = G2Affine;

    #[inline]
    fn neg(self) -> G2Affine {
        -&self
    }
}

impl<'a, 'b> Add<&'b G2Projective> for &'a G2Affine {
    type Output = G2Projective;

    #[inline]
    fn add(self, rhs: &'b G2Projective) -> G2Projective {
        rhs.add_mixed(self)
    }
}

impl<'a, 'b> Add<&'b G2Affine> for &'a G2Projective {
    type Output = G2Projective;

    #[inline]
    fn add(self, rhs: &'b G2Affine) -> G2Projective {
        self.add_mixed(rhs)
    }
}

impl<'a, 'b> Sub<&'b G2Projective> for &'a G2Affine {
    type Output = G2Projective;

    #[inline]
    fn sub(self, rhs: &'b G2Projective) -> G2Projective {
        self + (-rhs)
    }
}

impl<'a, 'b> Sub<&'b G2Affine> for &'a G2Projective {
    type Output = G2Projective;

    #[inline]
    fn sub(self, rhs: &'b G2Affine) -> G2Projective {
        self + (-rhs)
    }
}

impl<T> Sum<T> for G2Projective
where
    T: Borrow<G2Projective>,
{
    fn sum<I>(iter: I) -> Self
    where
        I: Iterator<Item = T>,
    {
        iter.fold(Self::identity(), |acc, item| acc + item.borrow())
    }
}

impl_binops_additive!(G2Projective, G2Affine);
impl_binops_additive_specify_output!(G2Affine, G2Projective, G2Projective);

const B: Fp2 = Fp2 {
    c0: Fp::from_raw_unchecked([
        0xaa270000000cfff3,
        0x53cc0032fc34000a,
        0x478fe97a6b0a807f,
        0xb1d37ebee6ba24d7,
        0x8ec9733bbf78ab2f,
        0x9d645513d83de7e,
    ]),
    c1: Fp::from_raw_unchecked([
        0xaa270000000cfff3,
        0x53cc0032fc34000a,
        0x478fe97a6b0a807f,
        0xb1d37ebee6ba24d7,
        0x8ec9733bbf78ab2f,
        0x9d645513d83de7e,
    ]),
};

impl G2Affine {
    /// Bytes size of the raw representation
    pub const RAW_SIZE: usize = 193;

    /// Returns the identity of the group: the point at infinity.
    pub fn identity() -> G2Affine {
        G2Affine {
            x: Fp2::zero(),
            y: Fp2::one(),
            infinity: Choice::from(1u8),
        }
    }

    /// Returns a fixed generator of the group. See [`notes::design`](notes/design/index.html#fixed-generators)
    /// for how this generator is chosen.
    pub fn generator() -> G2Affine {
        G2Affine {
            x: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0xf5f28fa202940a10,
                    0xb3f5fb2687b4961a,
                    0xa1a893b53e2ae580,
                    0x9894999d1a3caee9,
                    0x6f67b7631863366b,
                    0x58191924350bcd7,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xa5a9c0759e23f606,
                    0xaaa0c59dbccd60c3,
                    0x3bb17e18e2867806,
                    0x1b1ab6cc8541b367,
                    0xc2b6ed0ef2158547,
                    0x11922a097360edf3,
                ]),
            },
            y: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x4c730af860494c4a,
                    0x597cfa1f5e369c5a,
                    0xe7e6856caa0a635a,
                    0xbbefb5e96e0d495f,
                    0x7d3a975f0ef25a2,
                    0x83fd8e7e80dae5,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xadc0fc92df64b05d,
                    0x18aa270a2b1461dc,
                    0x86adac6a3be4eba0,
                    0x79495c4ec93da33a,
                    0xe7175850a43ccaed,
                    0xb2bc2a163de1bf2,
                ]),
            },
            infinity: Choice::from(0u8),
        }
    }

    /// Raw bytes representation
    ///
    /// The intended usage of this function is for trusted sets of data where performance is
    /// critical.
    ///
    /// For secure serialization, check `to_bytes`
    pub fn to_raw_bytes(&self) -> [u8; Self::RAW_SIZE] {
        let mut bytes = [0u8; Self::RAW_SIZE];
        let chunks = bytes.chunks_mut(8);

        self.x
            .c0
            .internal_repr()
            .iter()
            .chain(self.x.c1.internal_repr().iter())
            .chain(self.y.c0.internal_repr().iter())
            .chain(self.y.c1.internal_repr().iter())
            .zip(chunks)
            .for_each(|(n, c)| c.copy_from_slice(&n.to_le_bytes()));

        bytes[Self::RAW_SIZE - 1] = self.infinity.unwrap_u8();

        bytes
    }

    /// Create a `G2Affine` from a set of bytes created by `G2Affine::to_raw_bytes`.
    ///
    /// No check is performed and no constant time is granted. The expected usage of this function
    /// is for trusted bytes where performance is critical.
    ///
    /// For secure serialization, check `from_bytes`
    ///
    /// After generating the point, you can check `is_on_curve` and `is_torsion_free` to grant its
    /// security
    pub unsafe fn from_slice_unchecked(bytes: &[u8]) -> Self {
        let mut xc0 = [0u64; 6];
        let mut xc1 = [0u64; 6];
        let mut yc0 = [0u64; 6];
        let mut yc1 = [0u64; 6];
        let mut z = [0u8; 8];

        xc0.iter_mut()
            .chain(xc1.iter_mut())
            .chain(yc0.iter_mut())
            .chain(yc1.iter_mut())
            .zip(bytes.chunks_exact(8))
            .for_each(|(n, c)| {
                z.copy_from_slice(c);
                *n = u64::from_le_bytes(z);
            });

        let c0 = Fp::from_raw_unchecked(xc0);
        let c1 = Fp::from_raw_unchecked(xc1);
        let x = Fp2 { c0, c1 };

        let c0 = Fp::from_raw_unchecked(yc0);
        let c1 = Fp::from_raw_unchecked(yc1);
        let y = Fp2 { c0, c1 };

        let infinity = if bytes.len() >= Self::RAW_SIZE {
            bytes[Self::RAW_SIZE - 1].into()
        } else {
            0u8.into()
        };

        Self { x, y, infinity }
    }

    /// Returns true if this element is the identity (the point at infinity).
    #[inline]
    pub fn is_identity(&self) -> Choice {
        self.infinity
    }

    /// Returns true if this point is free of an $h$-torsion component, and so it
    /// exists within the $q$-order subgroup $\mathbb{G}_2$. This should always return true
    /// unless an "unchecked" API was used.
    pub fn is_torsion_free(&self) -> Choice {
        const FQ_MODULUS_BYTES: [u8; 32] = [
            1, 0, 0, 0, 255, 255, 255, 255, 254, 91, 254, 255, 2, 164, 189, 83, 5, 216, 161, 9, 8,
            216, 57, 51, 72, 125, 157, 41, 83, 167, 237, 115,
        ];

        // Clear the r-torsion from the point and check if it is the identity
        G2Projective::from(*self)
            .multiply(&FQ_MODULUS_BYTES)
            .is_identity()
    }

    /// Returns true if this point is on the curve. This should always return
    /// true unless an "unchecked" API was used.
    pub fn is_on_curve(&self) -> Choice {
        // y^2 - x^3 ?= 4(u + 1)
        (self.y.square() - (self.x.square() * self.x)).ct_eq(&B) | self.infinity
    }
}

/// This is an element of $\mathbb{G}_2$ represented in the projective coordinate space.
#[derive(Copy, Clone, Debug)]
#[cfg_attr(feature = "canon", derive(Canon))]
pub struct G2Projective {
    pub(crate) x: Fp2,
    pub(crate) y: Fp2,
    pub(crate) z: Fp2,
}

impl<'a> From<&'a G2Affine> for G2Projective {
    fn from(p: &'a G2Affine) -> G2Projective {
        G2Projective {
            x: p.x,
            y: p.y,
            z: Fp2::conditional_select(&Fp2::one(), &Fp2::zero(), p.infinity),
        }
    }
}

impl From<G2Affine> for G2Projective {
    fn from(p: G2Affine) -> G2Projective {
        G2Projective::from(&p)
    }
}

impl ConstantTimeEq for G2Projective {
    fn ct_eq(&self, other: &Self) -> Choice {
        // Is (xz^2, yz^3, z) equal to (x'z'^2, yz'^3, z') when converted to affine?

        let z = other.z.square();
        let x1 = self.x * z;
        let z = z * other.z;
        let y1 = self.y * z;
        let z = self.z.square();
        let x2 = other.x * z;
        let z = z * self.z;
        let y2 = other.y * z;

        let self_is_zero = self.z.is_zero();
        let other_is_zero = other.z.is_zero();

        (self_is_zero & other_is_zero) // Both point at infinity
            | ((!self_is_zero) & (!other_is_zero) & x1.ct_eq(&x2) & y1.ct_eq(&y2))
        // Neither point at infinity, coordinates are the same
    }
}

impl ConditionallySelectable for G2Projective {
    fn conditional_select(a: &Self, b: &Self, choice: Choice) -> Self {
        G2Projective {
            x: Fp2::conditional_select(&a.x, &b.x, choice),
            y: Fp2::conditional_select(&a.y, &b.y, choice),
            z: Fp2::conditional_select(&a.z, &b.z, choice),
        }
    }
}

impl Eq for G2Projective {}
impl PartialEq for G2Projective {
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        bool::from(self.ct_eq(other))
    }
}

impl<'a> Neg for &'a G2Projective {
    type Output = G2Projective;

    #[inline]
    fn neg(self) -> G2Projective {
        G2Projective {
            x: self.x,
            y: -self.y,
            z: self.z,
        }
    }
}

impl Neg for G2Projective {
    type Output = G2Projective;

    #[inline]
    fn neg(self) -> G2Projective {
        -&self
    }
}

impl<'a, 'b> Add<&'b G2Projective> for &'a G2Projective {
    type Output = G2Projective;

    #[inline]
    fn add(self, rhs: &'b G2Projective) -> G2Projective {
        self.add(rhs)
    }
}

impl<'a, 'b> Sub<&'b G2Projective> for &'a G2Projective {
    type Output = G2Projective;

    #[inline]
    fn sub(self, rhs: &'b G2Projective) -> G2Projective {
        self + (-rhs)
    }
}

impl<'a, 'b> Mul<&'b BlsScalar> for &'a G2Projective {
    type Output = G2Projective;

    fn mul(self, other: &'b BlsScalar) -> Self::Output {
        self.multiply(&other.to_bytes())
    }
}

impl<'a, 'b> Mul<&'b BlsScalar> for &'a G2Affine {
    type Output = G2Projective;

    fn mul(self, other: &'b BlsScalar) -> Self::Output {
        G2Projective::from(self).multiply(&other.to_bytes())
    }
}

impl_binops_additive!(G2Projective, G2Projective);
impl_binops_multiplicative!(G2Projective, BlsScalar);
impl_binops_multiplicative_mixed!(G2Affine, BlsScalar, G2Projective);

impl G2Projective {
    /// Returns the identity of the group: the point at infinity.
    pub fn identity() -> G2Projective {
        G2Projective {
            x: Fp2::zero(),
            y: Fp2::one(),
            z: Fp2::zero(),
        }
    }

    /// Returns a fixed generator of the group. See [`notes::design`](notes/design/index.html#fixed-generators)
    /// for how this generator is chosen.
    pub fn generator() -> G2Projective {
        G2Projective {
            x: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0xf5f28fa202940a10,
                    0xb3f5fb2687b4961a,
                    0xa1a893b53e2ae580,
                    0x9894999d1a3caee9,
                    0x6f67b7631863366b,
                    0x58191924350bcd7,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xa5a9c0759e23f606,
                    0xaaa0c59dbccd60c3,
                    0x3bb17e18e2867806,
                    0x1b1ab6cc8541b367,
                    0xc2b6ed0ef2158547,
                    0x11922a097360edf3,
                ]),
            },
            y: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x4c730af860494c4a,
                    0x597cfa1f5e369c5a,
                    0xe7e6856caa0a635a,
                    0xbbefb5e96e0d495f,
                    0x7d3a975f0ef25a2,
                    0x83fd8e7e80dae5,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xadc0fc92df64b05d,
                    0x18aa270a2b1461dc,
                    0x86adac6a3be4eba0,
                    0x79495c4ec93da33a,
                    0xe7175850a43ccaed,
                    0xb2bc2a163de1bf2,
                ]),
            },
            z: Fp2::one(),
        }
    }

    /// Computes the doubling of this point.
    pub fn double(&self) -> G2Projective {
        // http://www.hyperelliptic.org/EFD/g2p/auto-shortw-jacobian-0.html#doubling-dbl-2009-l
        //
        // There are no points of order 2.

        let a = self.x.square();
        let b = self.y.square();
        let c = b.square();
        let d = self.x + b;
        let d = d.square();
        let d = d - a - c;
        let d = d + d;
        let e = a + a + a;
        let f = e.square();
        let z3 = self.z * self.y;
        let z3 = z3 + z3;
        let x3 = f - (d + d);
        let c = c + c;
        let c = c + c;
        let c = c + c;
        let y3 = e * (d - x3) - c;

        let tmp = G2Projective {
            x: x3,
            y: y3,
            z: z3,
        };

        G2Projective::conditional_select(&tmp, &G2Projective::identity(), self.is_identity())
    }

    /// Adds this point to another point.
    pub fn add(&self, rhs: &G2Projective) -> G2Projective {
        // This Jacobian point addition technique is based on the implementation in libsecp256k1,
        // which assumes that rhs has z=1. Let's address the case of zero z-coordinates generally.

        // If self is the identity, return rhs. Otherwise, return self. The other cases will be
        // predicated on neither self nor rhs being the identity.
        let f1 = self.is_identity();
        let res = G2Projective::conditional_select(self, rhs, f1);
        let f2 = rhs.is_identity();

        // If neither are the identity but x1 = x2 and y1 != y2, then return the identity
        let z = rhs.z.square();
        let u1 = self.x * z;
        let z = z * rhs.z;
        let s1 = self.y * z;
        let z = self.z.square();
        let u2 = rhs.x * z;
        let z = z * self.z;
        let s2 = rhs.y * z;
        let f3 = u1.ct_eq(&u2) & (!s1.ct_eq(&s2));
        let res =
            G2Projective::conditional_select(&res, &G2Projective::identity(), (!f1) & (!f2) & f3);

        let t = u1 + u2;
        let m = s1 + s2;
        let rr = t.square();
        let m_alt = -u2;
        let tt = u1 * m_alt;
        let rr = rr + tt;

        // Correct for x1 != x2 but y1 = -y2, which can occur because p - 1 is divisible by 3.
        // libsecp256k1 does this by substituting in an alternative (defined) expression for lambda.
        let degenerate = m.is_zero() & rr.is_zero();
        let rr_alt = s1 + s1;
        let m_alt = m_alt + u1;
        let rr_alt = Fp2::conditional_select(&rr_alt, &rr, !degenerate);
        let m_alt = Fp2::conditional_select(&m_alt, &m, !degenerate);

        let n = m_alt.square();
        let q = n * t;

        let n = n.square();
        let n = Fp2::conditional_select(&n, &m, degenerate);
        let t = rr_alt.square();
        let z3 = m_alt * self.z * rhs.z; // We allow rhs.z != 1, so we must account for this.
        let z3 = z3 + z3;
        let q = -q;
        let t = t + q;
        let x3 = t;
        let t = t + t;
        let t = t + q;
        let t = t * rr_alt;
        let t = t + n;
        let y3 = -t;
        let x3 = x3 + x3;
        let x3 = x3 + x3;
        let y3 = y3 + y3;
        let y3 = y3 + y3;

        let tmp = G2Projective {
            x: x3,
            y: y3,
            z: z3,
        };

        G2Projective::conditional_select(&res, &tmp, (!f1) & (!f2) & (!f3))
    }

    /// Adds this point to another point in the affine model.
    pub fn add_mixed(&self, rhs: &G2Affine) -> G2Projective {
        // This Jacobian point addition technique is based on the implementation in libsecp256k1,
        // which assumes that rhs has z=1. Let's address the case of zero z-coordinates generally.

        // If self is the identity, return rhs. Otherwise, return self. The other cases will be
        // predicated on neither self nor rhs being the identity.
        let f1 = self.is_identity();
        let res = G2Projective::conditional_select(self, &G2Projective::from(rhs), f1);
        let f2 = rhs.is_identity();

        // If neither are the identity but x1 = x2 and y1 != y2, then return the identity
        let u1 = self.x;
        let s1 = self.y;
        let z = self.z.square();
        let u2 = rhs.x * z;
        let z = z * self.z;
        let s2 = rhs.y * z;
        let f3 = u1.ct_eq(&u2) & (!s1.ct_eq(&s2));
        let res =
            G2Projective::conditional_select(&res, &G2Projective::identity(), (!f1) & (!f2) & f3);

        let t = u1 + u2;
        let m = s1 + s2;
        let rr = t.square();
        let m_alt = -u2;
        let tt = u1 * m_alt;
        let rr = rr + tt;

        // Correct for x1 != x2 but y1 = -y2, which can occur because p - 1 is divisible by 3.
        // libsecp256k1 does this by substituting in an alternative (defined) expression for lambda.
        let degenerate = m.is_zero() & rr.is_zero();
        let rr_alt = s1 + s1;
        let m_alt = m_alt + u1;
        let rr_alt = Fp2::conditional_select(&rr_alt, &rr, !degenerate);
        let m_alt = Fp2::conditional_select(&m_alt, &m, !degenerate);

        let n = m_alt.square();
        let q = n * t;

        let n = n.square();
        let n = Fp2::conditional_select(&n, &m, degenerate);
        let t = rr_alt.square();
        let z3 = m_alt * self.z;
        let z3 = z3 + z3;
        let q = -q;
        let t = t + q;
        let x3 = t;
        let t = t + t;
        let t = t + q;
        let t = t * rr_alt;
        let t = t + n;
        let y3 = -t;
        let x3 = x3 + x3;
        let x3 = x3 + x3;
        let y3 = y3 + y3;
        let y3 = y3 + y3;

        let tmp = G2Projective {
            x: x3,
            y: y3,
            z: z3,
        };

        G2Projective::conditional_select(&res, &tmp, (!f1) & (!f2) & (!f3))
    }

    fn multiply(&self, by: &[u8]) -> G2Projective {
        let mut acc = G2Projective::identity();

        // This is a simple double-and-add implementation of point
        // multiplication, moving from most significant to least
        // significant bit of the scalar.
        //
        // We skip the leading bit because it's always unset for Fq
        // elements.
        for bit in by
            .iter()
            .rev()
            .flat_map(|byte| (0..8).rev().map(move |i| Choice::from((byte >> i) & 1u8)))
            .skip(1)
        {
            acc = acc.double();
            acc = G2Projective::conditional_select(&acc, &(acc + self), bit);
        }

        acc
    }

    #[cfg(feature = "endo")]
    fn psi(&self) -> G2Projective {
        // 1 / ((u+1) ^ ((q-1)/3))
        let psi_coeff_x = Fp2 {
            c0: Fp::zero(),
            c1: Fp::from_raw_unchecked([
                0x890dc9e4867545c3,
                0x2af322533285a5d5,
                0x50880866309b7e2c,
                0xa20d1b8c7e881024,
                0x14e4f04fe2db9068,
                0x14e56d3f1564853a,
            ]),
        };
        // 1 / ((u+1) ^ (p-1)/2)
        let psi_coeff_y = Fp2 {
            c0: Fp::from_raw_unchecked([
                0x3e2f585da55c9ad1,
                0x4294213d86c18183,
                0x382844c88b623732,
                0x92ad2afd19103e18,
                0x1d794e4fac7cf0b9,
                0x0bd592fc7d825ec8,
            ]),
            c1: Fp::from_raw_unchecked([
                0x7bcfa7a25aa30fda,
                0xdc17dec12a927e7c,
                0x2f088dd86b4ebef1,
                0xd1ca2087da74d4a7,
                0x2da2596696cebc1d,
                0x0e2b7eedbbfd87d2,
            ]),
        };

        G2Projective {
            // x = frobenius(x)/((u+1)^((p-1)/3))
            x: self.x.frobenius_map() * psi_coeff_x,
            // y = frobenius(y)/(u+1)^((p-1)/2)
            y: self.y.frobenius_map() * psi_coeff_y,
            // z = frobenius(z)
            z: self.z.frobenius_map(),
        }
    }

    #[cfg(feature = "endo")]
    fn psi2(&self) -> G2Projective {
        // 1 / 2 ^ ((q-1)/3)
        let psi2_coeff_x = Fp2 {
            c0: Fp::from_raw_unchecked([
                0xcd03c9e48671f071,
                0x5dab22461fcda5d2,
                0x587042afd3851b95,
                0x8eb60ebe01bacb9e,
                0x03f97d6e83d050d2,
                0x18f0206554638741,
            ]),
            c1: Fp::zero(),
        };

        G2Projective {
            // x = frobenius^2(x)/2^((p-1)/3)
            x: self.x.frobenius_map().frobenius_map() * psi2_coeff_x,
            // y = -frobenius^2(y)
            y: self.y.frobenius_map().frobenius_map().neg(),
            // z = z
            z: self.z,
        }
    }

    /// Multiply `self` by `crate::BLS_X`, using double and add.
    #[cfg(feature = "endo")]
    fn mul_by_x(&self) -> G2Projective {
        let mut xself = G2Projective::identity();
        // NOTE: in BLS12-381 we can just skip the first bit.
        let mut x = crate::BLS_X >> 1;
        let mut acc = *self;
        while x != 0 {
            acc = acc.double();
            if x % 2 == 1 {
                xself += acc;
            }
            x >>= 1;
        }
        // finally, flip the sign
        if crate::BLS_X_IS_NEGATIVE {
            xself = -xself;
        }
        xself
    }

    /// Clears the cofactor, using [Budroni-Pintore](https://ia.cr/2017/419).
    /// This is equivalent to multiplying by $h\_\textrm{eff} = 3(z^2 - 1) \cdot
    /// h_2$, where $h_2$ is the cofactor of $\mathbb{G}\_2$ and $z$ is the
    /// parameter of BLS12-381.
    ///
    /// The endomorphism is only actually used if the crate feature `endo` is
    /// enabled, and it is disabled by default to mitigate potential patent
    /// issues.
    pub fn clear_cofactor(&self) -> G2Projective {
        #[cfg(feature = "endo")]
        fn clear_cofactor(this: &G2Projective) -> G2Projective {
            let t1 = this.mul_by_x(); // [x] P
            let t2 = this.psi(); // psi(P)

            this.double().psi2() // psi^2(2P)
                + (t1 + t2).mul_by_x() // psi^2(2P) + [x^2] P + [x] psi(P)
                - t1 // psi^2(2P) + [x^2 - x] P + [x] psi(P)
                - t2 // psi^2(2P) + [x^2 - x] P + [x - 1] psi(P)
                - this // psi^2(2P) + [x^2 - x - 1] P + [x - 1] psi(P)
        }

        #[cfg(not(feature = "endo"))]
        fn clear_cofactor(this: &G2Projective) -> G2Projective {
            this.multiply(&[
                0x51, 0x55, 0xa9, 0xaa, 0x5, 0x0, 0x2, 0xe8, 0xb4, 0xf6, 0xbb, 0xde, 0xa, 0x4c,
                0x89, 0x59, 0xa3, 0xf6, 0x89, 0x66, 0xc0, 0xcb, 0x54, 0xe9, 0x1a, 0x7c, 0x47, 0xd7,
                0x69, 0xec, 0xc0, 0x2e, 0xb0, 0x12, 0x12, 0x5d, 0x1, 0xbf, 0x82, 0x6d, 0x95, 0xdb,
                0x31, 0x87, 0x17, 0x2f, 0x9c, 0x32, 0xe1, 0xff, 0x8, 0x15, 0x3, 0xff, 0x86, 0x99,
                0x68, 0xd7, 0x5a, 0x14, 0xe9, 0xa8, 0xe2, 0x88, 0x28, 0x35, 0x1b, 0xa9, 0xe, 0x6a,
                0x4c, 0x58, 0xb3, 0x75, 0xee, 0xf2, 0x8, 0x9f, 0xc6, 0xb,
            ])
        }

        clear_cofactor(self)
    }

    /// Converts a batch of `G2Projective` elements into `G2Affine` elements. This
    /// function will panic if `p.len() != q.len()`.
    pub fn batch_normalize(p: &[Self], q: &mut [G2Affine]) {
        assert_eq!(p.len(), q.len());

        let mut acc = Fp2::one();
        for (p, q) in p.iter().zip(q.iter_mut()) {
            // We use the `x` field of `G2Affine` to store the product
            // of previous z-coordinates seen.
            q.x = acc;

            // We will end up skipping all identities in p
            acc = Fp2::conditional_select(&(acc * p.z), &acc, p.is_identity());
        }

        // This is the inverse, as all z-coordinates are nonzero and the ones
        // that are not are skipped.
        acc = acc.invert().unwrap();

        for (p, q) in p.iter().rev().zip(q.iter_mut().rev()) {
            let skip = p.is_identity();

            // Compute tmp = 1/z
            let tmp = q.x * acc;

            // Cancel out z-coordinate in denominator of `acc`
            acc = Fp2::conditional_select(&(acc * p.z), &acc, skip);

            // Set the coordinates to the correct value
            let tmp2 = tmp.square();
            let tmp3 = tmp2 * tmp;

            q.x = p.x * tmp2;
            q.y = p.y * tmp3;
            q.infinity = Choice::from(0u8);

            *q = G2Affine::conditional_select(&q, &G2Affine::identity(), skip);
        }
    }

    /// Returns true if this element is the identity (the point at infinity).
    #[inline]
    pub fn is_identity(&self) -> Choice {
        self.z.is_zero()
    }

    /// Returns true if this point is on the curve. This should always return
    /// true unless an "unchecked" API was used.
    pub fn is_on_curve(&self) -> Choice {
        // Y^2 - X^3 = 4(u + 1)(Z^6)

        (self.y.square() - (self.x.square() * self.x))
            .ct_eq(&((self.z.square() * self.z).square() * B))
            | self.z.is_zero()
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_is_on_curve() {
        assert!(bool::from(G2Affine::identity().is_on_curve()));
        assert!(bool::from(G2Affine::generator().is_on_curve()));
        assert!(bool::from(G2Projective::identity().is_on_curve()));
        assert!(bool::from(G2Projective::generator().is_on_curve()));

        let z = Fp2 {
            c0: Fp::from_raw_unchecked([
                0xba7afa1f9a6fe250,
                0xfa0f5b595eafe731,
                0x3bdc477694c306e7,
                0x2149be4b3949fa24,
                0x64aa6e0649b2078c,
                0x12b108ac33643c3e,
            ]),
            c1: Fp::from_raw_unchecked([
                0x125325df3d35b5a8,
                0xdc469ef5555d7fe3,
                0x2d716d2443106a9,
                0x5a1db59a6ff37d0,
                0x7cf7784e5300bb8f,
                0x16a88922c7a5e844,
            ]),
        };

        let gen = G2Affine::generator();
        let mut test = G2Projective {
            x: gen.x * (z.square()),
            y: gen.y * (z.square() * z),
            z,
        };

        assert!(bool::from(test.is_on_curve()));

        test.x = z;
        assert!(!bool::from(test.is_on_curve()));
    }

    #[test]
    fn test_affine_point_equality() {
        let a = G2Affine::generator();
        let b = G2Affine::identity();

        assert!(a == a);
        assert!(b == b);
        assert!(a != b);
        assert!(b != a);
    }

    #[test]
    fn test_projective_point_equality() {
        let a = G2Projective::generator();
        let b = G2Projective::identity();

        assert!(a == a);
        assert!(b == b);
        assert!(a != b);
        assert!(b != a);

        let z = Fp2 {
            c0: Fp::from_raw_unchecked([
                0xba7afa1f9a6fe250,
                0xfa0f5b595eafe731,
                0x3bdc477694c306e7,
                0x2149be4b3949fa24,
                0x64aa6e0649b2078c,
                0x12b108ac33643c3e,
            ]),
            c1: Fp::from_raw_unchecked([
                0x125325df3d35b5a8,
                0xdc469ef5555d7fe3,
                0x2d716d2443106a9,
                0x5a1db59a6ff37d0,
                0x7cf7784e5300bb8f,
                0x16a88922c7a5e844,
            ]),
        };

        let mut c = G2Projective {
            x: a.x * (z.square()),
            y: a.y * (z.square() * z),
            z,
        };
        assert!(bool::from(c.is_on_curve()));

        assert!(a == c);
        assert!(b != c);
        assert!(c == a);
        assert!(c != b);

        c.y = -c.y;
        assert!(bool::from(c.is_on_curve()));

        assert!(a != c);
        assert!(b != c);
        assert!(c != a);
        assert!(c != b);

        c.y = -c.y;
        c.x = z;
        assert!(!bool::from(c.is_on_curve()));
        assert!(a != b);
        assert!(a != c);
        assert!(b != c);
    }

    #[test]
    fn test_conditionally_select_affine() {
        let a = G2Affine::generator();
        let b = G2Affine::identity();

        assert_eq!(G2Affine::conditional_select(&a, &b, Choice::from(0u8)), a);
        assert_eq!(G2Affine::conditional_select(&a, &b, Choice::from(1u8)), b);
    }

    #[test]
    fn test_conditionally_select_projective() {
        let a = G2Projective::generator();
        let b = G2Projective::identity();

        assert_eq!(
            G2Projective::conditional_select(&a, &b, Choice::from(0u8)),
            a
        );
        assert_eq!(
            G2Projective::conditional_select(&a, &b, Choice::from(1u8)),
            b
        );
    }

    #[test]
    fn test_projective_to_affine() {
        let a = G2Projective::generator();
        let b = G2Projective::identity();

        assert!(bool::from(G2Affine::from(a).is_on_curve()));
        assert!(!bool::from(G2Affine::from(a).is_identity()));
        assert!(bool::from(G2Affine::from(b).is_on_curve()));
        assert!(bool::from(G2Affine::from(b).is_identity()));

        let z = Fp2 {
            c0: Fp::from_raw_unchecked([
                0xba7afa1f9a6fe250,
                0xfa0f5b595eafe731,
                0x3bdc477694c306e7,
                0x2149be4b3949fa24,
                0x64aa6e0649b2078c,
                0x12b108ac33643c3e,
            ]),
            c1: Fp::from_raw_unchecked([
                0x125325df3d35b5a8,
                0xdc469ef5555d7fe3,
                0x2d716d2443106a9,
                0x5a1db59a6ff37d0,
                0x7cf7784e5300bb8f,
                0x16a88922c7a5e844,
            ]),
        };

        let c = G2Projective {
            x: a.x * (z.square()),
            y: a.y * (z.square() * z),
            z,
        };

        assert_eq!(G2Affine::from(c), G2Affine::generator());
    }

    #[test]
    fn test_affine_to_projective() {
        let a = G2Affine::generator();
        let b = G2Affine::identity();

        assert!(bool::from(G2Projective::from(a).is_on_curve()));
        assert!(!bool::from(G2Projective::from(a).is_identity()));
        assert!(bool::from(G2Projective::from(b).is_on_curve()));
        assert!(bool::from(G2Projective::from(b).is_identity()));
    }

    #[test]
    fn test_doubling() {
        {
            let tmp = G2Projective::identity().double();
            assert!(bool::from(tmp.is_identity()));
            assert!(bool::from(tmp.is_on_curve()));
        }
        {
            let tmp = G2Projective::generator().double();
            assert!(!bool::from(tmp.is_identity()));
            assert!(bool::from(tmp.is_on_curve()));

            assert_eq!(
                G2Affine::from(tmp),
                G2Affine {
                    x: Fp2 {
                        c0: Fp::from_raw_unchecked([
                            0xe9d9e2da9620f98b,
                            0x54f1199346b97f36,
                            0x3db3b820376bed27,
                            0xcfdb31c9b0b64f4c,
                            0x41d7c12786354493,
                            0x5710794c255c064
                        ]),
                        c1: Fp::from_raw_unchecked([
                            0xd6c1d3ca6ea0d06e,
                            0xda0cbd905595489f,
                            0x4f5352d43479221d,
                            0x8ade5d736f8c97e0,
                            0x48cc8433925ef70e,
                            0x8d7ea71ea91ef81
                        ]),
                    },
                    y: Fp2 {
                        c0: Fp::from_raw_unchecked([
                            0x15ba26eb4b0d186f,
                            0xd086d64b7e9e01e,
                            0xc8b848dd652f4c78,
                            0xeecf46a6123bae4f,
                            0x255e8dd8b6dc812a,
                            0x164142af21dcf93f
                        ]),
                        c1: Fp::from_raw_unchecked([
                            0xf9b4a1a895984db4,
                            0xd417b114cccff748,
                            0x6856301fc89f086e,
                            0x41c777878931e3da,
                            0x3556b155066a2105,
                            0xacf7d325cb89cf
                        ]),
                    },
                    infinity: Choice::from(0u8)
                }
            );
        }
    }

    #[test]
    fn test_projective_addition() {
        {
            let a = G2Projective::identity();
            let b = G2Projective::identity();
            let c = a + b;
            assert!(bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
        }
        {
            let a = G2Projective::identity();
            let mut b = G2Projective::generator();
            {
                let z = Fp2 {
                    c0: Fp::from_raw_unchecked([
                        0xba7afa1f9a6fe250,
                        0xfa0f5b595eafe731,
                        0x3bdc477694c306e7,
                        0x2149be4b3949fa24,
                        0x64aa6e0649b2078c,
                        0x12b108ac33643c3e,
                    ]),
                    c1: Fp::from_raw_unchecked([
                        0x125325df3d35b5a8,
                        0xdc469ef5555d7fe3,
                        0x2d716d2443106a9,
                        0x5a1db59a6ff37d0,
                        0x7cf7784e5300bb8f,
                        0x16a88922c7a5e844,
                    ]),
                };

                b = G2Projective {
                    x: b.x * (z.square()),
                    y: b.y * (z.square() * z),
                    z,
                };
            }
            let c = a + b;
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
            assert!(c == G2Projective::generator());
        }
        {
            let a = G2Projective::identity();
            let mut b = G2Projective::generator();
            {
                let z = Fp2 {
                    c0: Fp::from_raw_unchecked([
                        0xba7afa1f9a6fe250,
                        0xfa0f5b595eafe731,
                        0x3bdc477694c306e7,
                        0x2149be4b3949fa24,
                        0x64aa6e0649b2078c,
                        0x12b108ac33643c3e,
                    ]),
                    c1: Fp::from_raw_unchecked([
                        0x125325df3d35b5a8,
                        0xdc469ef5555d7fe3,
                        0x2d716d2443106a9,
                        0x5a1db59a6ff37d0,
                        0x7cf7784e5300bb8f,
                        0x16a88922c7a5e844,
                    ]),
                };

                b = G2Projective {
                    x: b.x * (z.square()),
                    y: b.y * (z.square() * z),
                    z,
                };
            }
            let c = b + a;
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
            assert!(c == G2Projective::generator());
        }
        {
            let a = G2Projective::generator().double().double(); // 4P
            let b = G2Projective::generator().double(); // 2P
            let c = a + b;

            let mut d = G2Projective::generator();
            for _ in 0..5 {
                d = d + G2Projective::generator();
            }
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
            assert!(!bool::from(d.is_identity()));
            assert!(bool::from(d.is_on_curve()));
            assert_eq!(c, d);
        }

        // Degenerate case
        {
            let beta = Fp2 {
                c0: Fp::from_raw_unchecked([
                    0xcd03c9e48671f071,
                    0x5dab22461fcda5d2,
                    0x587042afd3851b95,
                    0x8eb60ebe01bacb9e,
                    0x3f97d6e83d050d2,
                    0x18f0206554638741,
                ]),
                c1: Fp::zero(),
            };
            let beta = beta.square();
            let a = G2Projective::generator().double().double();
            let b = G2Projective {
                x: a.x * beta,
                y: -a.y,
                z: a.z,
            };
            assert!(bool::from(a.is_on_curve()));
            assert!(bool::from(b.is_on_curve()));

            let c = a + b;
            assert_eq!(
                G2Affine::from(c),
                G2Affine::from(G2Projective {
                    x: Fp2 {
                        c0: Fp::from_raw_unchecked([
                            0x705abc799ca773d3,
                            0xfe132292c1d4bf08,
                            0xf37ece3e07b2b466,
                            0x887e1c43f447e301,
                            0x1e0970d033bc77e8,
                            0x1985c81e20a693f2
                        ]),
                        c1: Fp::from_raw_unchecked([
                            0x1d79b25db36ab924,
                            0x23948e4d529639d3,
                            0x471ba7fb0d006297,
                            0x2c36d4b4465dc4c0,
                            0x82bbc3cfec67f538,
                            0x51d2728b67bf952
                        ])
                    },
                    y: Fp2 {
                        c0: Fp::from_raw_unchecked([
                            0x41b1bbf6576c0abf,
                            0xb6cc93713f7a0f9a,
                            0x6b65b43e48f3f01f,
                            0xfb7a4cfcaf81be4f,
                            0x3e32dadc6ec22cb6,
                            0xbb0fc49d79807e3
                        ]),
                        c1: Fp::from_raw_unchecked([
                            0x7d1397788f5f2ddf,
                            0xab2907144ff0d8e8,
                            0x5b7573e0cdb91f92,
                            0x4cb8932dd31daf28,
                            0x62bbfac6db052a54,
                            0x11f95c16d14c3bbe
                        ])
                    },
                    z: Fp2::one()
                })
            );
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
        }
    }

    #[test]
    fn test_mixed_addition() {
        {
            let a = G2Affine::identity();
            let b = G2Projective::identity();
            let c = a + b;
            assert!(bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
        }
        {
            let a = G2Affine::identity();
            let mut b = G2Projective::generator();
            {
                let z = Fp2 {
                    c0: Fp::from_raw_unchecked([
                        0xba7afa1f9a6fe250,
                        0xfa0f5b595eafe731,
                        0x3bdc477694c306e7,
                        0x2149be4b3949fa24,
                        0x64aa6e0649b2078c,
                        0x12b108ac33643c3e,
                    ]),
                    c1: Fp::from_raw_unchecked([
                        0x125325df3d35b5a8,
                        0xdc469ef5555d7fe3,
                        0x2d716d2443106a9,
                        0x5a1db59a6ff37d0,
                        0x7cf7784e5300bb8f,
                        0x16a88922c7a5e844,
                    ]),
                };

                b = G2Projective {
                    x: b.x * (z.square()),
                    y: b.y * (z.square() * z),
                    z,
                };
            }
            let c = a + b;
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
            assert!(c == G2Projective::generator());
        }
        {
            let a = G2Affine::identity();
            let mut b = G2Projective::generator();
            {
                let z = Fp2 {
                    c0: Fp::from_raw_unchecked([
                        0xba7afa1f9a6fe250,
                        0xfa0f5b595eafe731,
                        0x3bdc477694c306e7,
                        0x2149be4b3949fa24,
                        0x64aa6e0649b2078c,
                        0x12b108ac33643c3e,
                    ]),
                    c1: Fp::from_raw_unchecked([
                        0x125325df3d35b5a8,
                        0xdc469ef5555d7fe3,
                        0x2d716d2443106a9,
                        0x5a1db59a6ff37d0,
                        0x7cf7784e5300bb8f,
                        0x16a88922c7a5e844,
                    ]),
                };

                b = G2Projective {
                    x: b.x * (z.square()),
                    y: b.y * (z.square() * z),
                    z,
                };
            }
            let c = b + a;
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
            assert!(c == G2Projective::generator());
        }
        {
            let a = G2Projective::generator().double().double(); // 4P
            let b = G2Projective::generator().double(); // 2P
            let c = a + b;

            let mut d = G2Projective::generator();
            for _ in 0..5 {
                d = d + G2Affine::generator();
            }
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
            assert!(!bool::from(d.is_identity()));
            assert!(bool::from(d.is_on_curve()));
            assert_eq!(c, d);
        }

        // Degenerate case
        {
            let beta = Fp2 {
                c0: Fp::from_raw_unchecked([
                    0xcd03c9e48671f071,
                    0x5dab22461fcda5d2,
                    0x587042afd3851b95,
                    0x8eb60ebe01bacb9e,
                    0x3f97d6e83d050d2,
                    0x18f0206554638741,
                ]),
                c1: Fp::zero(),
            };
            let beta = beta.square();
            let a = G2Projective::generator().double().double();
            let b = G2Projective {
                x: a.x * beta,
                y: -a.y,
                z: a.z,
            };
            let a = G2Affine::from(a);
            assert!(bool::from(a.is_on_curve()));
            assert!(bool::from(b.is_on_curve()));

            let c = a + b;
            assert_eq!(
                G2Affine::from(c),
                G2Affine::from(G2Projective {
                    x: Fp2 {
                        c0: Fp::from_raw_unchecked([
                            0x705abc799ca773d3,
                            0xfe132292c1d4bf08,
                            0xf37ece3e07b2b466,
                            0x887e1c43f447e301,
                            0x1e0970d033bc77e8,
                            0x1985c81e20a693f2
                        ]),
                        c1: Fp::from_raw_unchecked([
                            0x1d79b25db36ab924,
                            0x23948e4d529639d3,
                            0x471ba7fb0d006297,
                            0x2c36d4b4465dc4c0,
                            0x82bbc3cfec67f538,
                            0x51d2728b67bf952
                        ])
                    },
                    y: Fp2 {
                        c0: Fp::from_raw_unchecked([
                            0x41b1bbf6576c0abf,
                            0xb6cc93713f7a0f9a,
                            0x6b65b43e48f3f01f,
                            0xfb7a4cfcaf81be4f,
                            0x3e32dadc6ec22cb6,
                            0xbb0fc49d79807e3
                        ]),
                        c1: Fp::from_raw_unchecked([
                            0x7d1397788f5f2ddf,
                            0xab2907144ff0d8e8,
                            0x5b7573e0cdb91f92,
                            0x4cb8932dd31daf28,
                            0x62bbfac6db052a54,
                            0x11f95c16d14c3bbe
                        ])
                    },
                    z: Fp2::one()
                })
            );
            assert!(!bool::from(c.is_identity()));
            assert!(bool::from(c.is_on_curve()));
        }
    }

    #[test]
    fn test_projective_negation_and_subtraction() {
        let a = G2Projective::generator().double();
        assert_eq!(a + (-a), G2Projective::identity());
        assert_eq!(a + (-a), a - a);
    }

    #[test]
    fn test_affine_negation_and_subtraction() {
        let a = G2Affine::generator();
        assert_eq!(G2Projective::from(a) + (-a), G2Projective::identity());
        assert_eq!(G2Projective::from(a) + (-a), G2Projective::from(a) - a);
    }

    #[test]
    fn test_projective_scalar_multiplication() {
        let g = G2Projective::generator();
        let a = BlsScalar::from_raw([
            0x2b568297a56da71c,
            0xd8c39ecb0ef375d1,
            0x435c38da67bfbf96,
            0x8088a05026b659b2,
        ]);
        let b = BlsScalar::from_raw([
            0x785fdd9b26ef8b85,
            0xc997f25837695c18,
            0x4c8dbc39e7b756c1,
            0x70d9b6cc6d87df20,
        ]);
        let c = a * b;

        assert_eq!((g * a) * b, g * c);
    }

    #[test]
    fn test_affine_scalar_multiplication() {
        let g = G2Affine::generator();
        let a = BlsScalar::from_raw([
            0x2b568297a56da71c,
            0xd8c39ecb0ef375d1,
            0x435c38da67bfbf96,
            0x8088a05026b659b2,
        ]);
        let b = BlsScalar::from_raw([
            0x785fdd9b26ef8b85,
            0xc997f25837695c18,
            0x4c8dbc39e7b756c1,
            0x70d9b6cc6d87df20,
        ]);
        let c = a * b;

        assert_eq!(G2Affine::from(g * a) * b, g * c);
    }

    #[test]
    fn test_is_torsion_free() {
        let a = G2Affine {
            x: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x89f550c813db6431,
                    0xa50be8c456cd8a1a,
                    0xa45b374114cae851,
                    0xbb6190f5bf7fff63,
                    0x970ca02c3ba80bc7,
                    0x2b85d24e840fbac,
                ]),
                c1: Fp::from_raw_unchecked([
                    0x6888bc53d70716dc,
                    0x3dea6b4117682d70,
                    0xd8f5f930500ca354,
                    0x6b5ecb6556f5c155,
                    0xc96bef0434778ab0,
                    0x5081505515006ad,
                ]),
            },
            y: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x3cf1ea0d434b0f40,
                    0x1a0dc610e603e333,
                    0x7f89956160c72fa0,
                    0x25ee03decf6431c5,
                    0xeee8e206ec0fe137,
                    0x97592b226dfef28,
                ]),
                c1: Fp::from_raw_unchecked([
                    0x71e8bb5f29247367,
                    0xa5fe049e211831ce,
                    0xce6b354502a3896,
                    0x93b012000997314e,
                    0x6759f3b6aa5b42ac,
                    0x156944c4dfe92bbb,
                ]),
            },
            infinity: Choice::from(0u8),
        };
        assert!(!bool::from(a.is_torsion_free()));

        assert!(bool::from(G2Affine::identity().is_torsion_free()));
        assert!(bool::from(G2Affine::generator().is_torsion_free()));
    }

    #[cfg(feature = "endo")]
    #[test]
    fn test_mul_by_x() {
        // multiplying by `x` a point in G2 is the same as multiplying by
        // the equivalent scalar.
        let generator = G2Projective::generator();
        let x = if crate::BLS_X_IS_NEGATIVE {
            -BlsScalar::from(crate::BLS_X)
        } else {
            BlsScalar::from(crate::BLS_X)
        };
        assert_eq!(generator.mul_by_x(), generator * x);

        let point = G2Projective::generator() * BlsScalar::from(42);
        assert_eq!(point.mul_by_x(), point * x);
    }

    #[cfg(feature = "endo")]
    #[test]
    fn test_psi() {
        let generator = G2Projective::generator();

        // `point` is a random point in the curve
        let point = G2Projective {
            x: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0xee4c8cb7c047eaf2,
                    0x44ca22eee036b604,
                    0x33b3affb2aefe101,
                    0x15d3e45bbafaeb02,
                    0x7bfc2154cd7419a4,
                    0x0a2d0c2b756e5edc,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xfc224361029a8777,
                    0x4cbf2baab8740924,
                    0xc5008c6ec6592c89,
                    0xecc2c57b472a9c2d,
                    0x8613eafd9d81ffb1,
                    0x10fe54daa2d3d495,
                ]),
            },
            y: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x7de7edc43953b75c,
                    0x58be1d2de35e87dc,
                    0x5731d30b0e337b40,
                    0xbe93b60cfeaae4c9,
                    0x8b22c203764bedca,
                    0x01616c8d1033b771,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xea126fe476b5733b,
                    0x85cee68b5dae1652,
                    0x98247779f7272b04,
                    0xa649c8b468c6e808,
                    0xb5b9a62dff0c4e45,
                    0x1555b67fc7bbe73d,
                ]),
            },
            z: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x0ef2ddffab187c0a,
                    0x2424522b7d5ecbfc,
                    0xc6f341a3398054f4,
                    0x5523ddf409502df0,
                    0xd55c0b5a88e0dd97,
                    0x066428d704923e52,
                ]),
                c1: Fp::from_raw_unchecked([
                    0x538bbe0c95b4878d,
                    0xad04a50379522881,
                    0x6d5c05bf5c12fb64,
                    0x4ce4a069a2d34787,
                    0x59ea6c8d0dffaeaf,
                    0x0d42a083a75bd6f3,
                ]),
            },
        };
        assert!(bool::from(point.is_on_curve()));

        // psi2(P) = psi(psi(P))
        assert_eq!(generator.psi2(), generator.psi().psi());
        assert_eq!(point.psi2(), point.psi().psi());
        // psi(P) is a morphism
        assert_eq!(generator.double().psi(), generator.psi().double());
        assert_eq!(point.psi() + generator.psi(), (point + generator).psi());
        // psi(P) behaves in the same way on the same projective point
        let mut normalized_point = [G2Affine::identity()];
        G2Projective::batch_normalize(&[point], &mut normalized_point);
        let normalized_point = G2Projective::from(normalized_point[0]);
        assert_eq!(point.psi(), normalized_point.psi());
        assert_eq!(point.psi2(), normalized_point.psi2());
    }

    #[test]
    fn test_clear_cofactor() {
        // `point` is a random point in the curve
        let point = G2Projective {
            x: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0xee4c8cb7c047eaf2,
                    0x44ca22eee036b604,
                    0x33b3affb2aefe101,
                    0x15d3e45bbafaeb02,
                    0x7bfc2154cd7419a4,
                    0x0a2d0c2b756e5edc,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xfc224361029a8777,
                    0x4cbf2baab8740924,
                    0xc5008c6ec6592c89,
                    0xecc2c57b472a9c2d,
                    0x8613eafd9d81ffb1,
                    0x10fe54daa2d3d495,
                ]),
            },
            y: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x7de7edc43953b75c,
                    0x58be1d2de35e87dc,
                    0x5731d30b0e337b40,
                    0xbe93b60cfeaae4c9,
                    0x8b22c203764bedca,
                    0x01616c8d1033b771,
                ]),
                c1: Fp::from_raw_unchecked([
                    0xea126fe476b5733b,
                    0x85cee68b5dae1652,
                    0x98247779f7272b04,
                    0xa649c8b468c6e808,
                    0xb5b9a62dff0c4e45,
                    0x1555b67fc7bbe73d,
                ]),
            },
            z: Fp2 {
                c0: Fp::from_raw_unchecked([
                    0x0ef2ddffab187c0a,
                    0x2424522b7d5ecbfc,
                    0xc6f341a3398054f4,
                    0x5523ddf409502df0,
                    0xd55c0b5a88e0dd97,
                    0x066428d704923e52,
                ]),
                c1: Fp::from_raw_unchecked([
                    0x538bbe0c95b4878d,
                    0xad04a50379522881,
                    0x6d5c05bf5c12fb64,
                    0x4ce4a069a2d34787,
                    0x59ea6c8d0dffaeaf,
                    0x0d42a083a75bd6f3,
                ]),
            },
        };

        assert!(bool::from(point.is_on_curve()));
        assert!(!bool::from(G2Affine::from(point).is_torsion_free()));
        let cleared_point = point.clear_cofactor();

        assert!(bool::from(cleared_point.is_on_curve()));
        assert!(bool::from(G2Affine::from(cleared_point).is_torsion_free()));

        // the generator (and the identity) are always on the curve,
        // even after clearing the cofactor
        let generator = G2Projective::generator();
        assert!(bool::from(generator.clear_cofactor().is_on_curve()));
        let id = G2Projective::identity();
        assert!(bool::from(id.clear_cofactor().is_on_curve()));

        // test the effect on q-torsion points multiplying by h_eff modulo |BlsScalar|
        // h_eff % q = 0x2b116900400069009a40200040001ffff
        let h_eff_modq: [u8; 32] = [
            0xff, 0xff, 0x01, 0x00, 0x04, 0x00, 0x02, 0xa4, 0x09, 0x90, 0x06, 0x00, 0x04, 0x90,
            0x16, 0xb1, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
        ];
        assert_eq!(generator.clear_cofactor(), generator.multiply(&h_eff_modq));
        assert_eq!(
            cleared_point.clear_cofactor(),
            cleared_point.multiply(&h_eff_modq)
        );
    }

    #[test]
    fn test_batch_normalize() {
        let a = G2Projective::generator().double();
        let b = a.double();
        let c = b.double();

        for a_identity in (0..1).map(|n| n == 1) {
            for b_identity in (0..1).map(|n| n == 1) {
                for c_identity in (0..1).map(|n| n == 1) {
                    let mut v = [a, b, c];
                    if a_identity {
                        v[0] = G2Projective::identity()
                    }
                    if b_identity {
                        v[1] = G2Projective::identity()
                    }
                    if c_identity {
                        v[2] = G2Projective::identity()
                    }

                    let mut t = [
                        G2Affine::identity(),
                        G2Affine::identity(),
                        G2Affine::identity(),
                    ];
                    let expected = [
                        G2Affine::from(v[0]),
                        G2Affine::from(v[1]),
                        G2Affine::from(v[2]),
                    ];

                    G2Projective::batch_normalize(&v[..], &mut t[..]);

                    assert_eq!(&t[..], &expected[..]);
                }
            }
        }
    }

    #[test]
    #[cfg(feature = "serde_req")]
    fn g2_affine_serde_roundtrip() {
        use bincode;

        let gen = G2Affine::generator();
        let ser = bincode::serialize(&gen).unwrap();
        let deser: G2Affine = bincode::deserialize(&ser).unwrap();

        assert_eq!(gen, deser);
    }

    #[test]
    fn g2_affine_bytes_unchecked() {
        let gen = G2Affine::generator();
        let ident = G2Affine::identity();

        let gen_p = gen.to_raw_bytes();
        let gen_p = unsafe { G2Affine::from_slice_unchecked(&gen_p) };

        let ident_p = ident.to_raw_bytes();
        let ident_p = unsafe { G2Affine::from_slice_unchecked(&ident_p) };

        assert_eq!(gen, gen_p);
        assert_eq!(ident, ident_p);
    }

    #[test]
    fn g2_affine_hex() {
        use dusk_bytes::ParseHexStr;

        let gen = G2Affine::generator();
        let ident = G2Affine::identity();

        let gen_p = format!("{:x}", gen);
        let gen_p = G2Affine::from_hex_str(gen_p.as_str()).unwrap();

        let ident_p = format!("{:x}", ident);
        let ident_p = G2Affine::from_hex_str(ident_p.as_str()).unwrap();

        assert_eq!(gen, gen_p);
        assert_eq!(ident, ident_p);
    }
}
