/* SPDX-License-Identifier: MIT */

use anyhow::Result;
use clap::Parser as ClapParser;
use log::{info, warn};
use std::path::Path;
use std::time::UNIX_EPOCH;
use tracing::{error, Level};

#[derive(Debug, ClapParser)]
pub(crate) struct DumpArgs {
    /// Directory to process
    #[clap(long)]
    directory: String,
    /// Optional PCAP filter
    #[clap(long)]
    filter: Option<String>,
    /// Filename prefix
    ///
    /// If set, only files beginning with the provided prefix will be processed.
    #[clap(long)]
    prefix: Option<String>,
    /// Recurse through --directory
    #[clap(long)]
    recursive: bool,
    /// Start timestamp in seconds
    ///
    /// If provided, packets with a timestamp older than the one provided will
    /// not be exported.
    #[clap(long)]
    start_time: Option<i64>,
    /// Duration after start-time
    ///
    /// Sets the duration in seconds after start-time that packets will be
    /// considered for export. This refers to the packet timestamp, not real
    /// time.
    #[clap(long)]
    duration: Option<i64>,
    /// Enable more verbose logging
    #[clap(long, short)]
    verbose: bool,
    /// Output filename
    ///
    /// Filename to output packets to. "-" can be used for stdout.
    #[clap(long)]
    output: String,

    /// Output logs in JSON.
    #[clap(long)]
    json: bool,
}

pub(crate) fn main(args: DumpArgs) -> anyhow::Result<()> {
    let level = if args.verbose {
        Level::INFO
    } else {
        Level::ERROR
    };
    let logger = tracing_subscriber::fmt()
        .with_max_level(level)
        .with_writer(std::io::stderr);
    if args.json {
        logger.json().init();
    } else {
        logger.init();
    }
    let mut dump_out = None;
    process_dir(&args, Path::new(&args.directory), &mut dump_out);
    Ok(())
}

fn process_dir(args: &DumpArgs, directory: &Path, out: &mut Option<pcap::Savefile>) {
    for entry in std::fs::read_dir(directory).unwrap() {
        let entry = entry.unwrap();
        let path = entry.path();
        if path.is_dir() && args.recursive {
            process_dir(args, &path, out);
        } else if path.is_file() {
            if let Some(prefix) = &args.prefix {
                if !path.starts_with(prefix) {
                    continue;
                }
            }
            if let Err(err) = process_file(args, &path, out) {
                error!("{:?}", err);
                std::process::exit(1);
            }
        } else {
            info!("Ignoring {:?}", &path);
        }
    }
}

fn process_file(args: &DumpArgs, path: &Path, out: &mut Option<pcap::Savefile>) -> Result<()> {
    info!("Processing file {}", &path.display());
    let metadata = std::fs::metadata(path)?;
    let mtime = metadata.modified()?.duration_since(UNIX_EPOCH)?.as_secs();
    if let Some(start_time) = args.start_time {
        if mtime < start_time as u64 {
            info!(
                "Ignoring {}, last modified before {}",
                &path.display(),
                start_time
            );
            return Ok(());
        }
    }
    let mut cf = pcap::Capture::from_file(&path).unwrap();
    if let Some(filter) = &args.filter {
        cf.filter(filter, true)?;
    }
    loop {
        let n = cf.next();
        match n {
            Ok(pkt) => {
                let secs = pkt.header.ts.tv_sec;
                if let Some(start_time) = args.start_time {
                    if secs < start_time {
                        continue;
                    }
                    if let Some(duration) = args.duration {
                        if secs > start_time + duration {
                            return Ok(());
                        }
                    }
                }
                if out.is_none() {
                    // A bit of a hack, but prevents writing any data until we know we've found
                    // a packet.
                    let cf0 = pcap::Capture::from_file(&path).unwrap();
                    *out = Some(cf0.savefile(&args.output).unwrap());
                }
                out.as_mut().unwrap().write(&pkt);
            }
            Err(err) => {
                match err {
                    pcap::Error::NoMorePackets => {
                        break;
                    }
                    pcap::Error::PcapError(ref error) => {
                        // Truncation errors are expected.
                        if error.contains("truncated") {
                            break;
                        }
                    }
                    _ => {}
                }
                if args.verbose {
                    warn!("{}: {}", path.display(), err);
                }
                break;
            }
        }
    }
    Ok(())
}
