use crate::enums::{State, Tone};

use core::iter::Iterator;

const TONE_TABLE: [[Tone; 4]; 4] = [
	[Tone::One, Tone::Two, Tone::Three, Tone::A],
	[Tone::Four, Tone::Five, Tone::Six, Tone::B],
	[Tone::Seven, Tone::Eight, Tone::Nine, Tone::C],
	[Tone::Asterisk, Tone::Zero, Tone::Pound, Tone::D],
];

// https://www.ee.columbia.edu/~dpwe/classes/e4810-2000-09/projrpts/az209/dspproject/dspproject.pdf
// 512 samples @ 8KHz sampling frequency should be plenty
const SAMPLE_RATE: u32 = 8000;
// 697, 770, 852, 941
const LOW_FREQS: [usize; 4] = [45, 49, 55, 60];
// 1209, 1336, 1477, 1633
const HIGH_FREQS: [usize; 4] = [77, 85, 94, 104];
const NUM_SAMPLES: usize = 512;

pub struct Decoder<F: FnMut(Tone, State)> {
	sample_rate: u32,
	tone_change: F,
	cur_tone: Option<Tone>,

	// holds samples that haven't been processed yet
	samples: [f32; NUM_SAMPLES],
	sample_len: usize,
	// keeps track of our phase when downsampling
	downsample_phase: f32,
}

impl<F: FnMut(Tone, State)> Decoder<F> {
	pub fn new(sample_rate: u32, tone_change: F) -> Self {
		Self {
			sample_rate,
			tone_change,
			cur_tone: None,

			samples: [0.0; NUM_SAMPLES],
			sample_len: 0,

			downsample_phase: 0.0,
		}
	}

	/// Expects samples to be between -1.0 and 1.0
	/// Batch together as many (or as few) samples when processing
	pub fn process(&mut self, samples: &[f32]) {
		let mut iter = samples.iter();
		while let Some(sample) = self.downsample(&mut iter) {
			self.samples[self.sample_len] = *sample;
			self.sample_len += 1;
			if self.sample_len == NUM_SAMPLES {
				self.process_chunk();
				self.sample_len = 0;
			}
		}

		/*dbg!(
			avg_power,
			low_powers,
			high_powers //spectrum.iter().map(|x| x.norm_sqr()).collect::<Vec<_>>()
		);*/
	}

	fn process_chunk(&mut self) {
		let spectrum = microfft::real::rfft_512(&mut self.samples);
		spectrum[0].im = 0.0;

		let avg_power = spectrum.iter().map(|x| x.norm_sqr()).sum::<f32>() / NUM_SAMPLES as f32;
		let mut low_powers = [0.0; 4];
		for i in 0..4 {
			low_powers[i] = spectrum[LOW_FREQS[i]].norm_sqr();
		}

		let mut high_powers = [0.0; 4];
		for i in 0..4 {
			high_powers[i] = spectrum[HIGH_FREQS[i]].norm_sqr();
		}

		let low = main_freq(&low_powers, avg_power);
		let high = main_freq(&high_powers, avg_power);

		match (low, high) {
			(Some(l), Some(h)) => {
				// dtmf tone
				let tone = TONE_TABLE[l][h];

				if self.cur_tone != Some(tone) {
					if let Some(old) = self.cur_tone {
						(self.tone_change)(old, State::Off);
					}

					(self.tone_change)(tone, State::On);

					self.cur_tone = Some(tone);
				}
			}
			_ => {
				// no dtmf tone
				if let Some(tone) = self.cur_tone {
					(self.tone_change)(tone, State::Off);
					self.cur_tone = None;
				}
			}
		}
	}

	// gets a single downsampled sample
	// also returns how many input samples were used up
	fn downsample<T: Copy, I: Iterator<Item = T>>(&mut self, samples: &mut I) -> Option<T> {
		let scale = self.sample_rate as f32 / SAMPLE_RATE as f32;

		for s in samples {
			self.downsample_phase += 1.0;

			if self.downsample_phase > scale {
				self.downsample_phase -= scale;

				return Some(s);
			}
		}

		None
	}
}

// gets the dominant frequency in the array, if there is one and it's strong enough
fn main_freq(freqs: &[f32], avg_power: f32) -> Option<usize> {
	assert!(freqs.len() > 2);

	let mut idx = 0;
	for i in 1..freqs.len() {
		if freqs[i] > freqs[idx] {
			idx = i;
		}
	}

	// 2nd largest
	let mut snd = 0;
	for i in 0..freqs.len() {
		if (i != idx && freqs[i] > freqs[snd]) || snd == idx {
			snd = i;
		}
	}

	// main freq must be double the next biggest
	// and double avg power
	let threshold = freqs[snd].max(avg_power).max(1.0);
	if threshold * 2.0 < freqs[idx] {
		Some(idx)
	} else {
		None
	}
}
