//! Supporting functions for commands

use super::desktop;
use desktop::KeyValue;

use itertools::Itertools;
use std::fs;
use std::path::Path;
use std::{convert::identity, path::PathBuf};

pub struct ExecEntry {
    pub name: String,
    pub exec: String,
    pub path: PathBuf,
}

pub fn parse_exec_entries(path: PathBuf, bits: Vec<u8>) -> Vec<ExecEntry> {
    let ast = desktop::parse(bits);

    let mut key_values = ast
        .iter()
        .filter_map(|(name, key_values)| match name.as_str() {
            "Desktop Entry" => Some(key_values),
            _ => None,
        })
        .flat_map(identity);

    let exec_m = key_values
        .clone()
        .find(|KeyValue { key, .. }| key == "Exec")
        .map(get_value);

    let name_m = key_values
        .find(|KeyValue { key, .. }| key == "Name")
        .map(get_value);

    match (name_m, exec_m) {
        (Some(name), Some(exec)) => vec![ExecEntry {
            name: name.to_string(),
            exec: exec.to_string(),
            path: path,
        }],
        _ => Vec::new(),
    }
}

pub fn normalize_exec(exec: &str) -> String {
    exec.split(" ")
        .filter(|str| !str.contains("%"))
        .map(|str| String::from(str))
        .collect::<Vec<String>>()
        .join(" ")
}

fn get_value(key_value: &KeyValue) -> &str {
    &key_value.value
}

pub fn parse_data_dirs(xdg_data_dirs: &str) -> Vec<ExecEntry> {
    let desktop_extension: std::ffi::OsString = "desktop".into();
    xdg_data_dirs
        .split(":")
        .map(|dir| Path::new(dir))
        .map(|dir| dir.join("applications"))
        .filter(|dir| dir.exists())
        .filter_map(|dir| fs::read_dir(dir).ok())
        .flat_map(|dir| dir)
        .filter_map(|file| file.ok())
        .filter(|file| file.path().extension().as_deref() == Some(&desktop_extension))
        .map(|file| (file.path(), fs::read(file.path())))
        .filter_map(|(path, bytes_r)| bytes_r.ok().map(|bytes| (path, bytes)))
        .flat_map(|(path, bytes)| parse_exec_entries(path, bytes))
        .collect_vec()
}

pub fn parse_sorted_data_dirs(xdg_data_dirs: &str) -> Vec<ExecEntry> {
    parse_data_dirs(xdg_data_dirs)
        .into_iter()
        .unique_by(|ExecEntry { name, .. }| name.clone())
        .sorted_by(|a, b| Ord::cmp(&a.name, &b.name))
        .collect_vec()
}

pub fn path_to_string(path_buf: PathBuf) -> String {
    path_buf.into_os_string().into_string().unwrap()
}

pub fn find_by_name(xdg_data_dirs: &str, name: &str) -> Option<ExecEntry> {
    let exec_entries = parse_data_dirs(xdg_data_dirs);
    exec_entries.into_iter().find(
        |ExecEntry {
             name: curr_name, ..
         }| curr_name == name,
    )
}
