// Copyright 2021-2022 Sebastian Ramacher
// SPDX-License-Identifier: GPL-3.0-or-later

use std::io::{self, BufRead, BufReader};
use std::path::PathBuf;
use std::{collections::HashMap, fs::File};

use anyhow::Result;
use clap::Parser;
use serde::Deserialize;

use crate::{
    config::{self, CacheEntries},
    BaseOptions, BinNMUsOptions,
};
use assorted_debian_utils::{
    archive::Codename,
    wb::{BinNMU, SourceSpecifier, WBCommandBuilder},
};

#[derive(Debug, Deserialize)]
struct UDDBug {
    id: u32,
    source: String,
}

#[derive(Debug, Parser)]
pub(crate) struct PrepareBinNMUsOptions {
    #[clap(flatten)]
    binnmu_options: BinNMUsOptions,
    /// Input file with a list of packages. If not specfied, the list of packages will be read from he standard input.
    #[clap(parse(from_os_str))]
    input: Option<PathBuf>,
}

pub(crate) struct PrepareBinNMUs {
    cache: config::Cache,
    base_options: BaseOptions,
    options: PrepareBinNMUsOptions,
}

impl PrepareBinNMUs {
    pub(crate) fn new(base_options: BaseOptions, options: PrepareBinNMUsOptions) -> Result<Self> {
        Ok(Self {
            cache: config::Cache::new(base_options.force_download)?,
            base_options,
            options,
        })
    }

    #[tokio::main]
    async fn download_to_cache(&self, codename: &Codename) -> Result<()> {
        self.cache
            .download(&[CacheEntries::FTBFSBugs(codename.clone())])
            .await?;
        Ok(())
    }

    fn load_bugs(&self, codename: &Codename) -> Result<HashMap<String, u32>> {
        self.download_to_cache(codename)?;

        let bugs: Vec<UDDBug> = serde_yaml::from_reader(
            self.cache
                .get_cache_bufreader(format!("udd-ftbfs-bugs-{}.yaml", codename))
                .unwrap(),
        )
        .unwrap_or_default();
        Ok(bugs.into_iter().map(|bug| (bug.source, bug.id)).collect())
    }

    pub(crate) fn run(self) -> Result<()> {
        let codename: Codename = self.options.binnmu_options.suite.clone().into();
        let ftbfs_bugs = if !self.base_options.force_processing {
            self.load_bugs(&codename)?
        } else {
            HashMap::new()
        };

        let matcher = regex::Regex::new("([a-z0-9+.-]+)[ \t].* \\(?([0-9][^() \t]*)\\)?")?;

        let reader: Box<dyn BufRead> = match &self.options.input {
            None => Box::new(BufReader::new(io::stdin())),
            Some(filename) => Box::new(BufReader::new(File::open(filename)?)),
        };

        let mut wb_commands = Vec::new();
        for line in reader.lines() {
            if line.is_err() {
                break;
            }

            let line = line.unwrap();
            if let Some(capture) = matcher.captures(&line) {
                let package = capture.get(1);
                let version = capture.get(2);
                if package.is_none() || version.is_none() {
                    continue;
                }

                let source = package.unwrap().as_str();
                if let Some(bug) = ftbfs_bugs.get(source) {
                    println!("# Skipping {} due to FTBFS bug #{}", source, bug);
                    continue;
                }

                let mut source = SourceSpecifier::new(source);
                let version = version.unwrap().as_str().try_into()?;
                source
                    .with_version(&version)
                    .with_suite(&self.options.binnmu_options.suite);
                if let Some(architectures) = &self.options.binnmu_options.architecture {
                    source.with_archive_architectures(architectures);
                }

                let mut binnmu = BinNMU::new(&source, &self.options.binnmu_options.message)?;
                if let Some(bp) = self.options.binnmu_options.build_priority {
                    binnmu.with_build_priority(bp);
                }
                if let Some(dw) = &self.options.binnmu_options.dep_wait {
                    binnmu.with_dependency_wait(dw);
                }
                if let Some(extra_depends) = &self.options.binnmu_options.extra_depends {
                    binnmu.with_extra_depends(extra_depends);
                }
                wb_commands.push(binnmu.build())
            }
        }

        for commands in wb_commands {
            println!("{}", commands);
            if !self.base_options.dry_run {
                commands.execute()?;
            }
        }

        Ok(())
    }
}
