use crate::config::Context;
use crate::{util, AppId, Verbs};
use anyhow::{anyhow, Context as AnyhowContext, Result};
use oauth2::TokenResponse;
use reqwest::blocking::{Client, Response};
use reqwest::{StatusCode, Url};
use serde_json::{from_str, json, Value};
use std::process::exit;
use tabular::{Row, Table};

fn craft_url(base: &Url, app_id: Option<&str>) -> String {
    let app = match app_id {
        Some(app) => format!("/{}", app),
        None => String::new(),
    };
    format!("{}{}/apps{}", base, util::API_PATH, app)
}

pub fn create(
    config: &Context,
    app: AppId,
    data: serde_json::Value,
    file: Option<&str>,
) -> Result<()> {
    let client = Client::new();
    let url = craft_url(&config.registry_url, None);
    let body = match file {
        Some(f) => util::get_data_from_file(f)?,
        None => {
            json!({
            "metadata": {
                "name": app,
            },
            "spec": data,
            })
        }
    };

    client
        .post(&url)
        .header(reqwest::header::CONTENT_TYPE, "application/json")
        .body(body.to_string())
        .bearer_auth(&config.token.access_token().secret())
        .send()
        .context("Can't create app.")
        .map(|res| util::print_result(res, format!("App {}", app), Verbs::create))
}

pub fn read(config: &Context, app: AppId) -> Result<()> {
    get(config, &app).map(|res| util::print_result(res, app.to_string(), Verbs::get))
}

pub fn delete(config: &Context, app: AppId) -> Result<()> {
    let client = Client::new();
    let url = craft_url(&config.registry_url, Some(&app));

    client
        .delete(&url)
        .bearer_auth(&config.token.access_token().secret())
        .send()
        .context("Can't get app.")
        .map(|res| util::print_result(res, format!("App {}", &app), Verbs::delete))
}

pub fn edit(config: &Context, app: AppId, file: Option<&str>) -> Result<()> {
    match file {
        Some(f) => {
            let data = util::get_data_from_file(f)?;

            put(&config, &app, data)
                .map(|res| util::print_result(res, format!("App {}", &app), Verbs::edit))
        }
        None => {
            //read app data
            let res = get(config, &app);
            match res {
                Ok(r) => match r.status() {
                    StatusCode::OK => {
                        let body = r.text().unwrap_or_else(|_| "{}".to_string());
                        let insert = util::editor(body)?;

                        put(config, &app, insert)
                            .map(|p| util::print_result(p, format!("App {}", &app), Verbs::edit))
                    }
                    e => {
                        log::error!("Error : could not retrieve app: {}", e);
                        exit(2);
                    }
                },
                Err(e) => {
                    log::error!("Error : could not retrieve app: {}", e);
                    exit(2);
                }
            }
        }
    }
}

pub fn list(config: &Context, labels: Option<String>) -> Result<()> {
    let client = Client::new();
    let url = craft_url(&config.registry_url, None);

    let mut req = client
        .get(&url)
        .bearer_auth(&config.token.access_token().secret());

    if let Some(labels) = labels {
        req = req.query(&[("labels", labels)]);
    }

    let res = req.send().context("Can't list apps");

    if let Ok(r) = res {
        if r.status() == StatusCode::OK {
            pretty_list(r.text()?)?;
            Ok(())
        } else {
            Err(anyhow!("List operation failed with {}", r.status()))
        }
    } else {
        Err(anyhow!("Error while requesting app list."))
    }
}

fn get(config: &Context, app: &str) -> Result<Response> {
    let client = Client::new();
    let url = craft_url(&config.registry_url, Some(app));
    client
        .get(&url)
        .bearer_auth(&config.token.access_token().secret())
        .send()
        .context("Can't retrieve app data.")
}

fn put(config: &Context, app: &str, data: serde_json::Value) -> Result<Response> {
    let client = Client::new();
    let url = craft_url(&config.registry_url, Some(app));

    client
        .put(&url)
        .header(reqwest::header::CONTENT_TYPE, "application/json")
        .bearer_auth(&config.token.access_token().secret())
        .body(data.to_string())
        .send()
        .context("Can't update app data.")
}

// todo drogue-client and the types would be useful for this
fn pretty_list(data: String) -> Result<()> {
    let apps_array: Vec<Value> = from_str(data.as_str())?;

    let mut table = Table::new("{:<} {:<}");
    table.add_row(Row::new().with_cell("NAME").with_cell("AGE"));

    for app in apps_array {
        let name = app["metadata"]["name"].as_str();
        let creation = app["metadata"]["creationTimestamp"].as_str();
        if let Some(name) = name {
            table.add_row(
                Row::new()
                    .with_cell(name)
                    .with_cell(util::age(creation.unwrap())?),
            );
        }
    }

    print!("{}", table);
    Ok(())
}
