use bytes::Bytes;
use crate::lgt92::{LGT92, Alarm, Coordinate, BatteryUnit, Angle};

const SCALE: Coordinate = 1_000_000.0;

pub fn decode_lat(b: &Bytes) -> Coordinate {
    let r: i32 = (b[0] as i32) << 24 | (b[1] as i32) << 16 | (b[2] as i32) << 8 | b[3] as i32;

    r as Coordinate / SCALE
}

pub fn decode_long(b: &Bytes) -> Coordinate {
    let s: i32 = (b[4] as i32) << 24 | (b[5] as i32) << 16 | (b[6] as i32) << 8 | b[7] as i32;

    s as Coordinate / SCALE
}

pub fn decode_alarm(b: &Bytes) -> Alarm {
    if (b[8] & 0x40) == 0x40 {
	return Alarm::On
    }
    Alarm::Off
}

pub fn decode_battery(b: &Bytes) -> BatteryUnit {
    let bat = (b[8] as i16) << 8 | b[9] as i16;
    (bat & 0x3FFF) as BatteryUnit / 1_000.0
}

pub fn decode_roll(b: &Bytes) -> Angle {
    let roll = (b[11] as i16) << 8 | b[12] as i16;

    roll as Angle / 100.0
}

pub fn decode_pitch(b: &Bytes) -> Angle {
    let pitch = (b[13] as i16) << 8 | b[14] as i16;

    pitch as Angle / 100.0
}

pub fn decode(b: &Bytes) -> LGT92 {
    LGT92 {
	latitude: decode_lat(b),
	longitude: decode_long(b),
	alarm: decode_alarm(b),
	battery: decode_battery(b),
	roll: decode_roll(b),
	pitch: decode_pitch(b)
    }
}
