extern crate anyhow;
extern crate camino;
extern crate tempfile;
extern crate utils;
#[macro_use]
extern crate clap;
extern crate dirs;
extern crate env_logger;
#[macro_use]
extern crate log;
#[macro_use]
extern crate serde_derive;
extern crate toml;

mod commands;
mod dot_package;
pub mod dots;
pub mod plan;

use std::io::Write;

use clap::Parser;
use env_logger::fmt::Formatter;
use env_logger::Builder;
use utils::stylize::Stylable;

mod styles {
    use utils::{style, stylize::Style};

    const LOG: Style = style! { Bold };

    pub const DEBUG_LOG: Style = LOG;
    pub const INFO_LOG: Style = LOG.blue();
    pub const WARN_LOG: Style = LOG.yellow();
    pub const ERROR_LOG: Style = LOG.red();
    pub const TRACE_LOG: Style = LOG;
}

#[derive(Parser)]
#[clap(author, version, about, long_about = None)]
struct Cli {
    #[clap(subcommand)]
    commands: Option<Commands>,
}

#[derive(Subcommand)]
enum Commands {
    /// Downloads the given git repo as a dot
    Add {
        /// A git url that points to a Dot repo containing all your dotfiles
        repo: String,
        /// Will remove pre-existing packages of the same name
        #[clap(long)]
        overwrite: bool,
    },

    /// Installs all Dots
    Install {
        /// An optional git url that points to a Dot repo that you want to add before installing
        repo: Option<String>,

        /// Will remove pre-existing dots of the same name
        #[clap(long)]
        overwrite: bool,

        /// Will remove pre-existing directories when creating symlinks
        #[clap(short, long)]
        force: bool,

        /// Run through the install plan without actually making any changes
        #[clap(long)]
        dry: bool,
    },

    /// List the names of all installed dots
    #[clap(alias = "ls")]
    List {
        /// List the git origin of each dot
        #[clap(long)]
        origins: bool,
    },

    /// Returns the installed location of a given dot
    Path {
        /// The dot package name that you would like to search for
        dot: String,
    },
}

fn main() {
    let mut builder = Builder::new();

    let log_format = |buf: &mut Formatter, record: &log::Record| -> Result<(), std::io::Error> {
        use log::Level::*;
        let level = match record.level() {
            Debug => "[debug]".apply_style(styles::DEBUG_LOG),
            Info => "[info]".apply_style(styles::INFO_LOG),
            Warn => "[warn]".apply_style(styles::WARN_LOG),
            Error => "[error]".apply_style(styles::ERROR_LOG),
            Trace => "[trace]".apply_style(styles::TRACE_LOG),
        };
        let string = format!("{}", args = record.args());
        let indented = string
            .lines()
            .enumerate()
            .map(|(i, line)| {
                let mut indent = "";
                let mut new_line = "";
                if i > 0 {
                    indent = "  ";
                    new_line = "\n"
                }
                format!("{new_line}{level} {indent}{line}")
            })
            .collect::<String>();
        writeln!(buf, "{}", indented)
    };

    builder
        .format(log_format)
        .filter(None, log::LevelFilter::Info)
        .init();

    let cli = Cli::parse();

    match &cli.commands {
        Some(Commands::Add { repo, overwrite }) => commands::add(repo, overwrite),
        Some(Commands::Install {
            repo,
            overwrite,
            force,
            dry,
        }) => commands::install(repo, overwrite, force, dry),
        Some(Commands::List { origins }) => commands::list(origins),
        Some(Commands::Path { dot }) => commands::path(dot),
        _ => {
            println!("USAGE:\n    dots [SUBCOMMAND]")
        }
    }
}
