use std::collections::HashMap;
use std::sync::Arc;

use tera::Context;
use tokio::sync::Mutex;

use crate::config::FileSource;
use crate::error;
use crate::source_store::SourceStore;

pub fn get_file_name_from_source(file_source: &FileSource) -> &str {
    match file_source {
        FileSource::local(file) => file,
        FileSource::git { file, .. } => file,
    }
}

pub async fn load(
    contexts: HashMap<String, FileSource>,
    source_store: Arc<Mutex<SourceStore>>,
) -> error::Result<Context> {
    let mut context = Context::new();

    for (root_name, file_source) in contexts {
        let file_name = get_file_name_from_source(&file_source);

        let file_content = match source_store.lock().await.get_content(&file_source) {
            Ok(f) => f,
            Err(_) => continue,
        };

        let path = std::path::Path::new(file_name);
        let value: serde_yaml::Value = match path.extension().unwrap_or_default().to_str().unwrap()
        {
            "yaml" => serde_yaml::from_str(&file_content).unwrap(),
            _ => serde_yaml::Value::String(file_content.into()),
        };

        context.insert(root_name, &value);
    }

    Ok(context)
}
