use std::sync::Arc;

use mlua::{ExternalResult, LuaSerdeExt, UserData};
use tokio::sync::Mutex;
use crate::database::DataBaseManager;
use crate::value::DataValue;

#[derive(Clone)]
pub struct PluginDbManager {
    db: Arc<Mutex<DataBaseManager>>,
    current: String,
}

impl PluginDbManager {

    pub async fn init (db: Arc<Mutex<DataBaseManager>>, current: String) -> Self {
        Self { db: db.clone(), current }
    }

}

impl UserData for PluginDbManager {
    fn add_methods<'lua, M: mlua::UserDataMethods<'lua, Self>>(methods: &mut M) {
 
        methods.add_async_method("select", |_, mut this, db_name: String| async move {
            this.current = db_name.clone();
            this.db.lock().await.select_to(&db_name).to_lua_err()
        });
 
        methods.add_async_method(
            "setex", |_, this, (key, (value, expire)): (String, (String, u64)
        )| async move {
            this.db.lock().await.db_list.get_mut(&this.current).unwrap()
            .set(&key, DataValue::from(&value), expire).await.to_lua_err()
        });

        methods.add_async_method("get", |lua, this, key: String| async move {
            let val = this.db.lock().await.db_list.get_mut(&this.current).unwrap()
            .get(&key).await;

            let val = match val {
                Some(v) => {
                    let datatype = v.datatype();
                    match datatype.as_str() {
                        "None" => Ok(mlua::Value::Nil),
                        "String" => lua.to_value(&v.as_string()),
                        "Number" => lua.to_value(&v.as_number()),
                        "Boolean" => lua.to_value(&v.as_bool()),
                        "List" => lua.to_value(&v.as_list()),
                        "Dict" => lua.to_value(&v.as_dict()),
                        "Tuple" => lua.to_value(&v.as_tuple()),
                        _ => lua.to_value(&v)
                    }
                },
                None => Ok(mlua::Value::Nil),
            };

            Ok(val)
        });

        methods.add_async_method("delete", |_, this, key: String| async move {
            this.db.lock().await.db_list.get_mut(&this.current).unwrap()
            .delete(&key).await.to_lua_err()
        });

        methods.add_async_method("exist", |_, this, key: String| async move {
            Ok(this.db.lock().await.db_list.get_mut(&this.current).unwrap()
            .contains_key(&key).await)
        });

    }

    fn add_fields<'lua, F: mlua::UserDataFields<'lua, Self>>(_fields: &mut F) {}
}