//! The data types sent to and returned from the API client.
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

/**
* The business type.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BusinessType {
    #[serde(rename = "company")]
    Company,
    #[serde(rename = "government_entity")]
    GovernmentEntity,
    #[serde(rename = "individual")]
    Individual,
    #[serde(rename = "non_profit")]
    NonProfit,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BusinessType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BusinessType::Company => "company",
            BusinessType::GovernmentEntity => "government_entity",
            BusinessType::Individual => "individual",
            BusinessType::NonProfit => "non_profit",
            BusinessType::Noop => "",
            BusinessType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BusinessType {
    fn default() -> BusinessType {
        BusinessType::Noop
    }
}
impl BusinessType {
    pub fn is_noop(&self) -> bool {
        matches!(self, BusinessType::Noop)
    }
}

/// All of the following types:
///
/// - `BankAccount`
/// - `Card`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DataAnyOf {
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    BankAccount(BankAccount),
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    Card(Card),
}

impl DataAnyOf {
    pub fn bank_account(&self) -> Option<&BankAccount> {
        if let DataAnyOf::BankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn card(&self) -> Option<&Card> {
        if let DataAnyOf::Card(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
* String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Object {
    #[serde(rename = "list")]
    List,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Object {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Object::List => "list",
            Object::Noop => "",
            Object::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Object {
    fn default() -> Object {
        Object::Noop
    }
}
impl Object {
    pub fn is_noop(&self) -> bool {
        matches!(self, Object::Noop)
    }
}

/// External accounts (bank accounts and debit cards) currently attached to this account
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ExternalAccounts {
    /**
    * The list contains all external accounts that have been attached to the Stripe account. These may be bank accounts or cards.
    */
    #[serde()]
    pub data: Box<Vec<DataAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AccountObject {
    #[serde(rename = "account")]
    Account,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AccountObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountObject::Account => "account",
            AccountObject::Noop => "",
            AccountObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AccountObject {
    fn default() -> AccountObject {
        AccountObject::Noop
    }
}
impl AccountObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountObject::Noop)
    }
}

/**
* The Stripe account type. Can be `standard`, `express`, or `custom`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Type {
    #[serde(rename = "custom")]
    Custom,
    #[serde(rename = "express")]
    Express,
    #[serde(rename = "standard")]
    Standard,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Type {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Type::Custom => "custom",
            Type::Express => "express",
            Type::Standard => "standard",
            Type::Noop => "",
            Type::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Type {
    fn default() -> Type {
        Type::Noop
    }
}
impl Type {
    pub fn is_noop(&self) -> bool {
        matches!(self, Type::Noop)
    }
}

/// This is an object representing a Stripe account. You can retrieve it to see
/// properties on the account like its current e-mail address or if the account is
/// enabled yet to make live charges.
///
/// Some properties, marked below, are available only to platforms that want to
/// [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Account {
    /**
    * Business information about the account.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub business_profile: Option<AccountBusinessProfile>,
    /**
    * The business type.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub business_type: Option<BusinessType>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub capabilities: Option<AccountCapabilities>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub charges_enabled: bool,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub company: Option<LegalEntityCompany>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub controller: Option<AccountUnificationController>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub default_currency: String,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub details_submitted: bool,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub external_accounts: Option<ExternalAccounts>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub future_requirements: Option<AccountFutureRequirements>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub individual: Option<Person>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "AccountObject::is_noop")]
    pub object: AccountObject,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub payouts_enabled: bool,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub requirements: Option<AccountRequirements>,
    /**
    * Options for customizing how the account functions within Stripe.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub settings: Option<AccountSettings>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tos_acceptance: Option<AccountTosAcceptance>,
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<Type>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountBacsDebitPaymentsSettings {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
}

/// All of the following types:
///
/// - `String`
/// - `File`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum IconAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * This is an object representing a file hosted on Stripe's servers. The
    *  file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
    *  request (for example, when uploading dispute evidence) or it may have
    *  been created by Stripe (for example, the results of a [Sigma scheduled
    *  query](#scheduled_queries)).
    *  
    *  Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
    */
    File(File),
}

impl IconAnyOf {
    pub fn file(&self) -> Option<&File> {
        if let IconAnyOf::File(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let IconAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for IconAnyOf {
    fn from(f: String) -> Self {
        IconAnyOf::String(f)
    }
}

impl std::convert::From<IconAnyOf> for String {
    fn from(f: IconAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountBrandingSettings {
    /**
    *
    */
    #[serde()]
    pub icon: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub logo: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub primary_color: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub secondary_color: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountBusinessProfile {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub mcc: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub product_description: String,
    /**
    * A publicly available mailing address for sending support issues to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub support_address: Option<Address>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_email: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_phone: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub support_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* The status of the transfers capability of the account, or whether your platform can transfer funds to the account.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Transfers {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "inactive")]
    Inactive,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Transfers {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Transfers::Active => "active",
            Transfers::Inactive => "inactive",
            Transfers::Pending => "pending",
            Transfers::Noop => "",
            Transfers::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Transfers {
    fn default() -> Transfers {
        Transfers::Noop
    }
}
impl Transfers {
    pub fn is_noop(&self) -> bool {
        matches!(self, Transfers::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountCapabilities {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub acss_debit_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub afterpay_clearpay_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub au_becs_debit_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bacs_debit_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bancontact_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bank_transfer_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub boleto_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card_issuing: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub cartes_bancaires_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub eps_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub fpx_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub giropay_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub grabpay_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ideal_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub jcb_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub klarna_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub konbini_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub legacy_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub oxxo_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        rename = "p24_payments"
    )]
    pub p_24_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub paynow_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sepa_debit_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sofort_payments: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tax_reporting_us_1099_k: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tax_reporting_us_1099_misc: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transfers: Option<Transfers>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub us_bank_account_ach_payments: Option<Transfers>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountCapabilityFutureRequirements {
    /**
    * Fields that are due and can be satisfied by providing the corresponding alternative fields instead.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub alternatives: Vec<AccountRequirementsAlternative>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub current_deadline: i64,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub currently_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub disabled_reason: String,
    /**
    * Fields that are `currently_due` and need to be collected again because validation or verification failed.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub errors: Vec<AccountRequirementsError>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub eventually_due: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub past_due: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub pending_verification: Vec<String>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountCapabilityRequirements {
    /**
    * Fields that are due and can be satisfied by providing the corresponding alternative fields instead.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub alternatives: Vec<AccountRequirementsAlternative>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub current_deadline: i64,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub currently_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub disabled_reason: String,
    /**
    * Fields that are `currently_due` and need to be collected again because validation or verification failed.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub errors: Vec<AccountRequirementsError>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub eventually_due: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub past_due: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub pending_verification: Vec<String>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountCardIssuingSettings {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tos_acceptance: Option<CardIssuingAccountTermsOfService>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountCardPaymentsSettings {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub decline_on: Option<AccountDeclineChargeOn>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor_prefix: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountDashboardSettings {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub display_name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountDeclineChargeOn {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub avs_failure: bool,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cvc_failure: bool,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountFutureRequirements {
    /**
    * Fields that are due and can be satisfied by providing the corresponding alternative fields instead.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub alternatives: Vec<AccountRequirementsAlternative>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub current_deadline: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub currently_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub disabled_reason: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub errors: Vec<AccountRequirementsError>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub eventually_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub past_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub pending_verification: Vec<String>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AccountLinkObject {
    #[serde(rename = "account_link")]
    AccountLink,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AccountLinkObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountLinkObject::AccountLink => "account_link",
            AccountLinkObject::Noop => "",
            AccountLinkObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AccountLinkObject {
    fn default() -> AccountLinkObject {
        AccountLinkObject::Noop
    }
}
impl AccountLinkObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountLinkObject::Noop)
    }
}

/// Account Links are the means by which a Connect platform grants a connected account permission to access
/// Stripe-hosted applications, such as Connect Onboarding.
///
/// Related guide: [Connect Onboarding](https://stripe.com/docs/connect/connect-onboarding).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountLink {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_at: i64,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "AccountLinkObject::is_noop")]
    pub object: AccountLinkObject,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountPaymentsSettings {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor_kana: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor_kanji: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountPayoutSettings {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub debit_negative_balances: bool,
    /**
    *
    */
    #[serde()]
    pub schedule: TransferSchedule,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountRequirements {
    /**
    * Fields that are due and can be satisfied by providing the corresponding alternative fields instead.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub alternatives: Vec<AccountRequirementsAlternative>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub current_deadline: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub currently_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub disabled_reason: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub errors: Vec<AccountRequirementsError>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub eventually_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub past_due: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub pending_verification: Vec<String>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountRequirementsAlternative {
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub alternative_fields_due: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub original_fields_due: Vec<String>,
}

/**
* The code for the type of error.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Code {
    #[serde(rename = "invalid_address_city_state_postal_code")]
    InvalidAddressCityStatePostalCode,
    #[serde(rename = "invalid_street_address")]
    InvalidStreetAddress,
    #[serde(rename = "invalid_value_other")]
    InvalidValueOther,
    #[serde(rename = "verification_document_address_mismatch")]
    VerificationDocumentAddressMismatch,
    #[serde(rename = "verification_document_address_missing")]
    VerificationDocumentAddressMissing,
    #[serde(rename = "verification_document_corrupt")]
    VerificationDocumentCorrupt,
    #[serde(rename = "verification_document_country_not_supported")]
    VerificationDocumentCountryNotSupported,
    #[serde(rename = "verification_document_dob_mismatch")]
    VerificationDocumentDobMismatch,
    #[serde(rename = "verification_document_duplicate_type")]
    VerificationDocumentDuplicateType,
    #[serde(rename = "verification_document_expired")]
    VerificationDocumentExpired,
    #[serde(rename = "verification_document_failed_copy")]
    VerificationDocumentFailedCopy,
    #[serde(rename = "verification_document_failed_greyscale")]
    VerificationDocumentFailedGreyscale,
    #[serde(rename = "verification_document_failed_other")]
    VerificationDocumentFailedOther,
    #[serde(rename = "verification_document_failed_test_mode")]
    VerificationDocumentFailedTestMode,
    #[serde(rename = "verification_document_fraudulent")]
    VerificationDocumentFraudulent,
    #[serde(rename = "verification_document_id_number_mismatch")]
    VerificationDocumentNumberMismatch,
    #[serde(rename = "verification_document_id_number_missing")]
    VerificationDocumentNumberMissing,
    #[serde(rename = "verification_document_incomplete")]
    VerificationDocumentIncomplete,
    #[serde(rename = "verification_document_invalid")]
    VerificationDocumentInvalid,
    #[serde(rename = "verification_document_issue_or_expiry_date_missing")]
    VerificationDocumentIssueOrExpiryDateMissing,
    #[serde(rename = "verification_document_manipulated")]
    VerificationDocumentManipulated,
    #[serde(rename = "verification_document_missing_back")]
    VerificationDocumentMissingBack,
    #[serde(rename = "verification_document_missing_front")]
    VerificationDocumentMissingFront,
    #[serde(rename = "verification_document_name_mismatch")]
    VerificationDocumentNameMismatch,
    #[serde(rename = "verification_document_name_missing")]
    VerificationDocumentNameMissing,
    #[serde(rename = "verification_document_nationality_mismatch")]
    VerificationDocumentNationalityMismatch,
    #[serde(rename = "verification_document_not_readable")]
    VerificationDocumentNotReadable,
    #[serde(rename = "verification_document_not_signed")]
    VerificationDocumentNotSigned,
    #[serde(rename = "verification_document_not_uploaded")]
    VerificationDocumentNotUploaded,
    #[serde(rename = "verification_document_photo_mismatch")]
    VerificationDocumentPhotoMismatch,
    #[serde(rename = "verification_document_too_large")]
    VerificationDocumentTooLarge,
    #[serde(rename = "verification_document_type_not_supported")]
    VerificationDocumentTypeNotSupported,
    #[serde(rename = "verification_failed_address_match")]
    VerificationFailedAddressMatch,
    #[serde(rename = "verification_failed_business_iec_number")]
    VerificationFailedBusinessIecNumber,
    #[serde(rename = "verification_failed_document_match")]
    VerificationFailedDocumentMatch,
    #[serde(rename = "verification_failed_id_number_match")]
    VerificationFailedNumberMatch,
    #[serde(rename = "verification_failed_keyed_identity")]
    VerificationFailedKeyedIdentity,
    #[serde(rename = "verification_failed_keyed_match")]
    VerificationFailedKeyedMatch,
    #[serde(rename = "verification_failed_name_match")]
    VerificationFailedNameMatch,
    #[serde(rename = "verification_failed_other")]
    VerificationFailedOther,
    #[serde(rename = "verification_failed_tax_id_match")]
    VerificationFailedTaxMatch,
    #[serde(rename = "verification_failed_tax_id_not_issued")]
    VerificationFailedTaxNotIssued,
    #[serde(rename = "verification_missing_executives")]
    VerificationMissingExecutives,
    #[serde(rename = "verification_missing_owners")]
    VerificationMissingOwners,
    #[serde(rename = "verification_requires_additional_memorandum_of_associations")]
    VerificationRequiresAdditionalMemorandumOfAssociations,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Code {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Code::InvalidAddressCityStatePostalCode => "invalid_address_city_state_postal_code",
            Code::InvalidStreetAddress => "invalid_street_address",
            Code::InvalidValueOther => "invalid_value_other",
            Code::VerificationDocumentAddressMismatch => "verification_document_address_mismatch",
            Code::VerificationDocumentAddressMissing => "verification_document_address_missing",
            Code::VerificationDocumentCorrupt => "verification_document_corrupt",
            Code::VerificationDocumentCountryNotSupported => {
                "verification_document_country_not_supported"
            }
            Code::VerificationDocumentDobMismatch => "verification_document_dob_mismatch",
            Code::VerificationDocumentDuplicateType => "verification_document_duplicate_type",
            Code::VerificationDocumentExpired => "verification_document_expired",
            Code::VerificationDocumentFailedCopy => "verification_document_failed_copy",
            Code::VerificationDocumentFailedGreyscale => "verification_document_failed_greyscale",
            Code::VerificationDocumentFailedOther => "verification_document_failed_other",
            Code::VerificationDocumentFailedTestMode => "verification_document_failed_test_mode",
            Code::VerificationDocumentFraudulent => "verification_document_fraudulent",
            Code::VerificationDocumentNumberMismatch => "verification_document_id_number_mismatch",
            Code::VerificationDocumentNumberMissing => "verification_document_id_number_missing",
            Code::VerificationDocumentIncomplete => "verification_document_incomplete",
            Code::VerificationDocumentInvalid => "verification_document_invalid",
            Code::VerificationDocumentIssueOrExpiryDateMissing => {
                "verification_document_issue_or_expiry_date_missing"
            }
            Code::VerificationDocumentManipulated => "verification_document_manipulated",
            Code::VerificationDocumentMissingBack => "verification_document_missing_back",
            Code::VerificationDocumentMissingFront => "verification_document_missing_front",
            Code::VerificationDocumentNameMismatch => "verification_document_name_mismatch",
            Code::VerificationDocumentNameMissing => "verification_document_name_missing",
            Code::VerificationDocumentNationalityMismatch => {
                "verification_document_nationality_mismatch"
            }
            Code::VerificationDocumentNotReadable => "verification_document_not_readable",
            Code::VerificationDocumentNotSigned => "verification_document_not_signed",
            Code::VerificationDocumentNotUploaded => "verification_document_not_uploaded",
            Code::VerificationDocumentPhotoMismatch => "verification_document_photo_mismatch",
            Code::VerificationDocumentTooLarge => "verification_document_too_large",
            Code::VerificationDocumentTypeNotSupported => {
                "verification_document_type_not_supported"
            }
            Code::VerificationFailedAddressMatch => "verification_failed_address_match",
            Code::VerificationFailedBusinessIecNumber => "verification_failed_business_iec_number",
            Code::VerificationFailedDocumentMatch => "verification_failed_document_match",
            Code::VerificationFailedNumberMatch => "verification_failed_id_number_match",
            Code::VerificationFailedKeyedIdentity => "verification_failed_keyed_identity",
            Code::VerificationFailedKeyedMatch => "verification_failed_keyed_match",
            Code::VerificationFailedNameMatch => "verification_failed_name_match",
            Code::VerificationFailedOther => "verification_failed_other",
            Code::VerificationFailedTaxMatch => "verification_failed_tax_id_match",
            Code::VerificationFailedTaxNotIssued => "verification_failed_tax_id_not_issued",
            Code::VerificationMissingExecutives => "verification_missing_executives",
            Code::VerificationMissingOwners => "verification_missing_owners",
            Code::VerificationRequiresAdditionalMemorandumOfAssociations => {
                "verification_requires_additional_memorandum_of_associations"
            }
            Code::Noop => "",
            Code::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Code {
    fn default() -> Code {
        Code::Noop
    }
}
impl Code {
    pub fn is_noop(&self) -> bool {
        matches!(self, Code::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountRequirementsError {
    /**
    * The code for the type of error.
    */
    #[serde(default, skip_serializing_if = "Code::is_noop")]
    pub code: Code,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub requirement: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountSepaDebitPaymentsSettings {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub creditor_id: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountSettings {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bacs_debit_payments: Option<AccountBacsDebitPaymentsSettings>,
    /**
    *
    */
    #[serde()]
    pub branding: AccountBrandingSettings,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card_issuing: Option<AccountCardIssuingSettings>,
    /**
    *
    */
    #[serde()]
    pub card_payments: AccountCardPaymentsSettings,
    /**
    *
    */
    #[serde()]
    pub dashboard: AccountDashboardSettings,
    /**
    *
    */
    #[serde()]
    pub payments: AccountPaymentsSettings,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payouts: Option<AccountPayoutSettings>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sepa_debit_payments: Option<AccountSepaDebitPaymentsSettings>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountTosAcceptance {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub date: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub service_agreement: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_agent: String,
}

/**
* The controller type. Can be `application`, if a Connect application controls the account, or `account`, if the account controls itself.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AccountUnificationControllerType {
    #[serde(rename = "account")]
    Account,
    #[serde(rename = "application")]
    Application,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AccountUnificationControllerType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountUnificationControllerType::Account => "account",
            AccountUnificationControllerType::Application => "application",
            AccountUnificationControllerType::Noop => "",
            AccountUnificationControllerType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AccountUnificationControllerType {
    fn default() -> AccountUnificationControllerType {
        AccountUnificationControllerType::Noop
    }
}
impl AccountUnificationControllerType {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountUnificationControllerType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AccountUnificationController {
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub is_controller: bool,
    /**
    * The controller type. Can be `application`, if a Connect application controls the account, or `account`, if the account controls itself.
    */
    #[serde(
        default,
        skip_serializing_if = "AccountUnificationControllerType::is_noop",
        rename = "type"
    )]
    pub type_: AccountUnificationControllerType,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Address {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "line1"
    )]
    pub line_1: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "line2"
    )]
    pub line_2: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub postal_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
}

/// All of the following types:
///
/// - `String`
/// - `Customer`
/// - `DeletedCustomer`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CustomerAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    Customer(Customer),
    /**
    *
    */
    DeletedCustomer(DeletedCustomer),
}

impl CustomerAnyOf {
    pub fn customer(&self) -> Option<&Customer> {
        if let CustomerAnyOf::Customer(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn deleted_customer(&self) -> Option<&DeletedCustomer> {
        if let CustomerAnyOf::DeletedCustomer(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let CustomerAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for CustomerAnyOf {
    fn from(f: String) -> Self {
        CustomerAnyOf::String(f)
    }
}

impl std::convert::From<CustomerAnyOf> for String {
    fn from(f: CustomerAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AlipayAccountObject {
    #[serde(rename = "alipay_account")]
    AlipayAccount,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AlipayAccountObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AlipayAccountObject::AlipayAccount => "alipay_account",
            AlipayAccountObject::Noop => "",
            AlipayAccountObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AlipayAccountObject {
    fn default() -> AlipayAccountObject {
        AlipayAccountObject::Noop
    }
}
impl AlipayAccountObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, AlipayAccountObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AlipayAccount {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    *
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "AlipayAccountObject::is_noop")]
    pub object: AlipayAccountObject,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub payment_amount: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_currency: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub reusable: bool,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub used: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub username: String,
}

/// All of the following types:
///
/// - `BankAccount`
/// - `Card`
/// - `SourceData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SourceAnyOf {
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    BankAccount(BankAccount),
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    Card(Card),
    /**
    * `Source` objects allow you to accept a variety of payment methods. They
    *  represent a customer's payment instrument, and can be used with the Stripe API
    *  just like a `Card` object: once chargeable, they can be charged, or can be
    *  attached to customers.
    *  
    *  Related guides: [Sources API](https://stripe.com/docs/sources) and [Sources & Customers](https://stripe.com/docs/sources/customers).
    */
    SourceData(SourceData),
}

impl SourceAnyOf {
    pub fn bank_account(&self) -> Option<&BankAccount> {
        if let SourceAnyOf::BankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn card(&self) -> Option<&Card> {
        if let SourceAnyOf::Card(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn source_data(&self) -> Option<&SourceData> {
        if let SourceAnyOf::SourceData(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
* The type of error returned. One of `api_error`, `card_error`, `idempotency_error`, or `invalid_request_error`
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ApiErrorsType {
    #[serde(rename = "api_error")]
    ApiError,
    #[serde(rename = "card_error")]
    CardError,
    #[serde(rename = "idempotency_error")]
    IdempotencyError,
    #[serde(rename = "invalid_request_error")]
    InvalidRequestError,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ApiErrorsType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ApiErrorsType::ApiError => "api_error",
            ApiErrorsType::CardError => "card_error",
            ApiErrorsType::IdempotencyError => "idempotency_error",
            ApiErrorsType::InvalidRequestError => "invalid_request_error",
            ApiErrorsType::Noop => "",
            ApiErrorsType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ApiErrorsType {
    fn default() -> ApiErrorsType {
        ApiErrorsType::Noop
    }
}
impl ApiErrorsType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ApiErrorsType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ApiErrors {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub charge: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub decline_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub doc_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub message: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub param: String,
    /**
    *
    */
    #[serde()]
    pub payment_intent: Box<Option<PaymentIntent>>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_method: Option<PaymentMethod>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_method_type: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_intent: Option<SetupIntent>,
    /**
    *
    */
    #[serde()]
    pub source: Box<Option<SourceAnyOf>>,
    /**
    * The type of error returned. One of `api_error`, `card_error`, `idempotency_error`, or `invalid_request_error`
    */
    #[serde(
        default,
        skip_serializing_if = "ApiErrorsType::is_noop",
        rename = "type"
    )]
    pub type_: ApiErrorsType,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ApplePayDomainObject {
    #[serde(rename = "apple_pay_domain")]
    ApplePayDomain,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ApplePayDomainObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ApplePayDomainObject::ApplePayDomain => "apple_pay_domain",
            ApplePayDomainObject::Noop => "",
            ApplePayDomainObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ApplePayDomainObject {
    fn default() -> ApplePayDomainObject {
        ApplePayDomainObject::Noop
    }
}
impl ApplePayDomainObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, ApplePayDomainObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ApplePayDomain {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub domain_name: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "ApplePayDomainObject::is_noop")]
    pub object: ApplePayDomainObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ApplicationObject {
    #[serde(rename = "application")]
    Application,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ApplicationObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ApplicationObject::Application => "application",
            ApplicationObject::Noop => "",
            ApplicationObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ApplicationObject {
    fn default() -> ApplicationObject {
        ApplicationObject::Noop
    }
}
impl ApplicationObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, ApplicationObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Application {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "ApplicationObject::is_noop")]
    pub object: ApplicationObject,
}

/// All of the following types:
///
/// - `String`
/// - `Account`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * This is an object representing a Stripe account. You can retrieve it to see
    *  properties on the account like its current e-mail address or if the account is
    *  enabled yet to make live charges.
    *  
    *  Some properties, marked below, are available only to platforms that want to
    *  [create and manage Express or Custom accounts](https://stripe.com/docs/connect/accounts).
    */
    Account(Account),
}

impl AccountAnyOf {
    pub fn account(&self) -> Option<&Account> {
        if let AccountAnyOf::Account(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let AccountAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for AccountAnyOf {
    fn from(f: String) -> Self {
        AccountAnyOf::String(f)
    }
}

impl std::convert::From<AccountAnyOf> for String {
    fn from(f: AccountAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `Application`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ApplicationAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    *
    */
    Application(Application),
}

impl ApplicationAnyOf {
    pub fn application(&self) -> Option<&Application> {
        if let ApplicationAnyOf::Application(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let ApplicationAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for ApplicationAnyOf {
    fn from(f: String) -> Self {
        ApplicationAnyOf::String(f)
    }
}

impl std::convert::From<ApplicationAnyOf> for String {
    fn from(f: ApplicationAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `BalanceTransaction`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum BalanceTransactionAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Balance transactions represent funds moving through your Stripe account.
    *  They're created for every type of transaction that comes into or flows out of your Stripe account balance.
    *  
    *  Related guide: [Balance Transaction Types](https://stripe.com/docs/reports/balance-transaction-types).
    */
    BalanceTransaction(BalanceTransaction),
}

impl BalanceTransactionAnyOf {
    pub fn balance_transaction(&self) -> Option<&BalanceTransaction> {
        if let BalanceTransactionAnyOf::BalanceTransaction(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let BalanceTransactionAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for BalanceTransactionAnyOf {
    fn from(f: String) -> Self {
        BalanceTransactionAnyOf::String(f)
    }
}

impl std::convert::From<BalanceTransactionAnyOf> for String {
    fn from(f: BalanceTransactionAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `Charge`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ChargeAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    Charge(Charge),
}

impl ChargeAnyOf {
    pub fn charge(&self) -> Option<&Charge> {
        if let ChargeAnyOf::Charge(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let ChargeAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for ChargeAnyOf {
    fn from(f: String) -> Self {
        ChargeAnyOf::String(f)
    }
}

impl std::convert::From<ChargeAnyOf> for String {
    fn from(f: ChargeAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PlatformFeeObject {
    #[serde(rename = "application_fee")]
    ApplicationFee,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PlatformFeeObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PlatformFeeObject::ApplicationFee => "application_fee",
            PlatformFeeObject::Noop => "",
            PlatformFeeObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PlatformFeeObject {
    fn default() -> PlatformFeeObject {
        PlatformFeeObject::Noop
    }
}
impl PlatformFeeObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, PlatformFeeObject::Noop)
    }
}

/// A list of refunds that have been applied to the fee.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Refunds {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<FeeRefund>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PlatformFee {
    #[serde()]
    pub account: Box<AccountAnyOf>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_refunded: i64,
    #[serde()]
    pub application: Box<ApplicationAnyOf>,
    /**
    *
    */
    #[serde()]
    pub balance_transaction: Box<Option<BalanceTransactionAnyOf>>,
    #[serde()]
    pub charge: Box<ChargeAnyOf>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "PlatformFeeObject::is_noop")]
    pub object: PlatformFeeObject,
    /**
    *
    */
    #[serde()]
    pub originating_transaction: Box<Option<ChargeAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub refunded: bool,
    /**
    * A list of refunds that have been applied to the fee.
    */
    #[serde()]
    pub refunds: Refunds,
}

/**
* The status of the most recent automated tax calculation for this invoice.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Status {
    #[serde(rename = "complete")]
    Complete,
    #[serde(rename = "failed")]
    Failed,
    #[serde(rename = "requires_location_inputs")]
    RequiresLocationInputs,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Status {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Status::Complete => "complete",
            Status::Failed => "failed",
            Status::RequiresLocationInputs => "requires_location_inputs",
            Status::Noop => "",
            Status::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Status {
    fn default() -> Status {
        Status::Noop
    }
}
impl Status {
    pub fn is_noop(&self) -> bool {
        matches!(self, Status::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AutomaticTax {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enabled: bool,
    /**
    * The status of the most recent automated tax calculation for this invoice.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<Status>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BalanceObject {
    #[serde(rename = "balance")]
    Balance,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BalanceObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BalanceObject::Balance => "balance",
            BalanceObject::Noop => "",
            BalanceObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BalanceObject {
    fn default() -> BalanceObject {
        BalanceObject::Noop
    }
}
impl BalanceObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, BalanceObject::Noop)
    }
}

/// This is an object representing your Stripe balance. You can retrieve it to see
/// the balance currently on your Stripe account.
///
/// You can also retrieve the balance history, which contains a list of
/// [transactions](https://stripe.com/docs/reporting/balance-transaction-types) that contributed to the balance
/// (charges, payouts, and so forth).
///
/// The available and pending amounts for each currency are broken down further by
/// payment source types.
///
/// Related guide: [Understanding Connect Account Balances](https://stripe.com/docs/connect/account-balances).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Balance {
    /**
    * Funds that are available to be transferred or paid out, whether automatically by Stripe or explicitly via the [Transfers API](https://stripe.com/docs/api#transfers) or [Payouts API](https://stripe.com/docs/api#payouts). The available balance for each currency and payment type can be found in the `source_types` property.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub available: Vec<BalanceAmount>,
    /**
    * This is an object representing your Stripe balance. You can retrieve it to see
    *  the balance currently on your Stripe account.
    *  
    *  You can also retrieve the balance history, which contains a list of
    *  [transactions](https://stripe.com/docs/reporting/balance-transaction-types) that contributed to the balance
    *  (charges, payouts, and so forth).
    *  
    *  The available and pending amounts for each currency are broken down further by
    *  payment source types.
    *  
    *  Related guide: [Understanding Connect Account Balances](https://stripe.com/docs/connect/account-balances).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub connect_reserved: Vec<BalanceAmount>,
    /**
    * This is an object representing your Stripe balance. You can retrieve it to see
    *  the balance currently on your Stripe account.
    *  
    *  You can also retrieve the balance history, which contains a list of
    *  [transactions](https://stripe.com/docs/reporting/balance-transaction-types) that contributed to the balance
    *  (charges, payouts, and so forth).
    *  
    *  The available and pending amounts for each currency are broken down further by
    *  payment source types.
    *  
    *  Related guide: [Understanding Connect Account Balances](https://stripe.com/docs/connect/account-balances).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub instant_available: Vec<BalanceAmount>,
    /**
    * This is an object representing your Stripe balance. You can retrieve it to see
    *  the balance currently on your Stripe account.
    *  
    *  You can also retrieve the balance history, which contains a list of
    *  [transactions](https://stripe.com/docs/reporting/balance-transaction-types) that contributed to the balance
    *  (charges, payouts, and so forth).
    *  
    *  The available and pending amounts for each currency are broken down further by
    *  payment source types.
    *  
    *  Related guide: [Understanding Connect Account Balances](https://stripe.com/docs/connect/account-balances).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub issuing: Option<BalanceDetail>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BalanceObject::is_noop")]
    pub object: BalanceObject,
    /**
    * Funds that are available to be transferred or paid out, whether automatically by Stripe or explicitly via the [Transfers API](https://stripe.com/docs/api#transfers) or [Payouts API](https://stripe.com/docs/api#payouts). The available balance for each currency and payment type can be found in the `source_types` property.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub pending: Vec<BalanceAmount>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BalanceAmount {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub source_types: Option<BalanceAmountBySourceType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BalanceAmountBySourceType {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub bank_account: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub card: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub fpx: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BalanceDetail {
    /**
    * Funds that are available to be transferred or paid out, whether automatically by Stripe or explicitly via the [Transfers API](https://stripe.com/docs/api#transfers) or [Payouts API](https://stripe.com/docs/api#payouts). The available balance for each currency and payment type can be found in the `source_types` property.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub available: Vec<BalanceAmount>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BalanceTransactionObject {
    #[serde(rename = "balance_transaction")]
    BalanceTransaction,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BalanceTransactionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BalanceTransactionObject::BalanceTransaction => "balance_transaction",
            BalanceTransactionObject::Noop => "",
            BalanceTransactionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BalanceTransactionObject {
    fn default() -> BalanceTransactionObject {
        BalanceTransactionObject::Noop
    }
}
impl BalanceTransactionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, BalanceTransactionObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `Charge`
/// - `FeeRefund`
/// - `PlatformFee`
/// - `ConnectCollectionTransfer`
/// - `Dispute`
/// - `IssuingAuthorization`
/// - `IssuingDispute`
/// - `IssuingTransaction`
/// - `Payout`
/// - `PlatformTax`
/// - `Refund`
/// - `ReserveTransaction`
/// - `TaxDeductedAtSource`
/// - `Topup`
/// - `Transfer`
/// - `TransferReversal`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum BalanceTransactionSourceAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    Charge(Charge),
    /**
    * `Application Fee Refund` objects allow you to refund an application fee that
    *  has previously been created but not yet refunded. Funds will be refunded to
    *  the Stripe account from which the fee was originally collected.
    *  
    *  Related guide: [Refunding Application Fees](https://stripe.com/docs/connect/destination-charges#refunding-app-fee).
    */
    FeeRefund(FeeRefund),
    /**
    *
    */
    PlatformFee(PlatformFee),
    /**
    *
    */
    ConnectCollectionTransfer(ConnectCollectionTransfer),
    /**
    * A dispute occurs when a customer questions your charge with their card issuer.
    *  When this happens, you're given the opportunity to respond to the dispute with
    *  evidence that shows that the charge is legitimate. You can find more
    *  information about the dispute process in our [Disputes and
    *  Fraud](/docs/disputes) documentation.
    *  
    *  Related guide: [Disputes and Fraud](https://stripe.com/docs/disputes).
    */
    Dispute(Dispute),
    /**
    * When an [issued card](https://stripe.com/docs/issuing) is used to make a purchase, an Issuing `Authorization`
    *  object is created. [Authorizations](https://stripe.com/docs/issuing/purchases/authorizations) must be approved for the
    *  purchase to be completed successfully.
    *  
    *  Related guide: [Issued Card Authorizations](https://stripe.com/docs/issuing/purchases/authorizations).
    */
    IssuingAuthorization(IssuingAuthorization),
    /**
    * As a [card issuer](https://stripe.com/docs/issuing), you can dispute transactions that the cardholder does not recognize, suspects to be fraudulent, or has other issues with.
    *  
    *  Related guide: [Disputing Transactions](https://stripe.com/docs/issuing/purchases/disputes)
    */
    IssuingDispute(IssuingDispute),
    /**
    * Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
    *  your Stripe account, such as a completed purchase or refund, is represented by an Issuing
    *  `Transaction` object.
    *  
    *  Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
    */
    IssuingTransaction(IssuingTransaction),
    /**
    * A `Payout` object is created when you receive funds from Stripe, or when you
    *  initiate a payout to either a bank account or debit card of a [connected
    *  Stripe account](/docs/connect/bank-debit-card-payouts). You can retrieve individual payouts,
    *  as well as list all payouts. Payouts are made on [varying
    *  schedules](/docs/connect/manage-payout-schedule), depending on your country and
    *  industry.
    *  
    *  Related guide: [Receiving Payouts](https://stripe.com/docs/payouts).
    */
    Payout(Payout),
    /**
    *
    */
    PlatformTax(PlatformTax),
    /**
    * `Refund` objects allow you to refund a charge that has previously been created
    *  but not yet refunded. Funds will be refunded to the credit or debit card that
    *  was originally charged.
    *  
    *  Related guide: [Refunds](https://stripe.com/docs/refunds).
    */
    Refund(Refund),
    /**
    *
    */
    ReserveTransaction(ReserveTransaction),
    /**
    *
    */
    TaxDeductedAtSource(TaxDeductedAtSource),
    /**
    * To top up your Stripe balance, you create a top-up object. You can retrieve
    *  individual top-ups, as well as list all top-ups. Top-ups are identified by a
    *  unique, random ID.
    *  
    *  Related guide: [Topping Up your Platform Account](https://stripe.com/docs/connect/top-ups).
    */
    Topup(Topup),
    /**
    * A `Transfer` object is created when you move funds between Stripe accounts as
    *  part of Connect.
    *  
    *  Before April 6, 2017, transfers also represented movement of funds from a
    *  Stripe account to a card or bank account. This behavior has since been split
    *  out into a [Payout](https://stripe.com/docs/api#payout_object) object, with corresponding payout endpoints. For more
    *  information, read about the
    *  [transfer/payout split](https://stripe.com/docs/transfer-payout-split).
    *  
    *  Related guide: [Creating Separate Charges and Transfers](https://stripe.com/docs/connect/charges-transfers).
    */
    Transfer(Transfer),
    /**
    * [Stripe Connect](https://stripe.com/docs/connect) platforms can reverse transfers made to a
    *  connected account, either entirely or partially, and can also specify whether
    *  to refund any related application fees. Transfer reversals add to the
    *  platform's balance and subtract from the destination account's balance.
    *  
    *  Reversing a transfer that was made for a [destination
    *  charge](/docs/connect/destination-charges) is allowed only up to the amount of
    *  the charge. It is possible to reverse a
    *  [transfer_group](https://stripe.com/docs/connect/charges-transfers#transfer-options)
    *  transfer only if the destination account has enough balance to cover the
    *  reversal.
    *  
    *  Related guide: [Reversing Transfers](https://stripe.com/docs/connect/charges-transfers#reversing-transfers).
    */
    TransferReversal(TransferReversal),
}

impl BalanceTransactionSourceAnyOf {
    pub fn charge(&self) -> Option<&Charge> {
        if let BalanceTransactionSourceAnyOf::Charge(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn connect_collection_transfer(&self) -> Option<&ConnectCollectionTransfer> {
        if let BalanceTransactionSourceAnyOf::ConnectCollectionTransfer(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn dispute(&self) -> Option<&Dispute> {
        if let BalanceTransactionSourceAnyOf::Dispute(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn fee_refund(&self) -> Option<&FeeRefund> {
        if let BalanceTransactionSourceAnyOf::FeeRefund(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn issuing_authorization(&self) -> Option<&IssuingAuthorization> {
        if let BalanceTransactionSourceAnyOf::IssuingAuthorization(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn issuing_dispute(&self) -> Option<&IssuingDispute> {
        if let BalanceTransactionSourceAnyOf::IssuingDispute(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn issuing_transaction(&self) -> Option<&IssuingTransaction> {
        if let BalanceTransactionSourceAnyOf::IssuingTransaction(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payout(&self) -> Option<&Payout> {
        if let BalanceTransactionSourceAnyOf::Payout(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn platform_fee(&self) -> Option<&PlatformFee> {
        if let BalanceTransactionSourceAnyOf::PlatformFee(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn platform_tax(&self) -> Option<&PlatformTax> {
        if let BalanceTransactionSourceAnyOf::PlatformTax(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn refund(&self) -> Option<&Refund> {
        if let BalanceTransactionSourceAnyOf::Refund(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn reserve_transaction(&self) -> Option<&ReserveTransaction> {
        if let BalanceTransactionSourceAnyOf::ReserveTransaction(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let BalanceTransactionSourceAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn tax_deducted_at_source(&self) -> Option<&TaxDeductedAtSource> {
        if let BalanceTransactionSourceAnyOf::TaxDeductedAtSource(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn topup(&self) -> Option<&Topup> {
        if let BalanceTransactionSourceAnyOf::Topup(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn transfer(&self) -> Option<&Transfer> {
        if let BalanceTransactionSourceAnyOf::Transfer(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn transfer_reversal(&self) -> Option<&TransferReversal> {
        if let BalanceTransactionSourceAnyOf::TransferReversal(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for BalanceTransactionSourceAnyOf {
    fn from(f: String) -> Self {
        BalanceTransactionSourceAnyOf::String(f)
    }
}

impl std::convert::From<BalanceTransactionSourceAnyOf> for String {
    fn from(f: BalanceTransactionSourceAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* Transaction type: `adjustment`, `advance`, `advance_funding`, `anticipation_repayment`, `application_fee`, `application_fee_refund`, `charge`, `connect_collection_transfer`, `contribution`, `issuing_authorization_hold`, `issuing_authorization_release`, `issuing_dispute`, `issuing_transaction`, `payment`, `payment_failure_refund`, `payment_refund`, `payout`, `payout_cancel`, `payout_failure`, `refund`, `refund_failure`, `reserve_transaction`, `reserved_funds`, `stripe_fee`, `stripe_fx_fee`, `tax_fee`, `topup`, `topup_reversal`, `transfer`, `transfer_cancel`, `transfer_failure`, or `transfer_refund`. [Learn more](https://stripe.com/docs/reports/balance-transaction-types) about balance transaction types and what they represent. If you are looking to classify transactions for accounting purposes, you might want to consider `reporting_category` instead.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BalanceTransactionType {
    #[serde(rename = "adjustment")]
    Adjustment,
    #[serde(rename = "advance")]
    Advance,
    #[serde(rename = "advance_funding")]
    AdvanceFunding,
    #[serde(rename = "anticipation_repayment")]
    AnticipationRepayment,
    #[serde(rename = "application_fee")]
    ApplicationFee,
    #[serde(rename = "application_fee_refund")]
    ApplicationFeeRefund,
    #[serde(rename = "charge")]
    Charge,
    #[serde(rename = "connect_collection_transfer")]
    ConnectCollectionTransfer,
    #[serde(rename = "contribution")]
    Contribution,
    #[serde(rename = "issuing_authorization_hold")]
    IssuingAuthorizationHold,
    #[serde(rename = "issuing_authorization_release")]
    IssuingAuthorizationRelease,
    #[serde(rename = "issuing_dispute")]
    IssuingDispute,
    #[serde(rename = "issuing_transaction")]
    IssuingTransaction,
    #[serde(rename = "payment")]
    Payment,
    #[serde(rename = "payment_failure_refund")]
    PaymentFailureRefund,
    #[serde(rename = "payment_refund")]
    PaymentRefund,
    #[serde(rename = "payout")]
    Payout,
    #[serde(rename = "payout_cancel")]
    PayoutCancel,
    #[serde(rename = "payout_failure")]
    PayoutFailure,
    #[serde(rename = "refund")]
    Refund,
    #[serde(rename = "refund_failure")]
    RefundFailure,
    #[serde(rename = "reserve_transaction")]
    ReserveTransaction,
    #[serde(rename = "reserved_funds")]
    ReservedFunds,
    #[serde(rename = "stripe_fee")]
    StripeFee,
    #[serde(rename = "stripe_fx_fee")]
    StripeFxFee,
    #[serde(rename = "tax_fee")]
    TaxFee,
    #[serde(rename = "topup")]
    Topup,
    #[serde(rename = "topup_reversal")]
    TopupReversal,
    #[serde(rename = "transfer")]
    Transfer,
    #[serde(rename = "transfer_cancel")]
    TransferCancel,
    #[serde(rename = "transfer_failure")]
    TransferFailure,
    #[serde(rename = "transfer_refund")]
    TransferRefund,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BalanceTransactionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BalanceTransactionType::Adjustment => "adjustment",
            BalanceTransactionType::Advance => "advance",
            BalanceTransactionType::AdvanceFunding => "advance_funding",
            BalanceTransactionType::AnticipationRepayment => "anticipation_repayment",
            BalanceTransactionType::ApplicationFee => "application_fee",
            BalanceTransactionType::ApplicationFeeRefund => "application_fee_refund",
            BalanceTransactionType::Charge => "charge",
            BalanceTransactionType::ConnectCollectionTransfer => "connect_collection_transfer",
            BalanceTransactionType::Contribution => "contribution",
            BalanceTransactionType::IssuingAuthorizationHold => "issuing_authorization_hold",
            BalanceTransactionType::IssuingAuthorizationRelease => "issuing_authorization_release",
            BalanceTransactionType::IssuingDispute => "issuing_dispute",
            BalanceTransactionType::IssuingTransaction => "issuing_transaction",
            BalanceTransactionType::Payment => "payment",
            BalanceTransactionType::PaymentFailureRefund => "payment_failure_refund",
            BalanceTransactionType::PaymentRefund => "payment_refund",
            BalanceTransactionType::Payout => "payout",
            BalanceTransactionType::PayoutCancel => "payout_cancel",
            BalanceTransactionType::PayoutFailure => "payout_failure",
            BalanceTransactionType::Refund => "refund",
            BalanceTransactionType::RefundFailure => "refund_failure",
            BalanceTransactionType::ReserveTransaction => "reserve_transaction",
            BalanceTransactionType::ReservedFunds => "reserved_funds",
            BalanceTransactionType::StripeFee => "stripe_fee",
            BalanceTransactionType::StripeFxFee => "stripe_fx_fee",
            BalanceTransactionType::TaxFee => "tax_fee",
            BalanceTransactionType::Topup => "topup",
            BalanceTransactionType::TopupReversal => "topup_reversal",
            BalanceTransactionType::Transfer => "transfer",
            BalanceTransactionType::TransferCancel => "transfer_cancel",
            BalanceTransactionType::TransferFailure => "transfer_failure",
            BalanceTransactionType::TransferRefund => "transfer_refund",
            BalanceTransactionType::Noop => "",
            BalanceTransactionType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BalanceTransactionType {
    fn default() -> BalanceTransactionType {
        BalanceTransactionType::Noop
    }
}
impl BalanceTransactionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, BalanceTransactionType::Noop)
    }
}

/// Balance transactions represent funds moving through your Stripe account.
/// They're created for every type of transaction that comes into or flows out of your Stripe account balance.
///
/// Related guide: [Balance Transaction Types](https://stripe.com/docs/reports/balance-transaction-types).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BalanceTransaction {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub available_on: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Balance transactions represent funds moving through your Stripe account.
    *  They're created for every type of transaction that comes into or flows out of your Stripe account balance.
    *  
    *  Related guide: [Balance Transaction Types](https://stripe.com/docs/reports/balance-transaction-types).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub exchange_rate: f64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub fee: i64,
    /**
    * Detailed breakdown of fees (in %s) paid for this transaction.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub fee_details: Vec<Fee>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub net: i64,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BalanceTransactionObject::is_noop")]
    pub object: BalanceTransactionObject,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reporting_category: String,
    /**
    * Balance transactions represent funds moving through your Stripe account.
    *  They're created for every type of transaction that comes into or flows out of your Stripe account balance.
    *  
    *  Related guide: [Balance Transaction Types](https://stripe.com/docs/reports/balance-transaction-types).
    */
    #[serde()]
    pub source: Box<Option<BalanceTransactionSourceAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
    /**
    * Transaction type: `adjustment`, `advance`, `advance_funding`, `anticipation_repayment`, `application_fee`, `application_fee_refund`, `charge`, `connect_collection_transfer`, `contribution`, `issuing_authorization_hold`, `issuing_authorization_release`, `issuing_dispute`, `issuing_transaction`, `payment`, `payment_failure_refund`, `payment_refund`, `payout`, `payout_cancel`, `payout_failure`, `refund`, `refund_failure`, `reserve_transaction`, `reserved_funds`, `stripe_fee`, `stripe_fx_fee`, `tax_fee`, `topup`, `topup_reversal`, `transfer`, `transfer_cancel`, `transfer_failure`, or `transfer_refund`. [Learn more](https://stripe.com/docs/reports/balance-transaction-types) about balance transaction types and what they represent. If you are looking to classify transactions for accounting purposes, you might want to consider `reporting_category` instead.
    */
    #[serde(
        default,
        skip_serializing_if = "BalanceTransactionType::is_noop",
        rename = "type"
    )]
    pub type_: BalanceTransactionType,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AvailablePayoutMethods {
    #[serde(rename = "instant")]
    Instant,
    #[serde(rename = "standard")]
    Standard,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AvailablePayoutMethods {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AvailablePayoutMethods::Instant => "instant",
            AvailablePayoutMethods::Standard => "standard",
            AvailablePayoutMethods::Noop => "",
            AvailablePayoutMethods::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AvailablePayoutMethods {
    fn default() -> AvailablePayoutMethods {
        AvailablePayoutMethods::Noop
    }
}
impl AvailablePayoutMethods {
    pub fn is_noop(&self) -> bool {
        matches!(self, AvailablePayoutMethods::Noop)
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BankAccountObject {
    #[serde(rename = "bank_account")]
    BankAccount,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BankAccountObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BankAccountObject::BankAccount => "bank_account",
            BankAccountObject::Noop => "",
            BankAccountObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BankAccountObject {
    fn default() -> BankAccountObject {
        BankAccountObject::Noop
    }
}
impl BankAccountObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, BankAccountObject::Noop)
    }
}

/// These bank accounts are payment methods on `Customer` objects.
///
/// On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
/// destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
/// They can be bank accounts or debit cards as well, and are documented in the links above.
///
/// Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BankAccount {
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde()]
    pub account: Box<Option<AccountAnyOf>>,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_holder_name: String,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_holder_type: String,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_type: String,
    /**
    * A set of available payout methods for this bank account. Only values from this set should be passed as the `method` when creating a payout.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub available_payout_methods: Vec<AvailablePayoutMethods>,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bank_name: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub default_for_currency: bool,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BankAccountObject::is_noop")]
    pub object: BankAccountObject,
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub routing_number: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BillingDetails {
    /**
    * A publicly available mailing address for sending support issues to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address: Option<Address>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PortalConfigurationObject {
    #[serde(rename = "billing_portal.configuration")]
    BillingPortalConfiguration,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PortalConfigurationObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PortalConfigurationObject::BillingPortalConfiguration => "billing_portal.configuration",
            PortalConfigurationObject::Noop => "",
            PortalConfigurationObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PortalConfigurationObject {
    fn default() -> PortalConfigurationObject {
        PortalConfigurationObject::Noop
    }
}
impl PortalConfigurationObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, PortalConfigurationObject::Noop)
    }
}

/// A portal configuration describes the functionality and behavior of a portal session.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PortalConfiguration {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub active: bool,
    /**
    * A portal configuration describes the functionality and behavior of a portal session.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub application: String,
    /**
    *
    */
    #[serde()]
    pub business_profile: PortalBusinessProfile,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * A portal configuration describes the functionality and behavior of a portal session.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub default_return_url: String,
    /**
    *
    */
    #[serde()]
    pub features: PortalFeatures,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub is_default: bool,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * A portal configuration describes the functionality and behavior of a portal session.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "PortalConfigurationObject::is_noop")]
    pub object: PortalConfigurationObject,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub updated: i64,
}

/// All of the following types:
///
/// - `String`
/// - `PortalConfiguration`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ConfigurationAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A portal configuration describes the functionality and behavior of a portal session.
    */
    PortalConfiguration(PortalConfiguration),
}

impl ConfigurationAnyOf {
    pub fn portal_configuration(&self) -> Option<&PortalConfiguration> {
        if let ConfigurationAnyOf::PortalConfiguration(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let ConfigurationAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for ConfigurationAnyOf {
    fn from(f: String) -> Self {
        ConfigurationAnyOf::String(f)
    }
}

impl std::convert::From<ConfigurationAnyOf> for String {
    fn from(f: ConfigurationAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* The IETF language tag of the locale Customer Portal is displayed in. If blank or auto, the customer’s `preferred_locales` or browser’s locale is used.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Locale {
    #[serde(rename = "auto")]
    Auto,
    #[serde(rename = "bg")]
    Bg,
    #[serde(rename = "cs")]
    Cs,
    #[serde(rename = "da")]
    Da,
    #[serde(rename = "de")]
    De,
    #[serde(rename = "el")]
    El,
    #[serde(rename = "en")]
    En,
    #[serde(rename = "en-AU")]
    EnAu,
    #[serde(rename = "en-CA")]
    EnCa,
    #[serde(rename = "en-GB")]
    EnGb,
    #[serde(rename = "en-IE")]
    EnIe,
    #[serde(rename = "en-IN")]
    EnIn,
    #[serde(rename = "en-NZ")]
    EnNz,
    #[serde(rename = "en-SG")]
    EnSg,
    #[serde(rename = "es")]
    Es,
    #[serde(rename = "es-419")]
    Es419,
    #[serde(rename = "et")]
    Et,
    #[serde(rename = "fi")]
    Fi,
    #[serde(rename = "fil")]
    Fil,
    #[serde(rename = "fr")]
    Fr,
    #[serde(rename = "fr-CA")]
    FrCa,
    #[serde(rename = "hr")]
    Hr,
    #[serde(rename = "hu")]
    Hu,
    #[serde(rename = "id")]
    Id,
    #[serde(rename = "it")]
    It,
    #[serde(rename = "ja")]
    Ja,
    #[serde(rename = "ko")]
    Ko,
    #[serde(rename = "lt")]
    Lt,
    #[serde(rename = "lv")]
    Lv,
    #[serde(rename = "ms")]
    Ms,
    #[serde(rename = "mt")]
    Mt,
    #[serde(rename = "nb")]
    Nb,
    #[serde(rename = "nl")]
    Nl,
    #[serde(rename = "pl")]
    Pl,
    #[serde(rename = "pt")]
    Pt,
    #[serde(rename = "pt-BR")]
    PtBr,
    #[serde(rename = "ro")]
    Ro,
    #[serde(rename = "ru")]
    Ru,
    #[serde(rename = "sk")]
    Sk,
    #[serde(rename = "sl")]
    Sl,
    #[serde(rename = "sv")]
    Sv,
    #[serde(rename = "th")]
    Th,
    #[serde(rename = "tr")]
    Tr,
    #[serde(rename = "vi")]
    Vi,
    #[serde(rename = "zh")]
    Zh,
    #[serde(rename = "zh-HK")]
    ZhHk,
    #[serde(rename = "zh-TW")]
    ZhTw,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Locale {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Locale::Auto => "auto",
            Locale::Bg => "bg",
            Locale::Cs => "cs",
            Locale::Da => "da",
            Locale::De => "de",
            Locale::El => "el",
            Locale::En => "en",
            Locale::EnAu => "en-AU",
            Locale::EnCa => "en-CA",
            Locale::EnGb => "en-GB",
            Locale::EnIe => "en-IE",
            Locale::EnIn => "en-IN",
            Locale::EnNz => "en-NZ",
            Locale::EnSg => "en-SG",
            Locale::Es => "es",
            Locale::Es419 => "es-419",
            Locale::Et => "et",
            Locale::Fi => "fi",
            Locale::Fil => "fil",
            Locale::Fr => "fr",
            Locale::FrCa => "fr-CA",
            Locale::Hr => "hr",
            Locale::Hu => "hu",
            Locale::Id => "id",
            Locale::It => "it",
            Locale::Ja => "ja",
            Locale::Ko => "ko",
            Locale::Lt => "lt",
            Locale::Lv => "lv",
            Locale::Ms => "ms",
            Locale::Mt => "mt",
            Locale::Nb => "nb",
            Locale::Nl => "nl",
            Locale::Pl => "pl",
            Locale::Pt => "pt",
            Locale::PtBr => "pt-BR",
            Locale::Ro => "ro",
            Locale::Ru => "ru",
            Locale::Sk => "sk",
            Locale::Sl => "sl",
            Locale::Sv => "sv",
            Locale::Th => "th",
            Locale::Tr => "tr",
            Locale::Vi => "vi",
            Locale::Zh => "zh",
            Locale::ZhHk => "zh-HK",
            Locale::ZhTw => "zh-TW",
            Locale::Noop => "",
            Locale::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Locale {
    fn default() -> Locale {
        Locale::Noop
    }
}
impl Locale {
    pub fn is_noop(&self) -> bool {
        matches!(self, Locale::Noop)
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PortalSessionObject {
    #[serde(rename = "billing_portal.session")]
    BillingPortalSession,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PortalSessionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PortalSessionObject::BillingPortalSession => "billing_portal.session",
            PortalSessionObject::Noop => "",
            PortalSessionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PortalSessionObject {
    fn default() -> PortalSessionObject {
        PortalSessionObject::Noop
    }
}
impl PortalSessionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, PortalSessionObject::Noop)
    }
}

/// The Billing customer portal is a Stripe-hosted UI for subscription and
/// billing management.
///
/// A portal configuration describes the functionality and features that you
/// want to provide to your customers through the portal.
///
/// A portal session describes the instantiation of the customer portal for
/// a particular customer. By visiting the session's URL, the customer
/// can manage their subscriptions and billing details. For security reasons,
/// sessions are short-lived and will expire if the customer does not visit the URL.
/// Create sessions on-demand when customers intend to manage their subscriptions
/// and billing details.
///
/// Learn more in the [integration guide](https://stripe.com/docs/billing/subscriptions/integrating-customer-portal).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PortalSession {
    #[serde()]
    pub configuration: Box<ConfigurationAnyOf>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * The IETF language tag of the locale Customer Portal is displayed in. If blank or auto, the customer’s `preferred_locales` or browser’s locale is used.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub locale: Option<Locale>,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "PortalSessionObject::is_noop")]
    pub object: PortalSessionObject,
    /**
    * The Billing customer portal is a Stripe-hosted UI for subscription and
    *  billing management.
    *  
    *  A portal configuration describes the functionality and features that you
    *  want to provide to your customers through the portal.
    *  
    *  A portal session describes the instantiation of the customer portal for
    *  a particular customer. By visiting the session's URL, the customer
    *  can manage their subscriptions and billing details. For security reasons,
    *  sessions are short-lived and will expire if the customer does not visit the URL.
    *  Create sessions on-demand when customers intend to manage their subscriptions
    *  and billing details.
    *  
    *  Learn more in the [integration guide](https://stripe.com/docs/billing/subscriptions/integrating-customer-portal).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub on_behalf_of: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub return_url: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BitcoinReceiverObject {
    #[serde(rename = "bitcoin_receiver")]
    BitcoinReceiver,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BitcoinReceiverObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BitcoinReceiverObject::BitcoinReceiver => "bitcoin_receiver",
            BitcoinReceiverObject::Noop => "",
            BitcoinReceiverObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BitcoinReceiverObject {
    fn default() -> BitcoinReceiverObject {
        BitcoinReceiverObject::Noop
    }
}
impl BitcoinReceiverObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, BitcoinReceiverObject::Noop)
    }
}

/// A list with one entry for each time that the customer sent bitcoin to the receiver. Hidden when viewing the receiver with a publishable key.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Transactions {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<BitcoinTransaction>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BitcoinReceiver {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub active: bool,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_received: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub bitcoin_amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub bitcoin_amount_received: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bitcoin_uri: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub filled: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub inbound_address: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BitcoinReceiverObject::is_noop")]
    pub object: BitcoinReceiverObject,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub refund_address: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transactions: Option<Transactions>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub uncaptured_funds: bool,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub used_for_payment: bool,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BitcoinTransactionObject {
    #[serde(rename = "bitcoin_transaction")]
    BitcoinTransaction,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BitcoinTransactionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BitcoinTransactionObject::BitcoinTransaction => "bitcoin_transaction",
            BitcoinTransactionObject::Noop => "",
            BitcoinTransactionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BitcoinTransactionObject {
    fn default() -> BitcoinTransactionObject {
        BitcoinTransactionObject::Noop
    }
}
impl BitcoinTransactionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, BitcoinTransactionObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BitcoinTransaction {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub bitcoin_amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BitcoinTransactionObject::is_noop")]
    pub object: BitcoinTransactionObject,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub receiver: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CapabilityObject {
    #[serde(rename = "capability")]
    Capability,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CapabilityObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CapabilityObject::Capability => "capability",
            CapabilityObject::Noop => "",
            CapabilityObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CapabilityObject {
    fn default() -> CapabilityObject {
        CapabilityObject::Noop
    }
}
impl CapabilityObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CapabilityObject::Noop)
    }
}

/**
* The status of the capability. Can be `active`, `inactive`, `pending`, or `unrequested`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CapabilityStatus {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "disabled")]
    Disabled,
    #[serde(rename = "inactive")]
    Inactive,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "unrequested")]
    Unrequested,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CapabilityStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CapabilityStatus::Active => "active",
            CapabilityStatus::Disabled => "disabled",
            CapabilityStatus::Inactive => "inactive",
            CapabilityStatus::Pending => "pending",
            CapabilityStatus::Unrequested => "unrequested",
            CapabilityStatus::Noop => "",
            CapabilityStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CapabilityStatus {
    fn default() -> CapabilityStatus {
        CapabilityStatus::Noop
    }
}
impl CapabilityStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, CapabilityStatus::Noop)
    }
}

/// This is an object representing a capability for a Stripe account.
///
/// Related guide: [Account capabilities](https://stripe.com/docs/connect/account-capabilities).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Capability {
    #[serde()]
    pub account: Box<AccountAnyOf>,
    /**
    * This is an object representing a capability for a Stripe account.
    *  
    *  Related guide: [Account capabilities](https://stripe.com/docs/connect/account-capabilities).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub future_requirements: Option<AccountCapabilityFutureRequirements>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CapabilityObject::is_noop")]
    pub object: CapabilityObject,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub requested: bool,
    /**
    * This is an object representing a capability for a Stripe account.
    *  
    *  Related guide: [Account capabilities](https://stripe.com/docs/connect/account-capabilities).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub requested_at: i64,
    /**
    * This is an object representing a capability for a Stripe account.
    *  
    *  Related guide: [Account capabilities](https://stripe.com/docs/connect/account-capabilities).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub requirements: Option<AccountCapabilityRequirements>,
    /**
    * The status of the capability. Can be `active`, `inactive`, `pending`, or `unrequested`.
    */
    #[serde(default, skip_serializing_if = "CapabilityStatus::is_noop")]
    pub status: CapabilityStatus,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CardObject {
    #[serde(rename = "card")]
    Card,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CardObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CardObject::Card => "card",
            CardObject::Noop => "",
            CardObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CardObject {
    fn default() -> CardObject {
        CardObject::Noop
    }
}
impl CardObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CardObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `Recipient`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RecipientAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * With `Recipient` objects, you can transfer money from your Stripe account to a
    *  third-party bank account or debit card. The API allows you to create, delete,
    *  and update your recipients. You can retrieve individual recipients as well as
    *  a list of all your recipients.
    *  
    *  \*\*`Recipient` objects have been deprecated in favor of
    *  [Connect](https://stripe.com/docs/connect), specifically Connect's much more powerful
    *  [Account objects](https://stripe.com/docs/api#account). Stripe accounts that don't already use
    *  recipients can no longer begin doing so. Please use `Account` objects
    *  instead.\*\*
    */
    Recipient(Recipient),
}

impl RecipientAnyOf {
    pub fn recipient(&self) -> Option<&Recipient> {
        if let RecipientAnyOf::Recipient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let RecipientAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for RecipientAnyOf {
    fn from(f: String) -> Self {
        RecipientAnyOf::String(f)
    }
}

impl std::convert::From<RecipientAnyOf> for String {
    fn from(f: RecipientAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// You can store multiple cards on a customer in order to charge the customer
/// later. You can also store multiple debit cards on a recipient in order to
/// transfer to those cards later.
///
/// Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Card {
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde()]
    pub account: Box<Option<AccountAnyOf>>,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_city: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_country: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "address_line1"
    )]
    pub address_line_1: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "address_line1_check"
    )]
    pub address_line_1_check: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "address_line2"
    )]
    pub address_line_2: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_state: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_zip: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_zip_check: String,
    /**
    * A set of available payout methods for this bank account. Only values from this set should be passed as the `method` when creating a payout.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub available_payout_methods: Vec<AvailablePayoutMethods>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub brand: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cvc_check: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub default_for_currency: bool,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "dynamic_last4"
    )]
    pub dynamic_last_4: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub exp_month: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub exp_year: i64,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub funding: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CardObject::is_noop")]
    pub object: CardObject,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde()]
    pub recipient: Box<Option<RecipientAnyOf>>,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tokenization_method: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CardGeneratedFromPaymentMethodDetails {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card_present: Option<PaymentMethodDetailsCardPresent>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CardIssuingAccountTermsOfService {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub date: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_agent: String,
}

/// When confirming a PaymentIntent with Stripe.js, Stripe.js depends on the contents of this dictionary to invoke authentication flows. The shape of the contents is subject to change and is only intended to be used by Stripe.js.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct UseStripeSdk {}

/// All of the following types:
///
/// - `String`
/// - `PlatformFee`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum FeeAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    *
    */
    PlatformFee(PlatformFee),
}

impl FeeAnyOf {
    pub fn platform_fee(&self) -> Option<&PlatformFee> {
        if let FeeAnyOf::PlatformFee(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let FeeAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for FeeAnyOf {
    fn from(f: String) -> Self {
        FeeAnyOf::String(f)
    }
}

impl std::convert::From<FeeAnyOf> for String {
    fn from(f: FeeAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `Invoice`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum InvoiceAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    Invoice(Invoice),
}

impl InvoiceAnyOf {
    pub fn invoice(&self) -> Option<&Invoice> {
        if let InvoiceAnyOf::Invoice(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let InvoiceAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for InvoiceAnyOf {
    fn from(f: String) -> Self {
        InvoiceAnyOf::String(f)
    }
}

impl std::convert::From<InvoiceAnyOf> for String {
    fn from(f: InvoiceAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ChargeObject {
    #[serde(rename = "charge")]
    Charge,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ChargeObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ChargeObject::Charge => "charge",
            ChargeObject::Noop => "",
            ChargeObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ChargeObject {
    fn default() -> ChargeObject {
        ChargeObject::Noop
    }
}
impl ChargeObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, ChargeObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `Order`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum OrderAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    Order(Order),
}

impl OrderAnyOf {
    pub fn order(&self) -> Option<&Order> {
        if let OrderAnyOf::Order(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let OrderAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for OrderAnyOf {
    fn from(f: String) -> Self {
        OrderAnyOf::String(f)
    }
}

impl std::convert::From<OrderAnyOf> for String {
    fn from(f: OrderAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `PaymentIntent`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PaymentIntentAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    PaymentIntent(PaymentIntent),
}

impl PaymentIntentAnyOf {
    pub fn payment_intent(&self) -> Option<&PaymentIntent> {
        if let PaymentIntentAnyOf::PaymentIntent(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let PaymentIntentAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for PaymentIntentAnyOf {
    fn from(f: String) -> Self {
        PaymentIntentAnyOf::String(f)
    }
}

impl std::convert::From<PaymentIntentAnyOf> for String {
    fn from(f: PaymentIntentAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// A list of refunds that have been applied to the charge.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct RefundList {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<Refund>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/// All of the following types:
///
/// - `String`
/// - `Review`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ReviewAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Reviews can be used to supplement automated fraud detection with human expertise.
    *  
    *  Learn more about [Radar](/radar) and reviewing payments
    *  [here](https://stripe.com/docs/radar/reviews).
    */
    Review(Review),
}

impl ReviewAnyOf {
    pub fn review(&self) -> Option<&Review> {
        if let ReviewAnyOf::Review(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let ReviewAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for ReviewAnyOf {
    fn from(f: String) -> Self {
        ReviewAnyOf::String(f)
    }
}

impl std::convert::From<ReviewAnyOf> for String {
    fn from(f: ReviewAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `Transfer`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TransferAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A `Transfer` object is created when you move funds between Stripe accounts as
    *  part of Connect.
    *  
    *  Before April 6, 2017, transfers also represented movement of funds from a
    *  Stripe account to a card or bank account. This behavior has since been split
    *  out into a [Payout](https://stripe.com/docs/api#payout_object) object, with corresponding payout endpoints. For more
    *  information, read about the
    *  [transfer/payout split](https://stripe.com/docs/transfer-payout-split).
    *  
    *  Related guide: [Creating Separate Charges and Transfers](https://stripe.com/docs/connect/charges-transfers).
    */
    Transfer(Transfer),
}

impl TransferAnyOf {
    pub fn string(&self) -> Option<&String> {
        if let TransferAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn transfer(&self) -> Option<&Transfer> {
        if let TransferAnyOf::Transfer(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for TransferAnyOf {
    fn from(f: String) -> Self {
        TransferAnyOf::String(f)
    }
}

impl std::convert::From<TransferAnyOf> for String {
    fn from(f: TransferAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* The status of the payment is either `succeeded`, `pending`, or `failed`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ChargeStatus {
    #[serde(rename = "failed")]
    Failed,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "succeeded")]
    Succeeded,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ChargeStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ChargeStatus::Failed => "failed",
            ChargeStatus::Pending => "pending",
            ChargeStatus::Succeeded => "succeeded",
            ChargeStatus::Noop => "",
            ChargeStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ChargeStatus {
    fn default() -> ChargeStatus {
        ChargeStatus::Noop
    }
}
impl ChargeStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, ChargeStatus::Noop)
    }
}

/// To charge a credit or a debit card, you create a `Charge` object. You can
/// retrieve and refund individual charges as well as list all charges. Charges
/// are identified by a unique, random ID.
///
/// Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Charge {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_captured: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_refunded: i64,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub application: Box<Option<ApplicationAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub application_fee: Box<Option<FeeAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub application_fee_amount: i64,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub balance_transaction: Box<Option<BalanceTransactionAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub billing_details: BillingDetails,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub calculated_statement_descriptor: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub captured: bool,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub disputed: bool,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub failure_balance_transaction: Box<Option<BalanceTransactionAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub failure_code: String,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub failure_message: String,
    /**
    * Information on fraud assessments for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub fraud_details: Option<ChargeFraudDetails>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub invoice: Box<Option<InvoiceAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "ChargeObject::is_noop")]
    pub object: ChargeObject,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub on_behalf_of: Box<Option<AccountAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub order: Box<Option<OrderAnyOf>>,
    /**
    * Details about whether the payment was accepted, and why. See [understanding declines](https://stripe.com/docs/declines) for details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub outcome: Option<ChargeOutcome>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub paid: bool,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub payment_intent: Box<Option<PaymentIntentAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_method: String,
    /**
    * Details about the payment method at the time of the transaction.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_method_details: Option<PaymentMethodDetails>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub receipt_email: String,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub receipt_number: String,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub receipt_url: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub refunded: bool,
    /**
    * A list of refunds that have been applied to the charge.
    */
    #[serde()]
    pub refunds: RefundList,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub review: Box<Option<ReviewAnyOf>>,
    /**
    * Shipping information for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping: Option<ShippingData>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub source_transfer: Box<Option<TransferAnyOf>>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor: String,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor_suffix: String,
    /**
    * The status of the payment is either `succeeded`, `pending`, or `failed`.
    */
    #[serde(default, skip_serializing_if = "ChargeStatus::is_noop")]
    pub status: ChargeStatus,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde()]
    pub transfer: Box<Option<TransferAnyOf>>,
    /**
    * An optional dictionary including the account to automatically transfer to as part of a destination charge. [See the Connect documentation](https://stripe.com/docs/connect/destination-charges) for details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transfer_data: Option<ChargeTransferData>,
    /**
    * To charge a credit or a debit card, you create a `Charge` object. You can
    *  retrieve and refund individual charges as well as list all charges. Charges
    *  are identified by a unique, random ID.
    *  
    *  Related guide: [Accept a payment with the Charges API](https://stripe.com/docs/payments/accept-a-payment-charges).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub transfer_group: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ChargeFraudDetails {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub stripe_report: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_report: String,
}

/// All of the following types:
///
/// - `String`
/// - `Rule`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RuleAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    *
    */
    Rule(Rule),
}

impl RuleAnyOf {
    pub fn rule(&self) -> Option<&Rule> {
        if let RuleAnyOf::Rule(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let RuleAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for RuleAnyOf {
    fn from(f: String) -> Self {
        RuleAnyOf::String(f)
    }
}

impl std::convert::From<RuleAnyOf> for String {
    fn from(f: RuleAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ChargeOutcome {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub network_status: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub risk_level: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub risk_score: i64,
    /**
    *
    */
    #[serde()]
    pub rule: Box<Option<RuleAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub seller_message: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ChargeTransferData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    #[serde()]
    pub destination: Box<AccountAnyOf>,
}

/**
* Describes whether Checkout should collect the customer's billing address.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BillingAddressCollection {
    #[serde(rename = "auto")]
    Auto,
    #[serde(rename = "required")]
    Required,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BillingAddressCollection {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BillingAddressCollection::Auto => "auto",
            BillingAddressCollection::Required => "required",
            BillingAddressCollection::Noop => "",
            BillingAddressCollection::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BillingAddressCollection {
    fn default() -> BillingAddressCollection {
        BillingAddressCollection::Noop
    }
}
impl BillingAddressCollection {
    pub fn is_noop(&self) -> bool {
        matches!(self, BillingAddressCollection::Noop)
    }
}

/**
* Configure whether a Checkout Session creates a Customer when the Checkout Session completes.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CustomerCreation {
    #[serde(rename = "always")]
    Always,
    #[serde(rename = "if_required")]
    IfRequired,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CustomerCreation {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CustomerCreation::Always => "always",
            CustomerCreation::IfRequired => "if_required",
            CustomerCreation::Noop => "",
            CustomerCreation::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CustomerCreation {
    fn default() -> CustomerCreation {
        CustomerCreation::Noop
    }
}
impl CustomerCreation {
    pub fn is_noop(&self) -> bool {
        matches!(self, CustomerCreation::Noop)
    }
}

/// The line items purchased by the customer.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LineItems {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<Item>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* The IETF language tag of the locale Checkout is displayed in. If blank or `auto`, the browser's locale is used.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum SessionLocale {
    #[serde(rename = "auto")]
    Auto,
    #[serde(rename = "bg")]
    Bg,
    #[serde(rename = "cs")]
    Cs,
    #[serde(rename = "da")]
    Da,
    #[serde(rename = "de")]
    De,
    #[serde(rename = "el")]
    El,
    #[serde(rename = "en")]
    En,
    #[serde(rename = "en-GB")]
    EnGb,
    #[serde(rename = "es")]
    Es,
    #[serde(rename = "es-419")]
    Es419,
    #[serde(rename = "et")]
    Et,
    #[serde(rename = "fi")]
    Fi,
    #[serde(rename = "fil")]
    Fil,
    #[serde(rename = "fr")]
    Fr,
    #[serde(rename = "fr-CA")]
    FrCa,
    #[serde(rename = "hr")]
    Hr,
    #[serde(rename = "hu")]
    Hu,
    #[serde(rename = "id")]
    Id,
    #[serde(rename = "it")]
    It,
    #[serde(rename = "ja")]
    Ja,
    #[serde(rename = "ko")]
    Ko,
    #[serde(rename = "lt")]
    Lt,
    #[serde(rename = "lv")]
    Lv,
    #[serde(rename = "ms")]
    Ms,
    #[serde(rename = "mt")]
    Mt,
    #[serde(rename = "nb")]
    Nb,
    #[serde(rename = "nl")]
    Nl,
    #[serde(rename = "pl")]
    Pl,
    #[serde(rename = "pt")]
    Pt,
    #[serde(rename = "pt-BR")]
    PtBr,
    #[serde(rename = "ro")]
    Ro,
    #[serde(rename = "ru")]
    Ru,
    #[serde(rename = "sk")]
    Sk,
    #[serde(rename = "sl")]
    Sl,
    #[serde(rename = "sv")]
    Sv,
    #[serde(rename = "th")]
    Th,
    #[serde(rename = "tr")]
    Tr,
    #[serde(rename = "vi")]
    Vi,
    #[serde(rename = "zh")]
    Zh,
    #[serde(rename = "zh-HK")]
    ZhHk,
    #[serde(rename = "zh-TW")]
    ZhTw,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for SessionLocale {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SessionLocale::Auto => "auto",
            SessionLocale::Bg => "bg",
            SessionLocale::Cs => "cs",
            SessionLocale::Da => "da",
            SessionLocale::De => "de",
            SessionLocale::El => "el",
            SessionLocale::En => "en",
            SessionLocale::EnGb => "en-GB",
            SessionLocale::Es => "es",
            SessionLocale::Es419 => "es-419",
            SessionLocale::Et => "et",
            SessionLocale::Fi => "fi",
            SessionLocale::Fil => "fil",
            SessionLocale::Fr => "fr",
            SessionLocale::FrCa => "fr-CA",
            SessionLocale::Hr => "hr",
            SessionLocale::Hu => "hu",
            SessionLocale::Id => "id",
            SessionLocale::It => "it",
            SessionLocale::Ja => "ja",
            SessionLocale::Ko => "ko",
            SessionLocale::Lt => "lt",
            SessionLocale::Lv => "lv",
            SessionLocale::Ms => "ms",
            SessionLocale::Mt => "mt",
            SessionLocale::Nb => "nb",
            SessionLocale::Nl => "nl",
            SessionLocale::Pl => "pl",
            SessionLocale::Pt => "pt",
            SessionLocale::PtBr => "pt-BR",
            SessionLocale::Ro => "ro",
            SessionLocale::Ru => "ru",
            SessionLocale::Sk => "sk",
            SessionLocale::Sl => "sl",
            SessionLocale::Sv => "sv",
            SessionLocale::Th => "th",
            SessionLocale::Tr => "tr",
            SessionLocale::Vi => "vi",
            SessionLocale::Zh => "zh",
            SessionLocale::ZhHk => "zh-HK",
            SessionLocale::ZhTw => "zh-TW",
            SessionLocale::Noop => "",
            SessionLocale::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for SessionLocale {
    fn default() -> SessionLocale {
        SessionLocale::Noop
    }
}
impl SessionLocale {
    pub fn is_noop(&self) -> bool {
        matches!(self, SessionLocale::Noop)
    }
}

/**
* The mode of the Checkout Session.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Mode {
    #[serde(rename = "payment")]
    Payment,
    #[serde(rename = "setup")]
    Setup,
    #[serde(rename = "subscription")]
    Subscription,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Mode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Mode::Payment => "payment",
            Mode::Setup => "setup",
            Mode::Subscription => "subscription",
            Mode::Noop => "",
            Mode::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Mode {
    fn default() -> Mode {
        Mode::Noop
    }
}
impl Mode {
    pub fn is_noop(&self) -> bool {
        matches!(self, Mode::Noop)
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum SessionObject {
    #[serde(rename = "checkout.session")]
    CheckoutSession,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for SessionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SessionObject::CheckoutSession => "checkout.session",
            SessionObject::Noop => "",
            SessionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for SessionObject {
    fn default() -> SessionObject {
        SessionObject::Noop
    }
}
impl SessionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, SessionObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `PaymentLink`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PaymentLinkAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A payment link is a shareable URL that will take your customers to a hosted payment page. A payment link can be shared and used multiple times.
    *  
    *  When a customer opens a payment link it will open a new [checkout session](https://stripe.com/docs/api/checkout/sessions) to render the payment page. You can use [checkout session events](https://stripe.com/docs/api/events/types#event_types-checkout.session.completed) to track payments through payment links.
    *  
    *  Related guide: [Payment Links API](https://stripe.com/docs/payments/payment-links/api)
    */
    PaymentLink(PaymentLink),
}

impl PaymentLinkAnyOf {
    pub fn payment_link(&self) -> Option<&PaymentLink> {
        if let PaymentLinkAnyOf::PaymentLink(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let PaymentLinkAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for PaymentLinkAnyOf {
    fn from(f: String) -> Self {
        PaymentLinkAnyOf::String(f)
    }
}

impl std::convert::From<PaymentLinkAnyOf> for String {
    fn from(f: PaymentLinkAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* The payment status of the Checkout Session, one of `paid`, `unpaid`, or `no_payment_required`.
*   You can use this value to decide when to fulfill your customer's order.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentStatus {
    #[serde(rename = "no_payment_required")]
    NoPaymentRequired,
    #[serde(rename = "paid")]
    Paid,
    #[serde(rename = "unpaid")]
    Unpaid,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentStatus::NoPaymentRequired => "no_payment_required",
            PaymentStatus::Paid => "paid",
            PaymentStatus::Unpaid => "unpaid",
            PaymentStatus::Noop => "",
            PaymentStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentStatus {
    fn default() -> PaymentStatus {
        PaymentStatus::Noop
    }
}
impl PaymentStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentStatus::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `SetupIntent`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SetupIntentAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A SetupIntent guides you through the process of setting up and saving a customer's payment credentials for future payments.
    *  For example, you could use a SetupIntent to set up and save your customer's card without immediately collecting a payment.
    *  Later, you can use [PaymentIntents](https://stripe.com/docs/api#payment_intents) to drive the payment flow.
    *  
    *  Create a SetupIntent as soon as you're ready to collect your customer's payment credentials.
    *  Do not maintain long-lived, unconfirmed SetupIntents as they may no longer be valid.
    *  The SetupIntent then transitions through multiple [statuses](https://stripe.com/docs/payments/intents#intent-statuses) as it guides
    *  you through the setup process.
    *  
    *  Successful SetupIntents result in payment credentials that are optimized for future payments.
    *  For example, cardholders in [certain regions](/guides/strong-customer-authentication) may need to be run through
    *  [Strong Customer Authentication](https://stripe.com/docs/strong-customer-authentication) at the time of payment method collection
    *  in order to streamline later [off-session payments](https://stripe.com/docs/payments/setup-intents).
    *  If the SetupIntent is used with a [Customer](https://stripe.com/docs/api#setup_intent_object-customer), upon success,
    *  it will automatically attach the resulting payment method to that Customer.
    *  We recommend using SetupIntents or [setup_future_usage](https://stripe.com/docs/api#payment_intent_object-setup_future_usage) on
    *  PaymentIntents to save payment methods in order to prevent saving invalid or unoptimized payment methods.
    *  
    *  By using SetupIntents, you ensure that your customers experience the minimum set of required friction,
    *  even as regulations change over time.
    *  
    *  Related guide: [Setup Intents API](https://stripe.com/docs/payments/setup-intents).
    */
    SetupIntent(SetupIntent),
}

impl SetupIntentAnyOf {
    pub fn setup_intent(&self) -> Option<&SetupIntent> {
        if let SetupIntentAnyOf::SetupIntent(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let SetupIntentAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for SetupIntentAnyOf {
    fn from(f: String) -> Self {
        SetupIntentAnyOf::String(f)
    }
}

impl std::convert::From<SetupIntentAnyOf> for String {
    fn from(f: SetupIntentAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `ShippingRate`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ShippingRateAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Shipping rates describe the price of shipping presented to your customers and can be
    *  applied to [Checkout Sessions](https://stripe.com/docs/payments/checkout/shipping) to collect shipping costs.
    */
    ShippingRate(ShippingRate),
}

impl ShippingRateAnyOf {
    pub fn shipping_rate(&self) -> Option<&ShippingRate> {
        if let ShippingRateAnyOf::ShippingRate(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let ShippingRateAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for ShippingRateAnyOf {
    fn from(f: String) -> Self {
        ShippingRateAnyOf::String(f)
    }
}

impl std::convert::From<ShippingRateAnyOf> for String {
    fn from(f: ShippingRateAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* The status of the Checkout Session, one of `open`, `complete`, or `expired`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum SessionStatus {
    #[serde(rename = "complete")]
    Complete,
    #[serde(rename = "expired")]
    Expired,
    #[serde(rename = "open")]
    Open,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for SessionStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SessionStatus::Complete => "complete",
            SessionStatus::Expired => "expired",
            SessionStatus::Open => "open",
            SessionStatus::Noop => "",
            SessionStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for SessionStatus {
    fn default() -> SessionStatus {
        SessionStatus::Noop
    }
}
impl SessionStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, SessionStatus::Noop)
    }
}

/**
* Describes the type of transaction being performed by Checkout in order to customize
*   relevant text on the page, such as the submit button. `submit_type` can only be
*   specified on Checkout Sessions in `payment` mode, but not Checkout Sessions
*   in `subscription` or `setup` mode.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum SubmitType {
    #[serde(rename = "auto")]
    Auto,
    #[serde(rename = "book")]
    Book,
    #[serde(rename = "donate")]
    Donate,
    #[serde(rename = "pay")]
    Pay,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for SubmitType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SubmitType::Auto => "auto",
            SubmitType::Book => "book",
            SubmitType::Donate => "donate",
            SubmitType::Pay => "pay",
            SubmitType::Noop => "",
            SubmitType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for SubmitType {
    fn default() -> SubmitType {
        SubmitType::Noop
    }
}
impl SubmitType {
    pub fn is_noop(&self) -> bool {
        matches!(self, SubmitType::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `Subscription`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SubscriptionAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Subscriptions allow you to charge a customer on a recurring basis.
    *  
    *  Related guide: [Creating Subscriptions](https://stripe.com/docs/billing/subscriptions/creating).
    */
    Subscription(Subscription),
}

impl SubscriptionAnyOf {
    pub fn string(&self) -> Option<&String> {
        if let SubscriptionAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn subscription(&self) -> Option<&Subscription> {
        if let SubscriptionAnyOf::Subscription(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for SubscriptionAnyOf {
    fn from(f: String) -> Self {
        SubscriptionAnyOf::String(f)
    }
}

impl std::convert::From<SubscriptionAnyOf> for String {
    fn from(f: SubscriptionAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// A Checkout Session represents your customer's session as they pay for
/// one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
/// or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
/// new Session each time your customer attempts to pay.
///
/// Once payment is successful, the Checkout Session will contain a reference
/// to the [Customer](https://stripe.com/docs/api/customers), and either the successful
/// [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
/// [Subscription](https://stripe.com/docs/api/subscriptions).
///
/// You can create a Checkout Session on your server and pass its ID to the
/// client to begin Checkout.
///
/// Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Session {
    /**
    * When set, provides configuration for actions to take if this Checkout Session expires.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub after_expiration: Option<PaymentPagesCheckoutSessionAfterExpiration>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_promotion_codes: bool,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_subtotal: i64,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_total: i64,
    /**
    *
    */
    #[serde()]
    pub automatic_tax: AutomaticTax,
    /**
    * Describes whether Checkout should collect the customer's billing address.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub billing_address_collection: Option<BillingAddressCollection>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cancel_url: String,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub client_reference_id: String,
    /**
    * Results of `consent_collection` for this session.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub consent: Option<PaymentPagesCheckoutSessionConsent>,
    /**
    * When set, provides configuration for the Checkout Session to gather active consent from customers.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub consent_collection: Option<PaymentPagesCheckoutSessionConsentCollection>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * Configure whether a Checkout Session creates a Customer when the Checkout Session completes.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_creation: Option<CustomerCreation>,
    /**
    * The customer details including the customer's tax exempt status and the customer's tax IDs. Only present on Sessions in `payment` or `subscription` mode.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_details: Option<PaymentPagesCheckoutSessionCustomerDetails>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_email: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_at: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub line_items: Option<LineItems>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * The IETF language tag of the locale Checkout is displayed in. If blank or `auto`, the browser's locale is used.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub locale: Option<SessionLocale>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * The mode of the Checkout Session.
    */
    #[serde(default, skip_serializing_if = "Mode::is_noop")]
    pub mode: Mode,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "SessionObject::is_noop")]
    pub object: SessionObject,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde()]
    pub payment_intent: Box<Option<PaymentIntentAnyOf>>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde()]
    pub payment_link: Box<Option<PaymentLinkAnyOf>>,
    /**
    * Payment-method-specific configuration for the PaymentIntent or SetupIntent of this CheckoutSession.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_method_options: Option<CheckoutSessionPaymentMethodOptions>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub payment_method_types: Vec<String>,
    /**
    * The payment status of the Checkout Session, one of `paid`, `unpaid`, or `no_payment_required`.
    *  You can use this value to decide when to fulfill your customer's order.
    */
    #[serde(default, skip_serializing_if = "PaymentStatus::is_noop")]
    pub payment_status: PaymentStatus,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub phone_number_collection: Option<PortalInvoiceList>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub recovered_from: String,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde()]
    pub setup_intent: Box<Option<SetupIntentAnyOf>>,
    /**
    * Shipping information for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping: Option<ShippingData>,
    /**
    * When set, provides configuration for Checkout to collect a shipping address from a customer.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping_address_collection: Option<PaymentLinksResourceShippingAddressCollection>,
    /**
    * The shipping rate options applied to this Session.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub shipping_options: Vec<PaymentPagesCheckoutSessionShippingOption>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde()]
    pub shipping_rate: Box<Option<ShippingRateAnyOf>>,
    /**
    * The status of the Checkout Session, one of `open`, `complete`, or `expired`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<SessionStatus>,
    /**
    * Describes the type of transaction being performed by Checkout in order to customize
    *  relevant text on the page, such as the submit button. `submit_type` can only be
    *  specified on Checkout Sessions in `payment` mode, but not Checkout Sessions
    *  in `subscription` or `setup` mode.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub submit_type: Option<SubmitType>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde()]
    pub subscription: Box<Option<SubscriptionAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub success_url: String,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tax_id_collection: Option<PortalInvoiceList>,
    /**
    * Tax and discount details for the computed total amount.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub total_details: Option<PaymentPagesCheckoutSessionTotalDetails>,
    /**
    * A Checkout Session represents your customer's session as they pay for
    *  one-time purchases or subscriptions through [Checkout](https://stripe.com/docs/payments/checkout)
    *  or [Payment Links](https://stripe.com/docs/payments/payment-links). We recommend creating a
    *  new Session each time your customer attempts to pay.
    *  
    *  Once payment is successful, the Checkout Session will contain a reference
    *  to the [Customer](https://stripe.com/docs/api/customers), and either the successful
    *  [PaymentIntent](https://stripe.com/docs/api/payment_intents) or an active
    *  [Subscription](https://stripe.com/docs/api/subscriptions).
    *  
    *  You can create a Checkout Session on your server and pass its ID to the
    *  client to begin Checkout.
    *  
    *  Related guide: [Checkout Server Quickstart](https://stripe.com/docs/payments/checkout/api).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DefaultFor {
    #[serde(rename = "invoice")]
    Invoice,
    #[serde(rename = "subscription")]
    Subscription,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DefaultFor {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DefaultFor::Invoice => "invoice",
            DefaultFor::Subscription => "subscription",
            DefaultFor::Noop => "",
            DefaultFor::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DefaultFor {
    fn default() -> DefaultFor {
        DefaultFor::Noop
    }
}
impl DefaultFor {
    pub fn is_noop(&self) -> bool {
        matches!(self, DefaultFor::Noop)
    }
}

/**
* Payment schedule for the mandate.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentSchedule {
    #[serde(rename = "combined")]
    Combined,
    #[serde(rename = "interval")]
    Interval,
    #[serde(rename = "sporadic")]
    Sporadic,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentSchedule {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentSchedule::Combined => "combined",
            PaymentSchedule::Interval => "interval",
            PaymentSchedule::Sporadic => "sporadic",
            PaymentSchedule::Noop => "",
            PaymentSchedule::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentSchedule {
    fn default() -> PaymentSchedule {
        PaymentSchedule::Noop
    }
}
impl PaymentSchedule {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentSchedule::Noop)
    }
}

/**
* Transaction type of the mandate.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum TransactionType {
    #[serde(rename = "business")]
    Business,
    #[serde(rename = "personal")]
    Personal,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for TransactionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            TransactionType::Business => "business",
            TransactionType::Personal => "personal",
            TransactionType::Noop => "",
            TransactionType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for TransactionType {
    fn default() -> TransactionType {
        TransactionType::Noop
    }
}
impl TransactionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, TransactionType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CheckoutAcssDebitMandateOptions {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub custom_mandate_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub default_for: Vec<DefaultFor>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub interval_description: String,
    /**
    * Payment schedule for the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_schedule: Option<PaymentSchedule>,
    /**
    * Transaction type of the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transaction_type: Option<TransactionType>,
}

/**
* Currency supported by the bank account. Returned when the Session is in `setup` mode.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Currency {
    #[serde(rename = "cad")]
    Cad,
    #[serde(rename = "usd")]
    Usd,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Currency {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Currency::Cad => "cad",
            Currency::Usd => "usd",
            Currency::Noop => "",
            Currency::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Currency {
    fn default() -> Currency {
        Currency::Noop
    }
}
impl Currency {
    pub fn is_noop(&self) -> bool {
        matches!(self, Currency::Noop)
    }
}

/**
* Bank account verification method.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum VerificationMethod {
    #[serde(rename = "automatic")]
    Automatic,
    #[serde(rename = "instant")]
    Instant,
    #[serde(rename = "microdeposits")]
    Microdeposits,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for VerificationMethod {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            VerificationMethod::Automatic => "automatic",
            VerificationMethod::Instant => "instant",
            VerificationMethod::Microdeposits => "microdeposits",
            VerificationMethod::Noop => "",
            VerificationMethod::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for VerificationMethod {
    fn default() -> VerificationMethod {
        VerificationMethod::Noop
    }
}
impl VerificationMethod {
    pub fn is_noop(&self) -> bool {
        matches!(self, VerificationMethod::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CheckoutAcssDebitPaymentMethodOptions {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub currency: Option<Currency>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub mandate_options: Option<CheckoutAcssDebitMandateOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<VerificationMethod>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CheckoutOxxoPaymentMethodOptions {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_after_days: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CheckoutKonbiniPaymentMethodOptions {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_after_days: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CheckoutSessionPaymentMethodOptions {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub acss_debit: Option<CheckoutAcssDebitPaymentMethodOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub boleto: Option<CheckoutOxxoPaymentMethodOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub konbini: Option<CheckoutKonbiniPaymentMethodOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub oxxo: Option<CheckoutOxxoPaymentMethodOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub us_bank_account: Option<CheckoutUsBankAccountPaymentMethodOptions>,
}

/**
* Bank account verification method.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CheckoutUsBankAccountPaymentMethodOptionsVerification {
    #[serde(rename = "automatic")]
    Automatic,
    #[serde(rename = "instant")]
    Instant,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CheckoutUsBankAccountPaymentMethodOptionsVerification {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CheckoutUsBankAccountPaymentMethodOptionsVerification::Automatic => "automatic",
            CheckoutUsBankAccountPaymentMethodOptionsVerification::Instant => "instant",
            CheckoutUsBankAccountPaymentMethodOptionsVerification::Noop => "",
            CheckoutUsBankAccountPaymentMethodOptionsVerification::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CheckoutUsBankAccountPaymentMethodOptionsVerification {
    fn default() -> CheckoutUsBankAccountPaymentMethodOptionsVerification {
        CheckoutUsBankAccountPaymentMethodOptionsVerification::Noop
    }
}
impl CheckoutUsBankAccountPaymentMethodOptionsVerification {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            CheckoutUsBankAccountPaymentMethodOptionsVerification::Noop
        )
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CheckoutUsBankAccountPaymentMethodOptions {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<CheckoutUsBankAccountPaymentMethodOptionsVerification>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ConnectCollectionTransferObject {
    #[serde(rename = "connect_collection_transfer")]
    ConnectCollectionTransfer,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ConnectCollectionTransferObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ConnectCollectionTransferObject::ConnectCollectionTransfer => {
                "connect_collection_transfer"
            }
            ConnectCollectionTransferObject::Noop => "",
            ConnectCollectionTransferObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ConnectCollectionTransferObject {
    fn default() -> ConnectCollectionTransferObject {
        ConnectCollectionTransferObject::Noop
    }
}
impl ConnectCollectionTransferObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, ConnectCollectionTransferObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ConnectCollectionTransfer {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    #[serde()]
    pub destination: Box<AccountAnyOf>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(
        default,
        skip_serializing_if = "ConnectCollectionTransferObject::is_noop"
    )]
    pub object: ConnectCollectionTransferObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CountrySpecObject {
    #[serde(rename = "country_spec")]
    CountrySpec,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CountrySpecObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CountrySpecObject::CountrySpec => "country_spec",
            CountrySpecObject::Noop => "",
            CountrySpecObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CountrySpecObject {
    fn default() -> CountrySpecObject {
        CountrySpecObject::Noop
    }
}
impl CountrySpecObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CountrySpecObject::Noop)
    }
}

/// Stripe needs to collect certain pieces of information about each account
/// created. These requirements can differ depending on the account's country. The
/// Country Specs API makes these rules available to your integration.
///
/// You can also view the information from this API call as [an online
/// guide](/docs/connect/required-verification-information).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CountrySpec {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub default_currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CountrySpecObject::is_noop")]
    pub object: CountrySpecObject,
    /**
    * Currencies that can be accepted in the specific country (for transfers).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub supported_bank_account_currencies: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub supported_payment_currencies: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub supported_payment_methods: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub supported_transfer_countries: Vec<String>,
    /**
    *
    */
    #[serde()]
    pub verification_fields: CountrySpecVerificationFields,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CountrySpecVerificationFieldDetails {
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub additional: Vec<String>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub minimum: Vec<String>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CountrySpecVerificationFields {
    /**
    *
    */
    #[serde()]
    pub company: CountrySpecVerificationFieldDetails,
    /**
    *
    */
    #[serde()]
    pub individual: CountrySpecVerificationFieldDetails,
}

/**
* One of `forever`, `once`, and `repeating`. Describes how long a customer who applies this coupon will get the discount.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Duration {
    #[serde(rename = "forever")]
    Forever,
    #[serde(rename = "once")]
    Once,
    #[serde(rename = "repeating")]
    Repeating,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Duration {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Duration::Forever => "forever",
            Duration::Once => "once",
            Duration::Repeating => "repeating",
            Duration::Noop => "",
            Duration::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Duration {
    fn default() -> Duration {
        Duration::Noop
    }
}
impl Duration {
    pub fn is_noop(&self) -> bool {
        matches!(self, Duration::Noop)
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CouponObject {
    #[serde(rename = "coupon")]
    Coupon,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CouponObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CouponObject::Coupon => "coupon",
            CouponObject::Noop => "",
            CouponObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CouponObject {
    fn default() -> CouponObject {
        CouponObject::Noop
    }
}
impl CouponObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CouponObject::Noop)
    }
}

/// A coupon contains information about a percent-off or amount-off discount you
/// might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
/// [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Coupon {
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_off: i64,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub applies_to: Option<CouponAppliesTo>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * One of `forever`, `once`, and `repeating`. Describes how long a customer who applies this coupon will get the discount.
    */
    #[serde(default, skip_serializing_if = "Duration::is_noop")]
    pub duration: Duration,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub duration_in_months: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub max_redemptions: i64,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CouponObject::is_noop")]
    pub object: CouponObject,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub percent_off: f64,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub redeem_by: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub times_redeemed: i64,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub valid: bool,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CouponAppliesTo {
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub products: Vec<String>,
}

/// All of the following types:
///
/// - `String`
/// - `CustomerBalanceTransaction`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CustomerBalanceTransactionAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Each customer has a [`balance`](https://stripe.com/docs/api/customers/object#customer_object-balance) value,
    *  which denotes a debit or credit that's automatically applied to their next invoice upon finalization.
    *  You may modify the value directly by using the [update customer API](https://stripe.com/docs/api/customers/update),
    *  or by creating a Customer Balance Transaction, which increments or decrements the customer's `balance` by the specified `amount`.
    *  
    *  Related guide: [Customer Balance](https://stripe.com/docs/billing/customer/balance) to learn more.
    */
    CustomerBalanceTransaction(CustomerBalanceTransaction),
}

impl CustomerBalanceTransactionAnyOf {
    pub fn customer_balance_transaction(&self) -> Option<&CustomerBalanceTransaction> {
        if let CustomerBalanceTransactionAnyOf::CustomerBalanceTransaction(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let CustomerBalanceTransactionAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for CustomerBalanceTransactionAnyOf {
    fn from(f: String) -> Self {
        CustomerBalanceTransactionAnyOf::String(f)
    }
}

impl std::convert::From<CustomerBalanceTransactionAnyOf> for String {
    fn from(f: CustomerBalanceTransactionAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// Line items that make up the credit note
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Lines {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<CreditNoteLineItem>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CreditNoteObject {
    #[serde(rename = "credit_note")]
    CreditNote,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CreditNoteObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CreditNoteObject::CreditNote => "credit_note",
            CreditNoteObject::Noop => "",
            CreditNoteObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CreditNoteObject {
    fn default() -> CreditNoteObject {
        CreditNoteObject::Noop
    }
}
impl CreditNoteObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CreditNoteObject::Noop)
    }
}

/**
* Reason for issuing this credit note, one of `duplicate`, `fraudulent`, `order_change`, or `product_unsatisfactory`
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Reason {
    #[serde(rename = "duplicate")]
    Duplicate,
    #[serde(rename = "fraudulent")]
    Fraudulent,
    #[serde(rename = "order_change")]
    OrderChange,
    #[serde(rename = "product_unsatisfactory")]
    ProductUnsatisfactory,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Reason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Reason::Duplicate => "duplicate",
            Reason::Fraudulent => "fraudulent",
            Reason::OrderChange => "order_change",
            Reason::ProductUnsatisfactory => "product_unsatisfactory",
            Reason::Noop => "",
            Reason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Reason {
    fn default() -> Reason {
        Reason::Noop
    }
}
impl Reason {
    pub fn is_noop(&self) -> bool {
        matches!(self, Reason::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `Refund`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum RefundAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * `Refund` objects allow you to refund a charge that has previously been created
    *  but not yet refunded. Funds will be refunded to the credit or debit card that
    *  was originally charged.
    *  
    *  Related guide: [Refunds](https://stripe.com/docs/refunds).
    */
    Refund(Refund),
}

impl RefundAnyOf {
    pub fn refund(&self) -> Option<&Refund> {
        if let RefundAnyOf::Refund(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let RefundAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for RefundAnyOf {
    fn from(f: String) -> Self {
        RefundAnyOf::String(f)
    }
}

impl std::convert::From<RefundAnyOf> for String {
    fn from(f: RefundAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* Status of this credit note, one of `issued` or `void`. Learn more about [voiding credit notes](https://stripe.com/docs/billing/invoices/credit-notes#voiding).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CreditNoteStatus {
    #[serde(rename = "issued")]
    Issued,
    #[serde(rename = "void")]
    Void,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CreditNoteStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CreditNoteStatus::Issued => "issued",
            CreditNoteStatus::Void => "void",
            CreditNoteStatus::Noop => "",
            CreditNoteStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CreditNoteStatus {
    fn default() -> CreditNoteStatus {
        CreditNoteStatus::Noop
    }
}
impl CreditNoteStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, CreditNoteStatus::Noop)
    }
}

/**
* Type of this credit note, one of `pre_payment` or `post_payment`. A `pre_payment` credit note means it was issued when the invoice was open. A `post_payment` credit note means it was issued when the invoice was paid.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CreditNoteType {
    #[serde(rename = "post_payment")]
    PostPayment,
    #[serde(rename = "pre_payment")]
    PrePayment,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CreditNoteType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CreditNoteType::PostPayment => "post_payment",
            CreditNoteType::PrePayment => "pre_payment",
            CreditNoteType::Noop => "",
            CreditNoteType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CreditNoteType {
    fn default() -> CreditNoteType {
        CreditNoteType::Noop
    }
}
impl CreditNoteType {
    pub fn is_noop(&self) -> bool {
        matches!(self, CreditNoteType::Noop)
    }
}

/// Issue a credit note to adjust an invoice's amount after the invoice is finalized.
///
/// Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CreditNote {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    #[serde()]
    pub customer: Box<CustomerAnyOf>,
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    #[serde()]
    pub customer_balance_transaction: Box<Option<CustomerBalanceTransactionAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub discount_amount: i64,
    /**
    * The aggregate amounts calculated per discount for all line items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub discount_amounts: Vec<DiscountsResourceDiscountAmount>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    #[serde()]
    pub invoice: Box<InvoiceAnyOf>,
    /**
    * Line items that make up the credit note
    */
    #[serde()]
    pub lines: Lines,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub memo: String,
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CreditNoteObject::is_noop")]
    pub object: CreditNoteObject,
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub out_of_band_amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pdf: String,
    /**
    * Reason for issuing this credit note, one of `duplicate`, `fraudulent`, `order_change`, or `product_unsatisfactory`
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub reason: Option<Reason>,
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    #[serde()]
    pub refund: Box<Option<RefundAnyOf>>,
    /**
    * Status of this credit note, one of `issued` or `void`. Learn more about [voiding credit notes](https://stripe.com/docs/billing/invoices/credit-notes#voiding).
    */
    #[serde(default, skip_serializing_if = "CreditNoteStatus::is_noop")]
    pub status: CreditNoteStatus,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub subtotal: i64,
    /**
    * The aggregate amounts calculated per tax rate for all line items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub tax_amounts: Vec<InvoiceTaxAmount>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total: i64,
    /**
    * Type of this credit note, one of `pre_payment` or `post_payment`. A `pre_payment` credit note means it was issued when the invoice was open. A `post_payment` credit note means it was issued when the invoice was paid.
    */
    #[serde(
        default,
        skip_serializing_if = "CreditNoteType::is_noop",
        rename = "type"
    )]
    pub type_: CreditNoteType,
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub voided_at: i64,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CreditNoteLineItemObject {
    #[serde(rename = "credit_note_line_item")]
    CreditNoteLineItem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CreditNoteLineItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CreditNoteLineItemObject::CreditNoteLineItem => "credit_note_line_item",
            CreditNoteLineItemObject::Noop => "",
            CreditNoteLineItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CreditNoteLineItemObject {
    fn default() -> CreditNoteLineItemObject {
        CreditNoteLineItemObject::Noop
    }
}
impl CreditNoteLineItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CreditNoteLineItemObject::Noop)
    }
}

/**
* The type of the credit note line item, one of `invoice_line_item` or `custom_line_item`. When the type is `invoice_line_item` there is an additional `invoice_line_item` property on the resource the value of which is the id of the credited line item on the invoice.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CreditNoteLineItemType {
    #[serde(rename = "custom_line_item")]
    CustomLineItem,
    #[serde(rename = "invoice_line_item")]
    InvoiceLineItem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CreditNoteLineItemType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CreditNoteLineItemType::CustomLineItem => "custom_line_item",
            CreditNoteLineItemType::InvoiceLineItem => "invoice_line_item",
            CreditNoteLineItemType::Noop => "",
            CreditNoteLineItemType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CreditNoteLineItemType {
    fn default() -> CreditNoteLineItemType {
        CreditNoteLineItemType::Noop
    }
}
impl CreditNoteLineItemType {
    pub fn is_noop(&self) -> bool {
        matches!(self, CreditNoteLineItemType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CreditNoteLineItem {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub discount_amount: i64,
    /**
    * The aggregate amounts calculated per discount for all line items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub discount_amounts: Vec<DiscountsResourceDiscountAmount>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice_line_item: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CreditNoteLineItemObject::is_noop")]
    pub object: CreditNoteLineItemObject,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub quantity: i64,
    /**
    * The aggregate amounts calculated per tax rate for all line items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub tax_amounts: Vec<InvoiceTaxAmount>,
    /**
    * The tax rates which apply to the line item.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub tax_rates: Vec<TaxRate>,
    /**
    * The type of the credit note line item, one of `invoice_line_item` or `custom_line_item`. When the type is `invoice_line_item` there is an additional `invoice_line_item` property on the resource the value of which is the id of the credited line item on the invoice.
    */
    #[serde(
        default,
        skip_serializing_if = "CreditNoteLineItemType::is_noop",
        rename = "type"
    )]
    pub type_: CreditNoteLineItemType,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub unit_amount: i64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub unit_amount_decimal: f64,
}

/// All of the following types:
///
/// - `String`
/// - `TaxRate`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TaxRateAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Tax rates can be applied to [invoices](https://stripe.com/docs/billing/invoices/tax-rates), [subscriptions](https://stripe.com/docs/billing/subscriptions/taxes) and [Checkout Sessions](https://stripe.com/docs/payments/checkout/set-up-a-subscription#tax-rates) to collect tax.
    *  
    *  Related guide: [Tax Rates](https://stripe.com/docs/billing/taxes/tax-rates).
    */
    TaxRate(TaxRate),
}

impl TaxRateAnyOf {
    pub fn string(&self) -> Option<&String> {
        if let TaxRateAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn tax_rate(&self) -> Option<&TaxRate> {
        if let TaxRateAnyOf::TaxRate(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for TaxRateAnyOf {
    fn from(f: String) -> Self {
        TaxRateAnyOf::String(f)
    }
}

impl std::convert::From<TaxRateAnyOf> for String {
    fn from(f: TaxRateAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceTaxAmount {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub inclusive: bool,
    #[serde()]
    pub tax_rate: Box<TaxRateAnyOf>,
}

/// All of the following types:
///
/// - `String`
/// - `BankAccount`
/// - `Card`
/// - `AlipayAccount`
/// - `SourceData`
/// - `BitcoinReceiver`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DefaultSourceAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    BankAccount(BankAccount),
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    Card(Card),
    /**
    *
    */
    AlipayAccount(AlipayAccount),
    /**
    * `Source` objects allow you to accept a variety of payment methods. They
    *  represent a customer's payment instrument, and can be used with the Stripe API
    *  just like a `Card` object: once chargeable, they can be charged, or can be
    *  attached to customers.
    *  
    *  Related guides: [Sources API](https://stripe.com/docs/sources) and [Sources & Customers](https://stripe.com/docs/sources/customers).
    */
    SourceData(SourceData),
    /**
    *
    */
    BitcoinReceiver(BitcoinReceiver),
}

impl DefaultSourceAnyOf {
    pub fn alipay_account(&self) -> Option<&AlipayAccount> {
        if let DefaultSourceAnyOf::AlipayAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn bank_account(&self) -> Option<&BankAccount> {
        if let DefaultSourceAnyOf::BankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn bitcoin_receiver(&self) -> Option<&BitcoinReceiver> {
        if let DefaultSourceAnyOf::BitcoinReceiver(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn card(&self) -> Option<&Card> {
        if let DefaultSourceAnyOf::Card(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn source_data(&self) -> Option<&SourceData> {
        if let DefaultSourceAnyOf::SourceData(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let DefaultSourceAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for DefaultSourceAnyOf {
    fn from(f: String) -> Self {
        DefaultSourceAnyOf::String(f)
    }
}

impl std::convert::From<DefaultSourceAnyOf> for String {
    fn from(f: DefaultSourceAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CustomerObject {
    #[serde(rename = "customer")]
    Customer,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CustomerObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CustomerObject::Customer => "customer",
            CustomerObject::Noop => "",
            CustomerObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CustomerObject {
    fn default() -> CustomerObject {
        CustomerObject::Noop
    }
}
impl CustomerObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CustomerObject::Noop)
    }
}

/// All of the following types:
///
/// - `BankAccount`
/// - `Card`
/// - `AlipayAccount`
/// - `SourceData`
/// - `BitcoinReceiver`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CustomerSourcesDataAnyOf {
    /**
    * These bank accounts are payment methods on `Customer` objects.
    *  
    *  On the other hand [External Accounts](https://stripe.com/docs/api#external_accounts) are transfer
    *  destinations on `Account` objects for [Custom accounts](https://stripe.com/docs/connect/custom-accounts).
    *  They can be bank accounts or debit cards as well, and are documented in the links above.
    *  
    *  Related guide: [Bank Debits and Transfers](https://stripe.com/docs/payments/bank-debits-transfers).
    */
    BankAccount(BankAccount),
    /**
    * You can store multiple cards on a customer in order to charge the customer
    *  later. You can also store multiple debit cards on a recipient in order to
    *  transfer to those cards later.
    *  
    *  Related guide: [Card Payments with Sources](https://stripe.com/docs/sources/cards).
    */
    Card(Card),
    /**
    *
    */
    AlipayAccount(AlipayAccount),
    /**
    * `Source` objects allow you to accept a variety of payment methods. They
    *  represent a customer's payment instrument, and can be used with the Stripe API
    *  just like a `Card` object: once chargeable, they can be charged, or can be
    *  attached to customers.
    *  
    *  Related guides: [Sources API](https://stripe.com/docs/sources) and [Sources & Customers](https://stripe.com/docs/sources/customers).
    */
    SourceData(SourceData),
    /**
    *
    */
    BitcoinReceiver(BitcoinReceiver),
}

impl CustomerSourcesDataAnyOf {
    pub fn alipay_account(&self) -> Option<&AlipayAccount> {
        if let CustomerSourcesDataAnyOf::AlipayAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn bank_account(&self) -> Option<&BankAccount> {
        if let CustomerSourcesDataAnyOf::BankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn bitcoin_receiver(&self) -> Option<&BitcoinReceiver> {
        if let CustomerSourcesDataAnyOf::BitcoinReceiver(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn card(&self) -> Option<&Card> {
        if let CustomerSourcesDataAnyOf::Card(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn source_data(&self) -> Option<&SourceData> {
        if let CustomerSourcesDataAnyOf::SourceData(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// The customer's payment sources, if any.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Sources {
    /**
    * Details about each object.
    */
    #[serde()]
    pub data: Box<Vec<CustomerSourcesDataAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/// The customer's current subscriptions, if any.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Subscriptions {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<Subscription>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* Describes the customer's tax exemption status. One of `none`, `exempt`, or `reverse`. When set to `reverse`, invoice and receipt PDFs include the text **"Reverse charge"**.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum TaxExempt {
    #[serde(rename = "exempt")]
    Exempt,
    #[serde(rename = "none")]
    None,
    #[serde(rename = "reverse")]
    Reverse,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for TaxExempt {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            TaxExempt::Exempt => "exempt",
            TaxExempt::None => "none",
            TaxExempt::Reverse => "reverse",
            TaxExempt::Noop => "",
            TaxExempt::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for TaxExempt {
    fn default() -> TaxExempt {
        TaxExempt::Noop
    }
}
impl TaxExempt {
    pub fn is_noop(&self) -> bool {
        matches!(self, TaxExempt::Noop)
    }
}

/// The customer's tax IDs.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct TaxIds {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<TaxId>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/// All of the following types:
///
/// - `String`
/// - `TestClock`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TestClockAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A test clock enables deterministic control over objects in testmode. With a test clock, you can create
    *  objects at a frozen time in the past or future, and advance to a specific future time to observe webhooks and state changes. After the clock advances,
    *  you can either validate the current state of your scenario (and test your assumptions), change the current state of your scenario (and test more complex scenarios), or keep advancing forward in time.
    */
    TestClock(TestClock),
}

impl TestClockAnyOf {
    pub fn string(&self) -> Option<&String> {
        if let TestClockAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn test_clock(&self) -> Option<&TestClock> {
        if let TestClockAnyOf::TestClock(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for TestClockAnyOf {
    fn from(f: String) -> Self {
        TestClockAnyOf::String(f)
    }
}

impl std::convert::From<TestClockAnyOf> for String {
    fn from(f: TestClockAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
///
/// Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Customer {
    /**
    * A publicly available mailing address for sending support issues to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address: Option<Address>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub balance: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde()]
    pub default_source: Box<Option<DefaultSourceAnyOf>>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub delinquent: bool,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * Describes the current discount active on the customer, if there is one.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub discount: Option<DiscountData>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice_prefix: String,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub invoice_settings: Option<InvoiceSettingCustomer>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub next_invoice_sequence: i64,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CustomerObject::is_noop")]
    pub object: CustomerObject,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone: String,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub preferred_locales: Vec<String>,
    /**
    * Shipping information for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping: Option<ShippingData>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sources: Option<Sources>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub subscriptions: Option<Subscriptions>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tax: Option<CustomerTax>,
    /**
    * Describes the customer's tax exemption status. One of `none`, `exempt`, or `reverse`. When set to `reverse`, invoice and receipt PDFs include the text \*\*"Reverse charge"\*\*.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tax_exempt: Option<TaxExempt>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub tax_ids: Option<TaxIds>,
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    #[serde()]
    pub test_clock: Box<Option<TestClockAnyOf>>,
}

/**
* The type of customer acceptance information included with the Mandate. One of `online` or `offline`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CustomerAcceptanceType {
    #[serde(rename = "offline")]
    Offline,
    #[serde(rename = "online")]
    Online,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CustomerAcceptanceType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CustomerAcceptanceType::Offline => "offline",
            CustomerAcceptanceType::Online => "online",
            CustomerAcceptanceType::Noop => "",
            CustomerAcceptanceType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CustomerAcceptanceType {
    fn default() -> CustomerAcceptanceType {
        CustomerAcceptanceType::Noop
    }
}
impl CustomerAcceptanceType {
    pub fn is_noop(&self) -> bool {
        matches!(self, CustomerAcceptanceType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CustomerAcceptance {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub accepted_at: i64,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub offline: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub online: Option<OnlineAcceptance>,
    /**
    * The type of customer acceptance information included with the Mandate. One of `online` or `offline`.
    */
    #[serde(
        default,
        skip_serializing_if = "CustomerAcceptanceType::is_noop",
        rename = "type"
    )]
    pub type_: CustomerAcceptanceType,
}

/// All of the following types:
///
/// - `String`
/// - `CreditNote`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CreditNoteAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Issue a credit note to adjust an invoice's amount after the invoice is finalized.
    *  
    *  Related guide: [Credit Notes](https://stripe.com/docs/billing/invoices/credit-notes).
    */
    CreditNote(CreditNote),
}

impl CreditNoteAnyOf {
    pub fn credit_note(&self) -> Option<&CreditNote> {
        if let CreditNoteAnyOf::CreditNote(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let CreditNoteAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for CreditNoteAnyOf {
    fn from(f: String) -> Self {
        CreditNoteAnyOf::String(f)
    }
}

impl std::convert::From<CreditNoteAnyOf> for String {
    fn from(f: CreditNoteAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `Customer`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CustomerAnyOfData {
    /**
    * The account's country.
    */
    String(String),
    /**
    * This object represents a customer of your business. It lets you create recurring charges and track payments that belong to the same customer.
    *  
    *  Related guide: [Save a card during payment](https://stripe.com/docs/payments/save-during-payment).
    */
    Customer(Customer),
}

impl CustomerAnyOfData {
    pub fn customer(&self) -> Option<&Customer> {
        if let CustomerAnyOfData::Customer(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let CustomerAnyOfData::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for CustomerAnyOfData {
    fn from(f: String) -> Self {
        CustomerAnyOfData::String(f)
    }
}

impl std::convert::From<CustomerAnyOfData> for String {
    fn from(f: CustomerAnyOfData) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CustomerBalanceTransactionObject {
    #[serde(rename = "customer_balance_transaction")]
    CustomerBalanceTransaction,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CustomerBalanceTransactionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CustomerBalanceTransactionObject::CustomerBalanceTransaction => {
                "customer_balance_transaction"
            }
            CustomerBalanceTransactionObject::Noop => "",
            CustomerBalanceTransactionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CustomerBalanceTransactionObject {
    fn default() -> CustomerBalanceTransactionObject {
        CustomerBalanceTransactionObject::Noop
    }
}
impl CustomerBalanceTransactionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, CustomerBalanceTransactionObject::Noop)
    }
}

/**
* Transaction type: `adjustment`, `applied_to_invoice`, `credit_note`, `initial`, `invoice_too_large`, `invoice_too_small`, `unspent_receiver_credit`, or `unapplied_from_invoice`. See the [Customer Balance page](https://stripe.com/docs/billing/customer/balance#types) to learn more about transaction types.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CustomerBalanceTransactionType {
    #[serde(rename = "adjustment")]
    Adjustment,
    #[serde(rename = "applied_to_invoice")]
    AppliedToInvoice,
    #[serde(rename = "credit_note")]
    CreditNote,
    #[serde(rename = "initial")]
    Initial,
    #[serde(rename = "invoice_too_large")]
    InvoiceTooLarge,
    #[serde(rename = "invoice_too_small")]
    InvoiceTooSmall,
    #[serde(rename = "migration")]
    Migration,
    #[serde(rename = "unapplied_from_invoice")]
    UnappliedFromInvoice,
    #[serde(rename = "unspent_receiver_credit")]
    UnspentReceiverCredit,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CustomerBalanceTransactionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CustomerBalanceTransactionType::Adjustment => "adjustment",
            CustomerBalanceTransactionType::AppliedToInvoice => "applied_to_invoice",
            CustomerBalanceTransactionType::CreditNote => "credit_note",
            CustomerBalanceTransactionType::Initial => "initial",
            CustomerBalanceTransactionType::InvoiceTooLarge => "invoice_too_large",
            CustomerBalanceTransactionType::InvoiceTooSmall => "invoice_too_small",
            CustomerBalanceTransactionType::Migration => "migration",
            CustomerBalanceTransactionType::UnappliedFromInvoice => "unapplied_from_invoice",
            CustomerBalanceTransactionType::UnspentReceiverCredit => "unspent_receiver_credit",
            CustomerBalanceTransactionType::Noop => "",
            CustomerBalanceTransactionType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CustomerBalanceTransactionType {
    fn default() -> CustomerBalanceTransactionType {
        CustomerBalanceTransactionType::Noop
    }
}
impl CustomerBalanceTransactionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, CustomerBalanceTransactionType::Noop)
    }
}

/// Each customer has a [`balance`](https://stripe.com/docs/api/customers/object#customer_object-balance) value,
/// which denotes a debit or credit that's automatically applied to their next invoice upon finalization.
/// You may modify the value directly by using the [update customer API](https://stripe.com/docs/api/customers/update),
/// or by creating a Customer Balance Transaction, which increments or decrements the customer's `balance` by the specified `amount`.
///
/// Related guide: [Customer Balance](https://stripe.com/docs/billing/customer/balance) to learn more.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CustomerBalanceTransaction {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * Each customer has a [`balance`](https://stripe.com/docs/api/customers/object#customer_object-balance) value,
    *  which denotes a debit or credit that's automatically applied to their next invoice upon finalization.
    *  You may modify the value directly by using the [update customer API](https://stripe.com/docs/api/customers/update),
    *  or by creating a Customer Balance Transaction, which increments or decrements the customer's `balance` by the specified `amount`.
    *  
    *  Related guide: [Customer Balance](https://stripe.com/docs/billing/customer/balance) to learn more.
    */
    #[serde()]
    pub credit_note: Box<Option<CreditNoteAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    #[serde()]
    pub customer: CustomerAnyOfData,
    /**
    * Each customer has a [`balance`](https://stripe.com/docs/api/customers/object#customer_object-balance) value,
    *  which denotes a debit or credit that's automatically applied to their next invoice upon finalization.
    *  You may modify the value directly by using the [update customer API](https://stripe.com/docs/api/customers/update),
    *  or by creating a Customer Balance Transaction, which increments or decrements the customer's `balance` by the specified `amount`.
    *  
    *  Related guide: [Customer Balance](https://stripe.com/docs/billing/customer/balance) to learn more.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub ending_balance: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Each customer has a [`balance`](https://stripe.com/docs/api/customers/object#customer_object-balance) value,
    *  which denotes a debit or credit that's automatically applied to their next invoice upon finalization.
    *  You may modify the value directly by using the [update customer API](https://stripe.com/docs/api/customers/update),
    *  or by creating a Customer Balance Transaction, which increments or decrements the customer's `balance` by the specified `amount`.
    *  
    *  Related guide: [Customer Balance](https://stripe.com/docs/billing/customer/balance) to learn more.
    */
    #[serde()]
    pub invoice: Box<Option<InvoiceAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Each customer has a [`balance`](https://stripe.com/docs/api/customers/object#customer_object-balance) value,
    *  which denotes a debit or credit that's automatically applied to their next invoice upon finalization.
    *  You may modify the value directly by using the [update customer API](https://stripe.com/docs/api/customers/update),
    *  or by creating a Customer Balance Transaction, which increments or decrements the customer's `balance` by the specified `amount`.
    *  
    *  Related guide: [Customer Balance](https://stripe.com/docs/billing/customer/balance) to learn more.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(
        default,
        skip_serializing_if = "CustomerBalanceTransactionObject::is_noop"
    )]
    pub object: CustomerBalanceTransactionObject,
    /**
    * Transaction type: `adjustment`, `applied_to_invoice`, `credit_note`, `initial`, `invoice_too_large`, `invoice_too_small`, `unspent_receiver_credit`, or `unapplied_from_invoice`. See the [Customer Balance page](https://stripe.com/docs/billing/customer/balance#types) to learn more about transaction types.
    */
    #[serde(
        default,
        skip_serializing_if = "CustomerBalanceTransactionType::is_noop",
        rename = "type"
    )]
    pub type_: CustomerBalanceTransactionType,
}

/**
* Surfaces if automatic tax computation is possible given the current customer location information.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CustomerTaxAutomatic {
    #[serde(rename = "failed")]
    Failed,
    #[serde(rename = "not_collecting")]
    NotCollecting,
    #[serde(rename = "supported")]
    Supported,
    #[serde(rename = "unrecognized_location")]
    UnrecognizedLocation,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CustomerTaxAutomatic {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CustomerTaxAutomatic::Failed => "failed",
            CustomerTaxAutomatic::NotCollecting => "not_collecting",
            CustomerTaxAutomatic::Supported => "supported",
            CustomerTaxAutomatic::UnrecognizedLocation => "unrecognized_location",
            CustomerTaxAutomatic::Noop => "",
            CustomerTaxAutomatic::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CustomerTaxAutomatic {
    fn default() -> CustomerTaxAutomatic {
        CustomerTaxAutomatic::Noop
    }
}
impl CustomerTaxAutomatic {
    pub fn is_noop(&self) -> bool {
        matches!(self, CustomerTaxAutomatic::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CustomerTax {
    /**
    * Surfaces if automatic tax computation is possible given the current customer location information.
    */
    #[serde(default, skip_serializing_if = "CustomerTaxAutomatic::is_noop")]
    pub automatic_tax: CustomerTaxAutomatic,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_address: String,
    /**
    * The customer's location as identified by Stripe Tax.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub location: Option<CustomerTaxLocation>,
}

/**
* The data source used to infer the customer's location.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Source {
    #[serde(rename = "billing_address")]
    BillingAddress,
    #[serde(rename = "ip_address")]
    IpAddress,
    #[serde(rename = "payment_method")]
    PaymentMethod,
    #[serde(rename = "shipping_destination")]
    ShippingDestination,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Source {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Source::BillingAddress => "billing_address",
            Source::IpAddress => "ip_address",
            Source::PaymentMethod => "payment_method",
            Source::ShippingDestination => "shipping_destination",
            Source::Noop => "",
            Source::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Source {
    fn default() -> Source {
        Source::Noop
    }
}
impl Source {
    pub fn is_noop(&self) -> bool {
        matches!(self, Source::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct CustomerTaxLocation {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    * The data source used to infer the customer's location.
    */
    #[serde(default, skip_serializing_if = "Source::is_noop")]
    pub source: Source,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedAccount {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "AccountObject::is_noop")]
    pub object: AccountObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct AlipayDeletedAccount {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "AlipayAccountObject::is_noop")]
    pub object: AlipayAccountObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedApplePayDomain {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "ApplePayDomainObject::is_noop")]
    pub object: ApplePayDomainObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedBankAccount {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BankAccountObject::is_noop")]
    pub object: BankAccountObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct BitcoinDeletedReceiver {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "BitcoinReceiverObject::is_noop")]
    pub object: BitcoinReceiverObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedCard {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CardObject::is_noop")]
    pub object: CardObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedCoupon {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CouponObject::is_noop")]
    pub object: CouponObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedCustomer {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CustomerObject::is_noop")]
    pub object: CustomerObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedDiscountObject {
    #[serde(rename = "discount")]
    Discount,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedDiscountObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedDiscountObject::Discount => "discount",
            DeletedDiscountObject::Noop => "",
            DeletedDiscountObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedDiscountObject {
    fn default() -> DeletedDiscountObject {
        DeletedDiscountObject::Noop
    }
}
impl DeletedDiscountObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedDiscountObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `PromotionCode`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PromotionCodeAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A Promotion Code represents a customer-redeemable code for a coupon. It can be used to
    *  create multiple codes for a single coupon.
    */
    PromotionCode(PromotionCode),
}

impl PromotionCodeAnyOf {
    pub fn promotion_code(&self) -> Option<&PromotionCode> {
        if let PromotionCodeAnyOf::PromotionCode(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let PromotionCodeAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for PromotionCodeAnyOf {
    fn from(f: String) -> Self {
        PromotionCodeAnyOf::String(f)
    }
}

impl std::convert::From<PromotionCodeAnyOf> for String {
    fn from(f: PromotionCodeAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedDiscount {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub checkout_session: String,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde()]
    pub coupon: Coupon,
    /**
    *
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice_item: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedDiscountObject::is_noop")]
    pub object: DeletedDiscountObject,
    /**
    *
    */
    #[serde()]
    pub promotion_code: Box<Option<PromotionCodeAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub start: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub subscription: String,
}

/// All of the following types:
///
/// - `DeletedBankAccount`
/// - `DeletedCard`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DeletedExternalAccountAnyOf {
    /**
    *
    */
    DeletedBankAccount(DeletedBankAccount),
    /**
    *
    */
    DeletedCard(DeletedCard),
}

impl DeletedExternalAccountAnyOf {
    pub fn deleted_bank_account(&self) -> Option<&DeletedBankAccount> {
        if let DeletedExternalAccountAnyOf::DeletedBankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn deleted_card(&self) -> Option<&DeletedCard> {
        if let DeletedExternalAccountAnyOf::DeletedCard(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedInvoiceObject {
    #[serde(rename = "invoice")]
    Invoice,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedInvoiceObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedInvoiceObject::Invoice => "invoice",
            DeletedInvoiceObject::Noop => "",
            DeletedInvoiceObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedInvoiceObject {
    fn default() -> DeletedInvoiceObject {
        DeletedInvoiceObject::Noop
    }
}
impl DeletedInvoiceObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedInvoiceObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedInvoice {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedInvoiceObject::is_noop")]
    pub object: DeletedInvoiceObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedInvoiceItemObject {
    #[serde(rename = "invoiceitem")]
    Invoiceitem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedInvoiceItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedInvoiceItemObject::Invoiceitem => "invoiceitem",
            DeletedInvoiceItemObject::Noop => "",
            DeletedInvoiceItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedInvoiceItemObject {
    fn default() -> DeletedInvoiceItemObject {
        DeletedInvoiceItemObject::Noop
    }
}
impl DeletedInvoiceItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedInvoiceItemObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedInvoiceItem {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedInvoiceItemObject::is_noop")]
    pub object: DeletedInvoiceItemObject,
}

/// All of the following types:
///
/// - `AlipayDeletedAccount`
/// - `DeletedBankAccount`
/// - `BitcoinDeletedReceiver`
/// - `DeletedCard`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DeletedPaymentSourceAnyOf {
    /**
    *
    */
    AlipayDeletedAccount(AlipayDeletedAccount),
    /**
    *
    */
    DeletedBankAccount(DeletedBankAccount),
    /**
    *
    */
    BitcoinDeletedReceiver(BitcoinDeletedReceiver),
    /**
    *
    */
    DeletedCard(DeletedCard),
}

impl DeletedPaymentSourceAnyOf {
    pub fn alipay_deleted_account(&self) -> Option<&AlipayDeletedAccount> {
        if let DeletedPaymentSourceAnyOf::AlipayDeletedAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn bitcoin_deleted_receiver(&self) -> Option<&BitcoinDeletedReceiver> {
        if let DeletedPaymentSourceAnyOf::BitcoinDeletedReceiver(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn deleted_bank_account(&self) -> Option<&DeletedBankAccount> {
        if let DeletedPaymentSourceAnyOf::DeletedBankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn deleted_card(&self) -> Option<&DeletedCard> {
        if let DeletedPaymentSourceAnyOf::DeletedCard(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedPersonObject {
    #[serde(rename = "person")]
    Person,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedPersonObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedPersonObject::Person => "person",
            DeletedPersonObject::Noop => "",
            DeletedPersonObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedPersonObject {
    fn default() -> DeletedPersonObject {
        DeletedPersonObject::Noop
    }
}
impl DeletedPersonObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedPersonObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedPerson {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedPersonObject::is_noop")]
    pub object: DeletedPersonObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedPlanObject {
    #[serde(rename = "plan")]
    Plan,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedPlanObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedPlanObject::Plan => "plan",
            DeletedPlanObject::Noop => "",
            DeletedPlanObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedPlanObject {
    fn default() -> DeletedPlanObject {
        DeletedPlanObject::Noop
    }
}
impl DeletedPlanObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedPlanObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedPlan {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedPlanObject::is_noop")]
    pub object: DeletedPlanObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedPriceObject {
    #[serde(rename = "price")]
    Price,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedPriceObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedPriceObject::Price => "price",
            DeletedPriceObject::Noop => "",
            DeletedPriceObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedPriceObject {
    fn default() -> DeletedPriceObject {
        DeletedPriceObject::Noop
    }
}
impl DeletedPriceObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedPriceObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedPrice {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedPriceObject::is_noop")]
    pub object: DeletedPriceObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedProductObject {
    #[serde(rename = "product")]
    Product,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedProductObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedProductObject::Product => "product",
            DeletedProductObject::Noop => "",
            DeletedProductObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedProductObject {
    fn default() -> DeletedProductObject {
        DeletedProductObject::Noop
    }
}
impl DeletedProductObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedProductObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedProduct {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedProductObject::is_noop")]
    pub object: DeletedProductObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum RadarListDeletedObject {
    #[serde(rename = "radar.value_list")]
    RadarValueList,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for RadarListDeletedObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RadarListDeletedObject::RadarValueList => "radar.value_list",
            RadarListDeletedObject::Noop => "",
            RadarListDeletedObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for RadarListDeletedObject {
    fn default() -> RadarListDeletedObject {
        RadarListDeletedObject::Noop
    }
}
impl RadarListDeletedObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, RadarListDeletedObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct RadarListDeleted {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "RadarListDeletedObject::is_noop")]
    pub object: RadarListDeletedObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum RadarListDeletedItemObject {
    #[serde(rename = "radar.value_list_item")]
    RadarValueListItem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for RadarListDeletedItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RadarListDeletedItemObject::RadarValueListItem => "radar.value_list_item",
            RadarListDeletedItemObject::Noop => "",
            RadarListDeletedItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for RadarListDeletedItemObject {
    fn default() -> RadarListDeletedItemObject {
        RadarListDeletedItemObject::Noop
    }
}
impl RadarListDeletedItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, RadarListDeletedItemObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct RadarListDeletedItem {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "RadarListDeletedItemObject::is_noop")]
    pub object: RadarListDeletedItemObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedRecipientObject {
    #[serde(rename = "recipient")]
    Recipient,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedRecipientObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedRecipientObject::Recipient => "recipient",
            DeletedRecipientObject::Noop => "",
            DeletedRecipientObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedRecipientObject {
    fn default() -> DeletedRecipientObject {
        DeletedRecipientObject::Noop
    }
}
impl DeletedRecipientObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedRecipientObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedRecipient {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedRecipientObject::is_noop")]
    pub object: DeletedRecipientObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedSkuObject {
    #[serde(rename = "sku")]
    Sku,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedSkuObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedSkuObject::Sku => "sku",
            DeletedSkuObject::Noop => "",
            DeletedSkuObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedSkuObject {
    fn default() -> DeletedSkuObject {
        DeletedSkuObject::Noop
    }
}
impl DeletedSkuObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedSkuObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedSku {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedSkuObject::is_noop")]
    pub object: DeletedSkuObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedSubscriptionItemObject {
    #[serde(rename = "subscription_item")]
    SubscriptionItem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedSubscriptionItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedSubscriptionItemObject::SubscriptionItem => "subscription_item",
            DeletedSubscriptionItemObject::Noop => "",
            DeletedSubscriptionItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedSubscriptionItemObject {
    fn default() -> DeletedSubscriptionItemObject {
        DeletedSubscriptionItemObject::Noop
    }
}
impl DeletedSubscriptionItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedSubscriptionItemObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedSubscriptionItem {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(
        default,
        skip_serializing_if = "DeletedSubscriptionItemObject::is_noop"
    )]
    pub object: DeletedSubscriptionItemObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedTaxObject {
    #[serde(rename = "tax_id")]
    TaxId,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedTaxObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedTaxObject::TaxId => "tax_id",
            DeletedTaxObject::Noop => "",
            DeletedTaxObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedTaxObject {
    fn default() -> DeletedTaxObject {
        DeletedTaxObject::Noop
    }
}
impl DeletedTaxObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedTaxObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedTaxId {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedTaxObject::is_noop")]
    pub object: DeletedTaxObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedTerminalLocationObject {
    #[serde(rename = "terminal.location")]
    TerminalLocation,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedTerminalLocationObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedTerminalLocationObject::TerminalLocation => "terminal.location",
            DeletedTerminalLocationObject::Noop => "",
            DeletedTerminalLocationObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedTerminalLocationObject {
    fn default() -> DeletedTerminalLocationObject {
        DeletedTerminalLocationObject::Noop
    }
}
impl DeletedTerminalLocationObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedTerminalLocationObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedTerminalLocation {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(
        default,
        skip_serializing_if = "DeletedTerminalLocationObject::is_noop"
    )]
    pub object: DeletedTerminalLocationObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedTerminalReaderObject {
    #[serde(rename = "terminal.reader")]
    TerminalReader,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedTerminalReaderObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedTerminalReaderObject::TerminalReader => "terminal.reader",
            DeletedTerminalReaderObject::Noop => "",
            DeletedTerminalReaderObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedTerminalReaderObject {
    fn default() -> DeletedTerminalReaderObject {
        DeletedTerminalReaderObject::Noop
    }
}
impl DeletedTerminalReaderObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedTerminalReaderObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedTerminalReader {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedTerminalReaderObject::is_noop")]
    pub object: DeletedTerminalReaderObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedTestClockObject {
    #[serde(rename = "test_helpers.test_clock")]
    TestHelpersClock,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedTestClockObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedTestClockObject::TestHelpersClock => "test_helpers.test_clock",
            DeletedTestClockObject::Noop => "",
            DeletedTestClockObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedTestClockObject {
    fn default() -> DeletedTestClockObject {
        DeletedTestClockObject::Noop
    }
}
impl DeletedTestClockObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedTestClockObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedTestClock {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedTestClockObject::is_noop")]
    pub object: DeletedTestClockObject,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DeletedWebhookEndpointObject {
    #[serde(rename = "webhook_endpoint")]
    WebhookEndpoint,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DeletedWebhookEndpointObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DeletedWebhookEndpointObject::WebhookEndpoint => "webhook_endpoint",
            DeletedWebhookEndpointObject::Noop => "",
            DeletedWebhookEndpointObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DeletedWebhookEndpointObject {
    fn default() -> DeletedWebhookEndpointObject {
        DeletedWebhookEndpointObject::Noop
    }
}
impl DeletedWebhookEndpointObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DeletedWebhookEndpointObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeletedWebhookEndpoint {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub deleted: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedWebhookEndpointObject::is_noop")]
    pub object: DeletedWebhookEndpointObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DeliveryEstimate {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub date: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub earliest: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub latest: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/// A discount represents the actual application of a coupon to a particular
/// customer. It contains information about when the discount began and when it
/// will end.
///
/// Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DiscountData {
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub checkout_session: String,
    /**
    * A coupon contains information about a percent-off or amount-off discount you
    *  might want to apply to a customer. Coupons may be applied to [invoices](https://stripe.com/docs/api#invoices) or
    *  [orders](https://stripe.com/docs/api#create_order_legacy-coupon). Coupons do not work with conventional one-off [charges](https://stripe.com/docs/api#create_charge).
    */
    #[serde()]
    pub coupon: Coupon,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub end: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice: String,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice_item: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedDiscountObject::is_noop")]
    pub object: DeletedDiscountObject,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde()]
    pub promotion_code: Box<Option<PromotionCodeAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub start: i64,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub subscription: String,
}

/// All of the following types:
///
/// - `String`
/// - `DiscountData`
/// - `DeletedDiscount`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DiscountAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    DiscountData(DiscountData),
    /**
    *
    */
    DeletedDiscount(DeletedDiscount),
}

impl DiscountAnyOf {
    pub fn deleted_discount(&self) -> Option<&DeletedDiscount> {
        if let DiscountAnyOf::DeletedDiscount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn discount_data(&self) -> Option<&DiscountData> {
        if let DiscountAnyOf::DiscountData(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let DiscountAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for DiscountAnyOf {
    fn from(f: String) -> Self {
        DiscountAnyOf::String(f)
    }
}

impl std::convert::From<DiscountAnyOf> for String {
    fn from(f: DiscountAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DiscountsResourceDiscountAmount {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    #[serde()]
    pub discount: Box<DiscountAnyOf>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DisputeObject {
    #[serde(rename = "dispute")]
    Dispute,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DisputeObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DisputeObject::Dispute => "dispute",
            DisputeObject::Noop => "",
            DisputeObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DisputeObject {
    fn default() -> DisputeObject {
        DisputeObject::Noop
    }
}
impl DisputeObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, DisputeObject::Noop)
    }
}

/**
* Current status of dispute. Possible values are `warning_needs_response`, `warning_under_review`, `warning_closed`, `needs_response`, `under_review`, `charge_refunded`, `won`, or `lost`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DisputeStatus {
    #[serde(rename = "charge_refunded")]
    ChargeRefunded,
    #[serde(rename = "lost")]
    Lost,
    #[serde(rename = "needs_response")]
    NeedsResponse,
    #[serde(rename = "under_review")]
    UnderReview,
    #[serde(rename = "warning_closed")]
    WarningClosed,
    #[serde(rename = "warning_needs_response")]
    WarningNeedsResponse,
    #[serde(rename = "warning_under_review")]
    WarningUnderReview,
    #[serde(rename = "won")]
    Won,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DisputeStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DisputeStatus::ChargeRefunded => "charge_refunded",
            DisputeStatus::Lost => "lost",
            DisputeStatus::NeedsResponse => "needs_response",
            DisputeStatus::UnderReview => "under_review",
            DisputeStatus::WarningClosed => "warning_closed",
            DisputeStatus::WarningNeedsResponse => "warning_needs_response",
            DisputeStatus::WarningUnderReview => "warning_under_review",
            DisputeStatus::Won => "won",
            DisputeStatus::Noop => "",
            DisputeStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DisputeStatus {
    fn default() -> DisputeStatus {
        DisputeStatus::Noop
    }
}
impl DisputeStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, DisputeStatus::Noop)
    }
}

/// A dispute occurs when a customer questions your charge with their card issuer.
/// When this happens, you're given the opportunity to respond to the dispute with
/// evidence that shows that the charge is legitimate. You can find more
/// information about the dispute process in our [Disputes and
/// Fraud](/docs/disputes) documentation.
///
/// Related guide: [Disputes and Fraud](https://stripe.com/docs/disputes).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Dispute {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * List of zero, one, or two balance transactions that show funds withdrawn and reinstated to your Stripe account as a result of this dispute.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub balance_transactions: Vec<BalanceTransaction>,
    #[serde()]
    pub charge: Box<ChargeAnyOf>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde()]
    pub evidence: DisputeEvidence,
    /**
    *
    */
    #[serde()]
    pub evidence_details: DisputeEvidenceDetails,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub is_charge_refundable: bool,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DisputeObject::is_noop")]
    pub object: DisputeObject,
    /**
    * A dispute occurs when a customer questions your charge with their card issuer.
    *  When this happens, you're given the opportunity to respond to the dispute with
    *  evidence that shows that the charge is legitimate. You can find more
    *  information about the dispute process in our [Disputes and
    *  Fraud](/docs/disputes) documentation.
    *  
    *  Related guide: [Disputes and Fraud](https://stripe.com/docs/disputes).
    */
    #[serde()]
    pub payment_intent: Box<Option<PaymentIntentAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
    /**
    * Current status of dispute. Possible values are `warning_needs_response`, `warning_under_review`, `warning_closed`, `needs_response`, `under_review`, `charge_refunded`, `won`, or `lost`.
    */
    #[serde(default, skip_serializing_if = "DisputeStatus::is_noop")]
    pub status: DisputeStatus,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DisputeEvidence {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub access_activity_log: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub billing_address: String,
    /**
    *
    */
    #[serde()]
    pub cancellation_policy: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cancellation_policy_disclosure: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cancellation_rebuttal: String,
    /**
    *
    */
    #[serde()]
    pub customer_communication: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_email_address: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_purchase_ip: String,
    /**
    *
    */
    #[serde()]
    pub customer_signature: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub duplicate_charge_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duplicate_charge_explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub duplicate_charge_id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub product_description: String,
    /**
    *
    */
    #[serde()]
    pub receipt: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub refund_policy: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub refund_policy_disclosure: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub refund_refusal_explanation: String,
    /**
    * The date on which the customer received or began receiving the purchased service, in a clear human-readable format.
    */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_format::deserialize"
    )]
    pub service_date: Option<chrono::NaiveDate>,
    /**
    *
    */
    #[serde()]
    pub service_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub shipping_address: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub shipping_carrier: String,
    /**
    * The date on which the customer received or began receiving the purchased service, in a clear human-readable format.
    */
    #[serde(
        default,
        skip_serializing_if = "Option::is_none",
        deserialize_with = "crate::utils::date_format::deserialize"
    )]
    pub shipping_date: Option<chrono::NaiveDate>,
    /**
    *
    */
    #[serde()]
    pub shipping_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub shipping_tracking_number: String,
    /**
    *
    */
    #[serde()]
    pub uncategorized_file: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub uncategorized_text: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct DisputeEvidenceDetails {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub due_by: i64,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_evidence: bool,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub past_due: bool,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub submission_count: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct EmailSent {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub email_sent_at: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email_sent_to: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum EphemeralKeyObject {
    #[serde(rename = "ephemeral_key")]
    EphemeralKey,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for EphemeralKeyObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            EphemeralKeyObject::EphemeralKey => "ephemeral_key",
            EphemeralKeyObject::Noop => "",
            EphemeralKeyObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for EphemeralKeyObject {
    fn default() -> EphemeralKeyObject {
        EphemeralKeyObject::Noop
    }
}
impl EphemeralKeyObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, EphemeralKeyObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct EphemeralKey {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "EphemeralKeyObject::is_noop")]
    pub object: EphemeralKeyObject,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub secret: String,
}

/// An error response from the Stripe API
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Error {
    /**
    *
    */
    #[serde()]
    pub error: ApiErrors,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum EventObject {
    #[serde(rename = "event")]
    Event,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for EventObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            EventObject::Event => "event",
            EventObject::Noop => "",
            EventObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for EventObject {
    fn default() -> EventObject {
        EventObject::Noop
    }
}
impl EventObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, EventObject::Noop)
    }
}

/// Events are our way of letting you know when something interesting happens in
/// your account. When an interesting event occurs, we create a new `Event`
/// object. For example, when a charge succeeds, we create a `charge.succeeded`
/// event; and when an invoice payment attempt fails, we create an
/// `invoice.payment_failed` event. Note that many API requests may cause multiple
/// events to be created. For example, if you create a new subscription for a
/// customer, you will receive both a `customer.subscription.created` event and a
/// `charge.succeeded` event.
///
/// Events occur when the state of another API resource changes. The state of that
/// resource at the time of the change is embedded in the event's data field. For
/// example, a `charge.succeeded` event will contain a charge, and an
/// `invoice.payment_failed` event will contain an invoice.
///
/// As with other API resources, you can use endpoints to retrieve an
/// [individual event](https://stripe.com/docs/api#retrieve_event) or a [list of events](https://stripe.com/docs/api#list_events)
/// from the API. We also have a separate
/// [webhooks](http://en.wikipedia.org/wiki/Webhook) system for sending the
/// `Event` objects directly to an endpoint on your server. Webhooks are managed
/// in your
/// [account settings](https://dashboard.stripe.com/account/webhooks),
/// and our [Using Webhooks](https://stripe.com/docs/webhooks) guide will help you get set up.
///
/// When using [Connect](https://stripe.com/docs/connect), you can also receive notifications of
/// events that occur in connected accounts. For these events, there will be an
/// additional `account` attribute in the received `Event` object.
///
/// **NOTE:** Right now, access to events through the [Retrieve Event API](https://stripe.com/docs/api#retrieve_event) is
/// guaranteed only for 30 days.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Event {
    /**
    * Events are our way of letting you know when something interesting happens in
    *  your account. When an interesting event occurs, we create a new `Event`
    *  object. For example, when a charge succeeds, we create a `charge.succeeded`
    *  event; and when an invoice payment attempt fails, we create an
    *  `invoice.payment_failed` event. Note that many API requests may cause multiple
    *  events to be created. For example, if you create a new subscription for a
    *  customer, you will receive both a `customer.subscription.created` event and a
    *  `charge.succeeded` event.
    *  
    *  Events occur when the state of another API resource changes. The state of that
    *  resource at the time of the change is embedded in the event's data field. For
    *  example, a `charge.succeeded` event will contain a charge, and an
    *  `invoice.payment_failed` event will contain an invoice.
    *  
    *  As with other API resources, you can use endpoints to retrieve an
    *  [individual event](https://stripe.com/docs/api#retrieve_event) or a [list of events](https://stripe.com/docs/api#list_events)
    *  from the API. We also have a separate
    *  [webhooks](http://en.wikipedia.org/wiki/Webhook) system for sending the
    *  `Event` objects directly to an endpoint on your server. Webhooks are managed
    *  in your
    *  [account settings](https://dashboard.stripe.com/account/webhooks),
    *  and our [Using Webhooks](https://stripe.com/docs/webhooks) guide will help you get set up.
    *  
    *  When using [Connect](https://stripe.com/docs/connect), you can also receive notifications of
    *  events that occur in connected accounts. For these events, there will be an
    *  additional `account` attribute in the received `Event` object.
    *  
    *  \*\*NOTE:\*\* Right now, access to events through the [Retrieve Event API](https://stripe.com/docs/api#retrieve_event) is
    *  guaranteed only for 30 days.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account: String,
    /**
    * Events are our way of letting you know when something interesting happens in
    *  your account. When an interesting event occurs, we create a new `Event`
    *  object. For example, when a charge succeeds, we create a `charge.succeeded`
    *  event; and when an invoice payment attempt fails, we create an
    *  `invoice.payment_failed` event. Note that many API requests may cause multiple
    *  events to be created. For example, if you create a new subscription for a
    *  customer, you will receive both a `customer.subscription.created` event and a
    *  `charge.succeeded` event.
    *  
    *  Events occur when the state of another API resource changes. The state of that
    *  resource at the time of the change is embedded in the event's data field. For
    *  example, a `charge.succeeded` event will contain a charge, and an
    *  `invoice.payment_failed` event will contain an invoice.
    *  
    *  As with other API resources, you can use endpoints to retrieve an
    *  [individual event](https://stripe.com/docs/api#retrieve_event) or a [list of events](https://stripe.com/docs/api#list_events)
    *  from the API. We also have a separate
    *  [webhooks](http://en.wikipedia.org/wiki/Webhook) system for sending the
    *  `Event` objects directly to an endpoint on your server. Webhooks are managed
    *  in your
    *  [account settings](https://dashboard.stripe.com/account/webhooks),
    *  and our [Using Webhooks](https://stripe.com/docs/webhooks) guide will help you get set up.
    *  
    *  When using [Connect](https://stripe.com/docs/connect), you can also receive notifications of
    *  events that occur in connected accounts. For these events, there will be an
    *  additional `account` attribute in the received `Event` object.
    *  
    *  \*\*NOTE:\*\* Right now, access to events through the [Retrieve Event API](https://stripe.com/docs/api#retrieve_event) is
    *  guaranteed only for 30 days.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub api_version: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    *
    */
    #[serde()]
    pub data: NotificationEventData,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "EventObject::is_noop")]
    pub object: EventObject,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pending_webhooks: i64,
    /**
    * Information on the API request that instigated the event.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub request: Option<NotificationEventRequest>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ExchangeRateObject {
    #[serde(rename = "exchange_rate")]
    ExchangeRate,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ExchangeRateObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ExchangeRateObject::ExchangeRate => "exchange_rate",
            ExchangeRateObject::Noop => "",
            ExchangeRateObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ExchangeRateObject {
    fn default() -> ExchangeRateObject {
        ExchangeRateObject::Noop
    }
}
impl ExchangeRateObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, ExchangeRateObject::Noop)
    }
}

/// `Exchange Rate` objects allow you to determine the rates that Stripe is
/// currently using to convert from one currency to another. Since this number is
/// variable throughout the day, there are various reasons why you might want to
/// know the current rate (for example, to dynamically price an item for a user
/// with a default payment in a foreign currency).
///
/// If you want a guarantee that the charge is made with a certain exchange rate
/// you expect is current, you can pass in `exchange_rate` to charges endpoints.
/// If the value is no longer up to date, the charge won't go through. Please
/// refer to our [Exchange Rates API](https://stripe.com/docs/exchange-rates) guide for more
/// details.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct ExchangeRate {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "ExchangeRateObject::is_noop")]
    pub object: ExchangeRateObject,
    /**
    * Hash where the keys are supported currencies and the values are the exchange rate at which the base id currency converts to the key currency.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub rates: f64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Fee {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub application: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum FeeRefundObject {
    #[serde(rename = "fee_refund")]
    FeeRefund,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for FeeRefundObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            FeeRefundObject::FeeRefund => "fee_refund",
            FeeRefundObject::Noop => "",
            FeeRefundObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for FeeRefundObject {
    fn default() -> FeeRefundObject {
        FeeRefundObject::Noop
    }
}
impl FeeRefundObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, FeeRefundObject::Noop)
    }
}

/// `Application Fee Refund` objects allow you to refund an application fee that
/// has previously been created but not yet refunded. Funds will be refunded to
/// the Stripe account from which the fee was originally collected.
///
/// Related guide: [Refunding Application Fees](https://stripe.com/docs/connect/destination-charges#refunding-app-fee).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct FeeRefund {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * `Application Fee Refund` objects allow you to refund an application fee that
    *  has previously been created but not yet refunded. Funds will be refunded to
    *  the Stripe account from which the fee was originally collected.
    *  
    *  Related guide: [Refunding Application Fees](https://stripe.com/docs/connect/destination-charges#refunding-app-fee).
    */
    #[serde()]
    pub balance_transaction: Box<Option<BalanceTransactionAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    #[serde()]
    pub fee: Box<FeeAnyOf>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * `Application Fee Refund` objects allow you to refund an application fee that
    *  has previously been created but not yet refunded. Funds will be refunded to
    *  the Stripe account from which the fee was originally collected.
    *  
    *  Related guide: [Refunding Application Fees](https://stripe.com/docs/connect/destination-charges#refunding-app-fee).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "FeeRefundObject::is_noop")]
    pub object: FeeRefundObject,
}

/// A list of [file links](https://stripe.com/docs/api#file_links) that point at this file.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Links {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<FileLink>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum FileObject {
    #[serde(rename = "file")]
    File,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for FileObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            FileObject::File => "file",
            FileObject::Noop => "",
            FileObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for FileObject {
    fn default() -> FileObject {
        FileObject::Noop
    }
}
impl FileObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, FileObject::Noop)
    }
}

/**
* The [purpose](https://stripe.com/docs/file-upload#uploading-a-file) of the uploaded file.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Purpose {
    #[serde(rename = "account_requirement")]
    AccountRequirement,
    #[serde(rename = "additional_verification")]
    AdditionalVerification,
    #[serde(rename = "business_icon")]
    BusinessIcon,
    #[serde(rename = "business_logo")]
    BusinessLogo,
    #[serde(rename = "customer_signature")]
    CustomerSignature,
    #[serde(rename = "dispute_evidence")]
    DisputeEvidence,
    #[serde(rename = "document_provider_identity_document")]
    DocumentProviderIdentity,
    #[serde(rename = "finance_report_run")]
    FinanceReportRun,
    #[serde(rename = "identity_document")]
    IdentityDocument,
    #[serde(rename = "identity_document_downloadable")]
    IdentityDocumentDownloadable,
    #[serde(rename = "pci_document")]
    PciDocument,
    #[serde(rename = "selfie")]
    Selfie,
    #[serde(rename = "sigma_scheduled_query")]
    SigmaScheduledQuery,
    #[serde(rename = "tax_document_user_upload")]
    TaxDocumentUserUpload,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Purpose {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Purpose::AccountRequirement => "account_requirement",
            Purpose::AdditionalVerification => "additional_verification",
            Purpose::BusinessIcon => "business_icon",
            Purpose::BusinessLogo => "business_logo",
            Purpose::CustomerSignature => "customer_signature",
            Purpose::DisputeEvidence => "dispute_evidence",
            Purpose::DocumentProviderIdentity => "document_provider_identity_document",
            Purpose::FinanceReportRun => "finance_report_run",
            Purpose::IdentityDocument => "identity_document",
            Purpose::IdentityDocumentDownloadable => "identity_document_downloadable",
            Purpose::PciDocument => "pci_document",
            Purpose::Selfie => "selfie",
            Purpose::SigmaScheduledQuery => "sigma_scheduled_query",
            Purpose::TaxDocumentUserUpload => "tax_document_user_upload",
            Purpose::Noop => "",
            Purpose::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Purpose {
    fn default() -> Purpose {
        Purpose::Noop
    }
}
impl Purpose {
    pub fn is_noop(&self) -> bool {
        matches!(self, Purpose::Noop)
    }
}

/// This is an object representing a file hosted on Stripe's servers. The
/// file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
/// request (for example, when uploading dispute evidence) or it may have
/// been created by Stripe (for example, the results of a [Sigma scheduled
/// query](#scheduled_queries)).
///
/// Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
#[derive(Serialize, Default, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct File {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * This is an object representing a file hosted on Stripe's servers. The
    *  file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
    *  request (for example, when uploading dispute evidence) or it may have
    *  been created by Stripe (for example, the results of a [Sigma scheduled
    *  query](#scheduled_queries)).
    *  
    *  Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_at: i64,
    /**
    * This is an object representing a file hosted on Stripe's servers. The
    *  file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
    *  request (for example, when uploading dispute evidence) or it may have
    *  been created by Stripe (for example, the results of a [Sigma scheduled
    *  query](#scheduled_queries)).
    *  
    *  Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub filename: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * A list of [file links](https://stripe.com/docs/api#file_links) that point at this file.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub links: Option<Links>,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "FileObject::is_noop")]
    pub object: FileObject,
    /**
    * The [purpose](https://stripe.com/docs/file-upload#uploading-a-file) of the uploaded file.
    */
    #[serde(default, skip_serializing_if = "Purpose::is_noop")]
    pub purpose: Purpose,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub size: i64,
    /**
    * This is an object representing a file hosted on Stripe's servers. The
    *  file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
    *  request (for example, when uploading dispute evidence) or it may have
    *  been created by Stripe (for example, the results of a [Sigma scheduled
    *  query](#scheduled_queries)).
    *  
    *  Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub title: String,
    /**
    * This is an object representing a file hosted on Stripe's servers. The
    *  file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
    *  request (for example, when uploading dispute evidence) or it may have
    *  been created by Stripe (for example, the results of a [Sigma scheduled
    *  query](#scheduled_queries)).
    *  
    *  Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
    * This is an object representing a file hosted on Stripe's servers. The
    *  file may have been uploaded by yourself using the [create file](https://stripe.com/docs/api#create_file)
    *  request (for example, when uploading dispute evidence) or it may have
    *  been created by Stripe (for example, the results of a [Sigma scheduled
    *  query](#scheduled_queries)).
    *  
    *  Related guide: [File Upload Guide](https://stripe.com/docs/file-upload).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum FileLinkObject {
    #[serde(rename = "file_link")]
    FileLink,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for FileLinkObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            FileLinkObject::FileLink => "file_link",
            FileLinkObject::Noop => "",
            FileLinkObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for FileLinkObject {
    fn default() -> FileLinkObject {
        FileLinkObject::Noop
    }
}
impl FileLinkObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, FileLinkObject::Noop)
    }
}

/// To share the contents of a `File` object with non-Stripe users, you can
/// create a `FileLink`. `FileLink`s contain a URL that can be used to
/// retrieve the contents of the file without authentication.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct FileLink {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub expired: bool,
    /**
    * To share the contents of a `File` object with non-Stripe users, you can
    *  create a `FileLink`. `FileLink`s contain a URL that can be used to
    *  retrieve the contents of the file without authentication.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_at: i64,
    #[serde()]
    pub file: Box<IconAnyOf>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "FileLinkObject::is_noop")]
    pub object: FileLinkObject,
    /**
    * To share the contents of a `File` object with non-Stripe users, you can
    *  create a `FileLink`. `FileLink`s contain a URL that can be used to
    *  retrieve the contents of the file without authentication.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct FinancialReportingFinanceReportRunParameters {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub columns: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub connected_account: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub interval_end: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub interval_start: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payout: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reporting_category: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timezone: String,
}

/// Point in Time
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDataDocumentReportDateOfBirth {
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/// Point in Time
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDataDocumentReportExpirationDate {
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/// Point in Time
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDataDocumentReportIssuedDate {
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/// Point in Time
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDataNumberReportDate {
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/// Point in Time
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDataVerifiedOutputsDate {
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    * Point in Time
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/**
* Status of this `document` check.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoDocumentReportStatus {
    #[serde(rename = "unverified")]
    Unverified,
    #[serde(rename = "verified")]
    Verified,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoDocumentReportStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoDocumentReportStatus::Unverified => "unverified",
            GelatoDocumentReportStatus::Verified => "verified",
            GelatoDocumentReportStatus::Noop => "",
            GelatoDocumentReportStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoDocumentReportStatus {
    fn default() -> GelatoDocumentReportStatus {
        GelatoDocumentReportStatus::Noop
    }
}
impl GelatoDocumentReportStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoDocumentReportStatus::Noop)
    }
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AllowedTypes {
    #[serde(rename = "driving_license")]
    DrivingLicense,
    #[serde(rename = "id_card")]
    IdCard,
    #[serde(rename = "passport")]
    Passport,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AllowedTypes {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AllowedTypes::DrivingLicense => "driving_license",
            AllowedTypes::IdCard => "id_card",
            AllowedTypes::Passport => "passport",
            AllowedTypes::Noop => "",
            AllowedTypes::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AllowedTypes {
    fn default() -> AllowedTypes {
        AllowedTypes::Noop
    }
}
impl AllowedTypes {
    pub fn is_noop(&self) -> bool {
        matches!(self, AllowedTypes::Noop)
    }
}

/// Result from a document check
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDocumentReport {
    /**
    * A publicly available mailing address for sending support issues to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address: Option<Address>,
    /**
    * Date of birth as it appears in the document.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub dob: Option<GelatoDataDocumentReportDateOfBirth>,
    /**
    * Details on the verification error. Present when status is `unverified`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub error: Option<GelatoDocumentReportErrorData>,
    /**
    * Expiration date of the document.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub expiration_date: Option<GelatoDataDocumentReportExpirationDate>,
    /**
    * Result from a document check
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub files: Vec<String>,
    /**
    * Result from a document check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
    * Issued date of the document.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub issued_date: Option<GelatoDataDocumentReportIssuedDate>,
    /**
    * Result from a document check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub issuing_country: String,
    /**
    * Result from a document check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
    * Result from a document check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
    * Status of this `document` check.
    */
    #[serde(default, skip_serializing_if = "GelatoDocumentReportStatus::is_noop")]
    pub status: GelatoDocumentReportStatus,
    /**
    * Type of the document.
    */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "type")]
    pub type_: Option<AllowedTypes>,
}

/**
* A short machine-readable string giving the reason for the verification failure.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoDocumentReportErrorCode {
    #[serde(rename = "document_expired")]
    DocumentExpired,
    #[serde(rename = "document_type_not_supported")]
    DocumentTypeNotSupported,
    #[serde(rename = "document_unverified_other")]
    DocumentUnverifiedOther,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoDocumentReportErrorCode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoDocumentReportErrorCode::DocumentExpired => "document_expired",
            GelatoDocumentReportErrorCode::DocumentTypeNotSupported => {
                "document_type_not_supported"
            }
            GelatoDocumentReportErrorCode::DocumentUnverifiedOther => "document_unverified_other",
            GelatoDocumentReportErrorCode::Noop => "",
            GelatoDocumentReportErrorCode::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoDocumentReportErrorCode {
    fn default() -> GelatoDocumentReportErrorCode {
        GelatoDocumentReportErrorCode::Noop
    }
}
impl GelatoDocumentReportErrorCode {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoDocumentReportErrorCode::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoDocumentReportErrorData {
    /**
    * A short machine-readable string giving the reason for the verification failure.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub code: Option<GelatoDocumentReportErrorCode>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
}

/**
* Type of ID number.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IdNumberType {
    #[serde(rename = "br_cpf")]
    BrCpf,
    #[serde(rename = "sg_nric")]
    SgNric,
    #[serde(rename = "us_ssn")]
    UsSsn,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IdNumberType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IdNumberType::BrCpf => "br_cpf",
            IdNumberType::SgNric => "sg_nric",
            IdNumberType::UsSsn => "us_ssn",
            IdNumberType::Noop => "",
            IdNumberType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IdNumberType {
    fn default() -> IdNumberType {
        IdNumberType::Noop
    }
}
impl IdNumberType {
    pub fn is_noop(&self) -> bool {
        matches!(self, IdNumberType::Noop)
    }
}

/// Result from an id_number check
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoNumberReport {
    /**
    * Date of birth.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub dob: Option<GelatoDataNumberReportDate>,
    /**
    * Details on the verification error. Present when status is `unverified`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub error: Option<GelatoNumberReportErrorData>,
    /**
    * Result from an id_number check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
    * Result from an id_number check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id_number: String,
    /**
    * Type of ID number.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub id_number_type: Option<IdNumberType>,
    /**
    * Result from an id_number check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
    * Status of this `document` check.
    */
    #[serde(default, skip_serializing_if = "GelatoDocumentReportStatus::is_noop")]
    pub status: GelatoDocumentReportStatus,
}

/**
* A short machine-readable string giving the reason for the verification failure.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoNumberReportErrorCode {
    #[serde(rename = "id_number_insufficient_document_data")]
    IdNumberInsufficientDocumentData,
    #[serde(rename = "id_number_mismatch")]
    IdNumberMismatch,
    #[serde(rename = "id_number_unverified_other")]
    IdNumberUnverifiedOther,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoNumberReportErrorCode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoNumberReportErrorCode::IdNumberInsufficientDocumentData => {
                "id_number_insufficient_document_data"
            }
            GelatoNumberReportErrorCode::IdNumberMismatch => "id_number_mismatch",
            GelatoNumberReportErrorCode::IdNumberUnverifiedOther => "id_number_unverified_other",
            GelatoNumberReportErrorCode::Noop => "",
            GelatoNumberReportErrorCode::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoNumberReportErrorCode {
    fn default() -> GelatoNumberReportErrorCode {
        GelatoNumberReportErrorCode::Noop
    }
}
impl GelatoNumberReportErrorCode {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoNumberReportErrorCode::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoNumberReportErrorData {
    /**
    * A short machine-readable string giving the reason for the verification failure.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub code: Option<GelatoNumberReportErrorCode>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoReportDocumentOptions {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub allowed_types: Vec<AllowedTypes>,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_id_number: bool,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_live_capture: bool,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_matching_selfie: bool,
}

/// Result from a selfie check
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoSelfieReport {
    /**
    * Result from a selfie check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub document: String,
    /**
    * Details on the verification error. Present when status is `unverified`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub error: Option<GelatoSelfieReportErrorData>,
    /**
    * Result from a selfie check
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub selfie: String,
    /**
    * Status of this `document` check.
    */
    #[serde(default, skip_serializing_if = "GelatoDocumentReportStatus::is_noop")]
    pub status: GelatoDocumentReportStatus,
}

/**
* A short machine-readable string giving the reason for the verification failure.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoSelfieReportErrorCode {
    #[serde(rename = "selfie_document_missing_photo")]
    SelfieDocumentMissingPhoto,
    #[serde(rename = "selfie_face_mismatch")]
    SelfieFaceMismatch,
    #[serde(rename = "selfie_manipulated")]
    SelfieManipulated,
    #[serde(rename = "selfie_unverified_other")]
    SelfieUnverifiedOther,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoSelfieReportErrorCode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoSelfieReportErrorCode::SelfieDocumentMissingPhoto => {
                "selfie_document_missing_photo"
            }
            GelatoSelfieReportErrorCode::SelfieFaceMismatch => "selfie_face_mismatch",
            GelatoSelfieReportErrorCode::SelfieManipulated => "selfie_manipulated",
            GelatoSelfieReportErrorCode::SelfieUnverifiedOther => "selfie_unverified_other",
            GelatoSelfieReportErrorCode::Noop => "",
            GelatoSelfieReportErrorCode::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoSelfieReportErrorCode {
    fn default() -> GelatoSelfieReportErrorCode {
        GelatoSelfieReportErrorCode::Noop
    }
}
impl GelatoSelfieReportErrorCode {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoSelfieReportErrorCode::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoSelfieReportErrorData {
    /**
    * A short machine-readable string giving the reason for the verification failure.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub code: Option<GelatoSelfieReportErrorCode>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoSessionDocumentOptions {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub allowed_types: Vec<AllowedTypes>,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_id_number: bool,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_live_capture: bool,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub require_matching_selfie: bool,
}

/**
* A short machine-readable string giving the reason for the verification or user-session failure.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoSessionLastErrorCode {
    #[serde(rename = "abandoned")]
    Abandoned,
    #[serde(rename = "consent_declined")]
    ConsentDeclined,
    #[serde(rename = "country_not_supported")]
    CountryNotSupported,
    #[serde(rename = "device_not_supported")]
    DeviceNotSupported,
    #[serde(rename = "document_expired")]
    DocumentExpired,
    #[serde(rename = "document_type_not_supported")]
    DocumentTypeNotSupported,
    #[serde(rename = "document_unverified_other")]
    DocumentUnverifiedOther,
    #[serde(rename = "id_number_insufficient_document_data")]
    IdNumberInsufficientDocumentData,
    #[serde(rename = "id_number_mismatch")]
    IdNumberMismatch,
    #[serde(rename = "id_number_unverified_other")]
    IdNumberUnverifiedOther,
    #[serde(rename = "selfie_document_missing_photo")]
    SelfieDocumentMissingPhoto,
    #[serde(rename = "selfie_face_mismatch")]
    SelfieFaceMismatch,
    #[serde(rename = "selfie_manipulated")]
    SelfieManipulated,
    #[serde(rename = "selfie_unverified_other")]
    SelfieUnverifiedOther,
    #[serde(rename = "under_supported_age")]
    UnderSupportedAge,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoSessionLastErrorCode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoSessionLastErrorCode::Abandoned => "abandoned",
            GelatoSessionLastErrorCode::ConsentDeclined => "consent_declined",
            GelatoSessionLastErrorCode::CountryNotSupported => "country_not_supported",
            GelatoSessionLastErrorCode::DeviceNotSupported => "device_not_supported",
            GelatoSessionLastErrorCode::DocumentExpired => "document_expired",
            GelatoSessionLastErrorCode::DocumentTypeNotSupported => "document_type_not_supported",
            GelatoSessionLastErrorCode::DocumentUnverifiedOther => "document_unverified_other",
            GelatoSessionLastErrorCode::IdNumberInsufficientDocumentData => {
                "id_number_insufficient_document_data"
            }
            GelatoSessionLastErrorCode::IdNumberMismatch => "id_number_mismatch",
            GelatoSessionLastErrorCode::IdNumberUnverifiedOther => "id_number_unverified_other",
            GelatoSessionLastErrorCode::SelfieDocumentMissingPhoto => {
                "selfie_document_missing_photo"
            }
            GelatoSessionLastErrorCode::SelfieFaceMismatch => "selfie_face_mismatch",
            GelatoSessionLastErrorCode::SelfieManipulated => "selfie_manipulated",
            GelatoSessionLastErrorCode::SelfieUnverifiedOther => "selfie_unverified_other",
            GelatoSessionLastErrorCode::UnderSupportedAge => "under_supported_age",
            GelatoSessionLastErrorCode::Noop => "",
            GelatoSessionLastErrorCode::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoSessionLastErrorCode {
    fn default() -> GelatoSessionLastErrorCode {
        GelatoSessionLastErrorCode::Noop
    }
}
impl GelatoSessionLastErrorCode {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoSessionLastErrorCode::Noop)
    }
}

/// Shows last VerificationSession error
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoSessionLastError {
    /**
    * A short machine-readable string giving the reason for the verification or user-session failure.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub code: Option<GelatoSessionLastErrorCode>,
    /**
    * Shows last VerificationSession error
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reason: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoVerificationReportOptions {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub document: Option<GelatoReportDocumentOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub id_number: Option<UseStripeSdk>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoVerificationSessionOptions {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub document: Option<GelatoSessionDocumentOptions>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub id_number: Option<UseStripeSdk>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoVerifiedOutputs {
    /**
    * A publicly available mailing address for sending support issues to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address: Option<Address>,
    /**
    * The user’s verified date of birth.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub dob: Option<GelatoDataVerifiedOutputsDate>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id_number: String,
    /**
    * Type of ID number.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub id_number_type: Option<IdNumberType>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoVerificationReportObject {
    #[serde(rename = "identity.verification_report")]
    IdentityVerificationReport,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoVerificationReportObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoVerificationReportObject::IdentityVerificationReport => {
                "identity.verification_report"
            }
            GelatoVerificationReportObject::Noop => "",
            GelatoVerificationReportObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoVerificationReportObject {
    fn default() -> GelatoVerificationReportObject {
        GelatoVerificationReportObject::Noop
    }
}
impl GelatoVerificationReportObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoVerificationReportObject::Noop)
    }
}

/**
* Type of report.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoVerificationReportType {
    #[serde(rename = "document")]
    Document,
    #[serde(rename = "id_number")]
    IdNumber,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoVerificationReportType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoVerificationReportType::Document => "document",
            GelatoVerificationReportType::IdNumber => "id_number",
            GelatoVerificationReportType::Noop => "",
            GelatoVerificationReportType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoVerificationReportType {
    fn default() -> GelatoVerificationReportType {
        GelatoVerificationReportType::Noop
    }
}
impl GelatoVerificationReportType {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoVerificationReportType::Noop)
    }
}

/// A VerificationReport is the result of an attempt to collect and verify data from a user.
/// The collection of verification checks performed is determined from the `type` and `options`
/// parameters used. You can find the result of each verification check performed in the
/// appropriate sub-resource: `document`, `id_number`, `selfie`.
///
/// Each VerificationReport contains a copy of any data collected by the user as well as
/// reference IDs which can be used to access collected images through the [FileUpload](https://stripe.com/docs/api/files)
/// API. To configure and create VerificationReports, use the
/// [VerificationSession](https://stripe.com/docs/api/identity/verification_sessions) API.
///
/// Related guides: [Accessing verification results](https://stripe.com/docs/identity/verification-sessions#results).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoVerificationReport {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * A VerificationReport is the result of an attempt to collect and verify data from a user.
    *  The collection of verification checks performed is determined from the `type` and `options`
    *  parameters used. You can find the result of each verification check performed in the
    *  appropriate sub-resource: `document`, `id_number`, `selfie`.
    *  
    *  Each VerificationReport contains a copy of any data collected by the user as well as
    *  reference IDs which can be used to access collected images through the [FileUpload](https://stripe.com/docs/api/files)
    *  API. To configure and create VerificationReports, use the
    *  [VerificationSession](https://stripe.com/docs/api/identity/verification_sessions) API.
    *  
    *  Related guides: [Accessing verification results](https://stripe.com/docs/identity/verification-sessions#results).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub document: Option<GelatoDocumentReport>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * A VerificationReport is the result of an attempt to collect and verify data from a user.
    *  The collection of verification checks performed is determined from the `type` and `options`
    *  parameters used. You can find the result of each verification check performed in the
    *  appropriate sub-resource: `document`, `id_number`, `selfie`.
    *  
    *  Each VerificationReport contains a copy of any data collected by the user as well as
    *  reference IDs which can be used to access collected images through the [FileUpload](https://stripe.com/docs/api/files)
    *  API. To configure and create VerificationReports, use the
    *  [VerificationSession](https://stripe.com/docs/api/identity/verification_sessions) API.
    *  
    *  Related guides: [Accessing verification results](https://stripe.com/docs/identity/verification-sessions#results).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub id_number: Option<GelatoNumberReport>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(
        default,
        skip_serializing_if = "GelatoVerificationReportObject::is_noop"
    )]
    pub object: GelatoVerificationReportObject,
    /**
    *
    */
    #[serde()]
    pub options: GelatoVerificationReportOptions,
    /**
    * A VerificationReport is the result of an attempt to collect and verify data from a user.
    *  The collection of verification checks performed is determined from the `type` and `options`
    *  parameters used. You can find the result of each verification check performed in the
    *  appropriate sub-resource: `document`, `id_number`, `selfie`.
    *  
    *  Each VerificationReport contains a copy of any data collected by the user as well as
    *  reference IDs which can be used to access collected images through the [FileUpload](https://stripe.com/docs/api/files)
    *  API. To configure and create VerificationReports, use the
    *  [VerificationSession](https://stripe.com/docs/api/identity/verification_sessions) API.
    *  
    *  Related guides: [Accessing verification results](https://stripe.com/docs/identity/verification-sessions#results).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub selfie: Option<GelatoSelfieReport>,
    /**
    * Type of report.
    */
    #[serde(
        default,
        skip_serializing_if = "GelatoVerificationReportType::is_noop",
        rename = "type"
    )]
    pub type_: GelatoVerificationReportType,
    /**
    * A VerificationReport is the result of an attempt to collect and verify data from a user.
    *  The collection of verification checks performed is determined from the `type` and `options`
    *  parameters used. You can find the result of each verification check performed in the
    *  appropriate sub-resource: `document`, `id_number`, `selfie`.
    *  
    *  Each VerificationReport contains a copy of any data collected by the user as well as
    *  reference IDs which can be used to access collected images through the [FileUpload](https://stripe.com/docs/api/files)
    *  API. To configure and create VerificationReports, use the
    *  [VerificationSession](https://stripe.com/docs/api/identity/verification_sessions) API.
    *  
    *  Related guides: [Accessing verification results](https://stripe.com/docs/identity/verification-sessions#results).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub verification_session: String,
}

/// All of the following types:
///
/// - `String`
/// - `GelatoVerificationReport`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum LastVerificationReportAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A VerificationReport is the result of an attempt to collect and verify data from a user.
    *  The collection of verification checks performed is determined from the `type` and `options`
    *  parameters used. You can find the result of each verification check performed in the
    *  appropriate sub-resource: `document`, `id_number`, `selfie`.
    *  
    *  Each VerificationReport contains a copy of any data collected by the user as well as
    *  reference IDs which can be used to access collected images through the [FileUpload](https://stripe.com/docs/api/files)
    *  API. To configure and create VerificationReports, use the
    *  [VerificationSession](https://stripe.com/docs/api/identity/verification_sessions) API.
    *  
    *  Related guides: [Accessing verification results](https://stripe.com/docs/identity/verification-sessions#results).
    */
    GelatoVerificationReport(GelatoVerificationReport),
}

impl LastVerificationReportAnyOf {
    pub fn gelato_verification_report(&self) -> Option<&GelatoVerificationReport> {
        if let LastVerificationReportAnyOf::GelatoVerificationReport(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let LastVerificationReportAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for LastVerificationReportAnyOf {
    fn from(f: String) -> Self {
        LastVerificationReportAnyOf::String(f)
    }
}

impl std::convert::From<LastVerificationReportAnyOf> for String {
    fn from(f: LastVerificationReportAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoVerificationSessionObject {
    #[serde(rename = "identity.verification_session")]
    IdentityVerificationSession,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoVerificationSessionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoVerificationSessionObject::IdentityVerificationSession => {
                "identity.verification_session"
            }
            GelatoVerificationSessionObject::Noop => "",
            GelatoVerificationSessionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoVerificationSessionObject {
    fn default() -> GelatoVerificationSessionObject {
        GelatoVerificationSessionObject::Noop
    }
}
impl GelatoVerificationSessionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoVerificationSessionObject::Noop)
    }
}

/**
* Status of this VerificationSession. [Learn more about the lifecycle of sessions](https://stripe.com/docs/identity/how-sessions-work).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum GelatoVerificationSessionStatus {
    #[serde(rename = "canceled")]
    Canceled,
    #[serde(rename = "processing")]
    Processing,
    #[serde(rename = "requires_input")]
    RequiresInput,
    #[serde(rename = "verified")]
    Verified,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for GelatoVerificationSessionStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            GelatoVerificationSessionStatus::Canceled => "canceled",
            GelatoVerificationSessionStatus::Processing => "processing",
            GelatoVerificationSessionStatus::RequiresInput => "requires_input",
            GelatoVerificationSessionStatus::Verified => "verified",
            GelatoVerificationSessionStatus::Noop => "",
            GelatoVerificationSessionStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for GelatoVerificationSessionStatus {
    fn default() -> GelatoVerificationSessionStatus {
        GelatoVerificationSessionStatus::Noop
    }
}
impl GelatoVerificationSessionStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, GelatoVerificationSessionStatus::Noop)
    }
}

/// A VerificationSession guides you through the process of collecting and verifying the identities
/// of your users. It contains details about the type of verification, such as what [verification
/// check](/docs/identity/verification-checks) to perform. Only create one VerificationSession for
/// each verification in your system.
///
/// A VerificationSession transitions through [multiple
/// statuses](/docs/identity/how-sessions-work) throughout its lifetime as it progresses through
/// the verification flow. The VerificationSession contains the user’s verified data after
/// verification checks are complete.
///
/// Related guide: [The Verification Sessions API](https://stripe.com/docs/identity/verification-sessions)
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct GelatoVerificationSession {
    /**
    * A VerificationSession guides you through the process of collecting and verifying the identities
    *  of your users. It contains details about the type of verification, such as what [verification
    *  check](/docs/identity/verification-checks) to perform. Only create one VerificationSession for
    *  each verification in your system.
    *  
    *  A VerificationSession transitions through [multiple
    *  statuses](/docs/identity/how-sessions-work) throughout its lifetime as it progresses through
    *  the verification flow. The VerificationSession contains the user’s verified data after
    *  verification checks are complete.
    *  
    *  Related guide: [The Verification Sessions API](https://stripe.com/docs/identity/verification-sessions)
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub client_secret: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * If present, this property tells you the last error encountered when processing the verification.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub last_error: Option<GelatoSessionLastError>,
    /**
    * A VerificationSession guides you through the process of collecting and verifying the identities
    *  of your users. It contains details about the type of verification, such as what [verification
    *  check](/docs/identity/verification-checks) to perform. Only create one VerificationSession for
    *  each verification in your system.
    *  
    *  A VerificationSession transitions through [multiple
    *  statuses](/docs/identity/how-sessions-work) throughout its lifetime as it progresses through
    *  the verification flow. The VerificationSession contains the user’s verified data after
    *  verification checks are complete.
    *  
    *  Related guide: [The Verification Sessions API](https://stripe.com/docs/identity/verification-sessions)
    */
    #[serde()]
    pub last_verification_report: Box<Option<LastVerificationReportAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(
        default,
        skip_serializing_if = "GelatoVerificationSessionObject::is_noop"
    )]
    pub object: GelatoVerificationSessionObject,
    /**
    *
    */
    #[serde()]
    pub options: GelatoVerificationSessionOptions,
    /**
    * Redaction status of this VerificationSession. If the VerificationSession is not redacted, this field will be null.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub redaction: Option<VerificationSessionRedaction>,
    /**
    * Status of this VerificationSession. [Learn more about the lifecycle of sessions](https://stripe.com/docs/identity/how-sessions-work).
    */
    #[serde(
        default,
        skip_serializing_if = "GelatoVerificationSessionStatus::is_noop"
    )]
    pub status: GelatoVerificationSessionStatus,
    /**
    * Type of report.
    */
    #[serde(
        default,
        skip_serializing_if = "GelatoVerificationReportType::is_noop",
        rename = "type"
    )]
    pub type_: GelatoVerificationReportType,
    /**
    * A VerificationSession guides you through the process of collecting and verifying the identities
    *  of your users. It contains details about the type of verification, such as what [verification
    *  check](/docs/identity/verification-checks) to perform. Only create one VerificationSession for
    *  each verification in your system.
    *  
    *  A VerificationSession transitions through [multiple
    *  statuses](/docs/identity/how-sessions-work) throughout its lifetime as it progresses through
    *  the verification flow. The VerificationSession contains the user’s verified data after
    *  verification checks are complete.
    *  
    *  Related guide: [The Verification Sessions API](https://stripe.com/docs/identity/verification-sessions)
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
    /**
    * The user’s verified data.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verified_outputs: Option<GelatoVerifiedOutputs>,
}

/// All of the following types:
///
/// - `String`
/// - `TaxId`
/// - `DeletedTaxId`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AccountTaxIdsAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * You can add one or multiple tax IDs to a [customer](https://stripe.com/docs/api/customers).
    *  A customer's tax IDs are displayed on invoices and credit notes issued for the customer.
    *  
    *  Related guide: [Customer Tax Identification Numbers](https://stripe.com/docs/billing/taxes/tax-ids).
    */
    TaxId(TaxId),
    /**
    *
    */
    DeletedTaxId(DeletedTaxId),
}

impl AccountTaxIdsAnyOf {
    pub fn deleted_tax_id(&self) -> Option<&DeletedTaxId> {
        if let AccountTaxIdsAnyOf::DeletedTaxId(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let AccountTaxIdsAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn tax_id(&self) -> Option<&TaxId> {
        if let AccountTaxIdsAnyOf::TaxId(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for AccountTaxIdsAnyOf {
    fn from(f: String) -> Self {
        AccountTaxIdsAnyOf::String(f)
    }
}

impl std::convert::From<AccountTaxIdsAnyOf> for String {
    fn from(f: AccountTaxIdsAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* Indicates the reason why the invoice was created. `subscription_cycle` indicates an invoice created by a subscription advancing into a new period. `subscription_create` indicates an invoice created due to creating a subscription. `subscription_update` indicates an invoice created due to updating a subscription. `subscription` is set for all old invoices to indicate either a change to a subscription or a period advancement. `manual` is set for all invoices unrelated to a subscription (for example: created via the invoice editor). The `upcoming` value is reserved for simulated invoices per the upcoming invoice endpoint. `subscription_threshold` indicates an invoice created due to a billing threshold being reached.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum BillingReason {
    #[serde(rename = "automatic_pending_invoice_item_invoice")]
    AutomaticPendingInvoiceItem,
    #[serde(rename = "manual")]
    Manual,
    #[serde(rename = "quote_accept")]
    QuoteAccept,
    #[serde(rename = "subscription")]
    Subscription,
    #[serde(rename = "subscription_create")]
    SubscriptionCreate,
    #[serde(rename = "subscription_cycle")]
    SubscriptionCycle,
    #[serde(rename = "subscription_threshold")]
    SubscriptionThreshold,
    #[serde(rename = "subscription_update")]
    SubscriptionUpdate,
    #[serde(rename = "upcoming")]
    Upcoming,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for BillingReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            BillingReason::AutomaticPendingInvoiceItem => "automatic_pending_invoice_item_invoice",
            BillingReason::Manual => "manual",
            BillingReason::QuoteAccept => "quote_accept",
            BillingReason::Subscription => "subscription",
            BillingReason::SubscriptionCreate => "subscription_create",
            BillingReason::SubscriptionCycle => "subscription_cycle",
            BillingReason::SubscriptionThreshold => "subscription_threshold",
            BillingReason::SubscriptionUpdate => "subscription_update",
            BillingReason::Upcoming => "upcoming",
            BillingReason::Noop => "",
            BillingReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for BillingReason {
    fn default() -> BillingReason {
        BillingReason::Noop
    }
}
impl BillingReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, BillingReason::Noop)
    }
}

/**
* Either `charge_automatically`, or `send_invoice`. When charging automatically, Stripe will attempt to pay this invoice using the default source attached to the customer. When sending an invoice, Stripe will email this invoice to the customer with payment instructions.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CollectionMethod {
    #[serde(rename = "charge_automatically")]
    ChargeAutomatically,
    #[serde(rename = "send_invoice")]
    SendInvoice,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CollectionMethod {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CollectionMethod::ChargeAutomatically => "charge_automatically",
            CollectionMethod::SendInvoice => "send_invoice",
            CollectionMethod::Noop => "",
            CollectionMethod::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CollectionMethod {
    fn default() -> CollectionMethod {
        CollectionMethod::Noop
    }
}
impl CollectionMethod {
    pub fn is_noop(&self) -> bool {
        matches!(self, CollectionMethod::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `PaymentMethod`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PaymentMethodAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    PaymentMethod(PaymentMethod),
}

impl PaymentMethodAnyOf {
    pub fn payment_method(&self) -> Option<&PaymentMethod> {
        if let PaymentMethodAnyOf::PaymentMethod(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let PaymentMethodAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for PaymentMethodAnyOf {
    fn from(f: String) -> Self {
        PaymentMethodAnyOf::String(f)
    }
}

impl std::convert::From<PaymentMethodAnyOf> for String {
    fn from(f: PaymentMethodAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// The individual line items that make up the invoice. `lines` is sorted as follows: invoice items in reverse chronological order, followed by the subscription, if any.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceLinesList {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<LineItem>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/// All of the following types:
///
/// - `String`
/// - `Quote`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum QuoteAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A Quote is a way to model prices that you'd like to provide to a customer.
    *  Once accepted, it will automatically create an invoice, subscription or subscription schedule.
    */
    Quote(Quote),
}

impl QuoteAnyOf {
    pub fn quote(&self) -> Option<&Quote> {
        if let QuoteAnyOf::Quote(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let QuoteAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for QuoteAnyOf {
    fn from(f: String) -> Self {
        QuoteAnyOf::String(f)
    }
}

impl std::convert::From<QuoteAnyOf> for String {
    fn from(f: QuoteAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* The status of the invoice, one of `draft`, `open`, `paid`, `uncollectible`, or `void`. [Learn more](https://stripe.com/docs/billing/invoices/workflow#workflow-overview)
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum InvoiceStatus {
    #[serde(rename = "deleted")]
    Deleted,
    #[serde(rename = "draft")]
    Draft,
    #[serde(rename = "open")]
    Open,
    #[serde(rename = "paid")]
    Paid,
    #[serde(rename = "uncollectible")]
    Uncollectible,
    #[serde(rename = "void")]
    Void,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for InvoiceStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            InvoiceStatus::Deleted => "deleted",
            InvoiceStatus::Draft => "draft",
            InvoiceStatus::Open => "open",
            InvoiceStatus::Paid => "paid",
            InvoiceStatus::Uncollectible => "uncollectible",
            InvoiceStatus::Void => "void",
            InvoiceStatus::Noop => "",
            InvoiceStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for InvoiceStatus {
    fn default() -> InvoiceStatus {
        InvoiceStatus::Noop
    }
}
impl InvoiceStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, InvoiceStatus::Noop)
    }
}

/// Invoices are statements of amounts owed by a customer, and are either
/// generated one-off, or generated periodically from a subscription.
///
/// They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
/// that may be caused by subscription upgrades/downgrades (if necessary).
///
/// If your invoice is configured to be billed through automatic charges,
/// Stripe automatically finalizes your invoice and attempts payment. Note
/// that finalizing the invoice,
/// [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
/// not happen immediately as the invoice is created. Stripe waits
/// until one hour after the last webhook was successfully sent (or the last
/// webhook timed out after failing). If you (and the platforms you may have
/// connected to) have no webhooks configured, Stripe waits one hour after
/// creation to finalize the invoice.
///
/// If your invoice is configured to be billed by sending an email, then based on your
/// [email settings](https://dashboard.stripe.com/account/billing/automatic),
/// Stripe will email the invoice to your customer and await payment. These
/// emails can contain a link to a hosted page to pay the invoice.
///
/// Stripe applies any customer credit on the account before determining the
/// amount due for the invoice (i.e., the amount that will be actually
/// charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
/// per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
/// invoice is automatically marked paid, and we add the amount due to the
/// customer's credit balance which is applied to the next invoice.
///
/// More details on the customer's credit balance are
/// [here](https://stripe.com/docs/billing/customer/balance).
///
/// Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Invoice {
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_country: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub account_name: String,
    /**
    * The account tax IDs associated with the invoice. Only editable when the invoice is a draft.
    */
    #[serde()]
    pub account_tax_ids: Box<Vec<AccountTaxIdsAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_due: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_paid: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_remaining: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub application_fee_amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub attempt_count: i64,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub attempted: bool,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub auto_advance: bool,
    /**
    *
    */
    #[serde()]
    pub automatic_tax: AutomaticTax,
    /**
    * Indicates the reason why the invoice was created. `subscription_cycle` indicates an invoice created by a subscription advancing into a new period. `subscription_create` indicates an invoice created due to creating a subscription. `subscription_update` indicates an invoice created due to updating a subscription. `subscription` is set for all old invoices to indicate either a change to a subscription or a period advancement. `manual` is set for all invoices unrelated to a subscription (for example: created via the invoice editor). The `upcoming` value is reserved for simulated invoices per the upcoming invoice endpoint. `subscription_threshold` indicates an invoice created due to a billing threshold being reached.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub billing_reason: Option<BillingReason>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub charge: Box<Option<ChargeAnyOf>>,
    /**
    * Either `charge_automatically`, or `send_invoice`. When charging automatically, Stripe will attempt to pay this invoice using the default source attached to the customer. When sending an invoice, Stripe will email this invoice to the customer with payment instructions.
    */
    #[serde(default, skip_serializing_if = "CollectionMethod::is_noop")]
    pub collection_method: CollectionMethod,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Custom fields displayed on the invoice.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub custom_fields: Vec<InvoiceSettingCustomField>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * A publicly available mailing address for sending support issues to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_address: Option<Address>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_email: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_name: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub customer_phone: String,
    /**
    * Shipping information for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_shipping: Option<ShippingData>,
    /**
    * Describes the customer's tax exemption status. One of `none`, `exempt`, or `reverse`. When set to `reverse`, invoice and receipt PDFs include the text \*\*"Reverse charge"\*\*.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_tax_exempt: Option<TaxExempt>,
    /**
    * The customer's tax IDs. Until the invoice is finalized, this field will contain the same tax IDs as `customer.tax_ids`. Once the invoice is finalized, this field will no longer be updated.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub customer_tax_ids: Vec<InvoicesResourceInvoiceTaxId>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub default_payment_method: Box<Option<PaymentMethodAnyOf>>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub default_source: Box<Option<DefaultSourceAnyOf>>,
    /**
    * The tax rates which apply to the line item.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub default_tax_rates: Vec<TaxRate>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * Describes the current discount active on the customer, if there is one.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub discount: Option<DiscountData>,
    /**
    * The discounts applied to the invoice. Line item discounts are applied before invoice discounts. Use `expand[]=discounts` to expand each discount.
    */
    #[serde()]
    pub discounts: Box<Vec<DiscountAnyOf>>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub due_date: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub ending_balance: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub footer: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub hosted_invoice_url: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice_pdf: String,
    /**
    * The error encountered during the previous attempt to finalize the invoice. This field is cleared when the invoice is successfully finalized.
    */
    #[serde()]
    pub last_finalization_error: Box<Option<ApiErrors>>,
    /**
    * The individual line items that make up the invoice. `lines` is sorted as follows: invoice items in reverse chronological order, followed by the subscription, if any.
    */
    #[serde()]
    pub lines: InvoiceLinesList,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub next_payment_attempt: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedInvoiceObject::is_noop")]
    pub object: DeletedInvoiceObject,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub on_behalf_of: Box<Option<AccountAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub paid: bool,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub paid_out_of_band: bool,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub payment_intent: Box<Option<PaymentIntentAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub payment_settings: InvoicesPaymentSettings,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub period_end: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub period_start: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub post_payment_credit_notes_amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub pre_payment_credit_notes_amount: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub quote: Box<Option<QuoteAnyOf>>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub receipt_number: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub starting_balance: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor: String,
    /**
    * The status of the invoice, one of `draft`, `open`, `paid`, `uncollectible`, or `void`. [Learn more](https://stripe.com/docs/billing/invoices/workflow#workflow-overview)
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<InvoiceStatus>,
    /**
    *
    */
    #[serde()]
    pub status_transitions: InvoicesStatusTransitions,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub subscription: Box<Option<SubscriptionAnyOf>>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub subscription_proration_date: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub subtotal: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub tax: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde()]
    pub test_clock: Box<Option<TestClockAnyOf>>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub threshold_reason: Option<InvoiceThresholdReason>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total: i64,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub total_discount_amounts: Vec<DiscountsResourceDiscountAmount>,
    /**
    * The aggregate amounts calculated per tax rate for all line items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub total_tax_amounts: Vec<InvoiceTaxAmount>,
    /**
    * The account (if any) the payment will be attributed to for tax reporting, and where funds from the payment will be transferred to for the invoice.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transfer_data: Option<InvoiceTransferDataType>,
    /**
    * Invoices are statements of amounts owed by a customer, and are either
    *  generated one-off, or generated periodically from a subscription.
    *  
    *  They contain [invoice items](https://stripe.com/docs/api#invoiceitems), and proration adjustments
    *  that may be caused by subscription upgrades/downgrades (if necessary).
    *  
    *  If your invoice is configured to be billed through automatic charges,
    *  Stripe automatically finalizes your invoice and attempts payment. Note
    *  that finalizing the invoice,
    *  [when automatic](https://stripe.com/docs/billing/invoices/workflow/#auto_advance), does
    *  not happen immediately as the invoice is created. Stripe waits
    *  until one hour after the last webhook was successfully sent (or the last
    *  webhook timed out after failing). If you (and the platforms you may have
    *  connected to) have no webhooks configured, Stripe waits one hour after
    *  creation to finalize the invoice.
    *  
    *  If your invoice is configured to be billed by sending an email, then based on your
    *  [email settings](https://dashboard.stripe.com/account/billing/automatic),
    *  Stripe will email the invoice to your customer and await payment. These
    *  emails can contain a link to a hosted page to pay the invoice.
    *  
    *  Stripe applies any customer credit on the account before determining the
    *  amount due for the invoice (i.e., the amount that will be actually
    *  charged). If the amount due for the invoice is less than Stripe's [minimum allowed charge
    *  per currency](/docs/currencies#minimum-and-maximum-charge-amounts), the
    *  invoice is automatically marked paid, and we add the amount due to the
    *  customer's credit balance which is applied to the next invoice.
    *  
    *  More details on the customer's credit balance are
    *  [here](https://stripe.com/docs/billing/customer/balance).
    *  
    *  Related guide: [Send Invoices to Customers](https://stripe.com/docs/billing/invoices/sending).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub webhooks_delivered_at: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceItemThresholdReason {
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub line_item_ids: Vec<String>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub usage_gte: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceLineItemPeriod {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub end: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub start: i64,
}

/**
* One of `fixed` or `maximum`. If `fixed`, the `amount` param refers to the exact amount to be charged in future payments. If `maximum`, the amount charged can be up to the value passed for the `amount` param.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AmountType {
    #[serde(rename = "fixed")]
    Fixed,
    #[serde(rename = "maximum")]
    Maximum,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AmountType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AmountType::Fixed => "fixed",
            AmountType::Maximum => "maximum",
            AmountType::Noop => "",
            AmountType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AmountType {
    fn default() -> AmountType {
        AmountType::Noop
    }
}
impl AmountType {
    pub fn is_noop(&self) -> bool {
        matches!(self, AmountType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceMandateOptionsCard {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * One of `fixed` or `maximum`. If `fixed`, the `amount` param refers to the exact amount to be charged in future payments. If `maximum`, the amount charged can be up to the value passed for the `amount` param.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub amount_type: Option<AmountType>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsAcssDebit {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub mandate_options: Option<InvoicePaymentMethodOptionsAcssDebitMandate>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<VerificationMethod>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsAcssDebitMandate {
    /**
    * Transaction type of the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transaction_type: Option<TransactionType>,
}

/**
* Preferred language of the Bancontact authorization page that the customer is redirected to.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PreferredLanguage {
    #[serde(rename = "de")]
    De,
    #[serde(rename = "en")]
    En,
    #[serde(rename = "fr")]
    Fr,
    #[serde(rename = "nl")]
    Nl,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PreferredLanguage {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PreferredLanguage::De => "de",
            PreferredLanguage::En => "en",
            PreferredLanguage::Fr => "fr",
            PreferredLanguage::Nl => "nl",
            PreferredLanguage::Noop => "",
            PreferredLanguage::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PreferredLanguage {
    fn default() -> PreferredLanguage {
        PreferredLanguage::Noop
    }
}
impl PreferredLanguage {
    pub fn is_noop(&self) -> bool {
        matches!(self, PreferredLanguage::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsBancontact {
    /**
    * Preferred language of the Bancontact authorization page that the customer is redirected to.
    */
    #[serde(default, skip_serializing_if = "PreferredLanguage::is_noop")]
    pub preferred_language: PreferredLanguage,
}

/**
* We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum RequestThreeDSecure {
    #[serde(rename = "any")]
    Any,
    #[serde(rename = "automatic")]
    Automatic,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for RequestThreeDSecure {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            RequestThreeDSecure::Any => "any",
            RequestThreeDSecure::Automatic => "automatic",
            RequestThreeDSecure::Noop => "",
            RequestThreeDSecure::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for RequestThreeDSecure {
    fn default() -> RequestThreeDSecure {
        RequestThreeDSecure::Noop
    }
}
impl RequestThreeDSecure {
    pub fn is_noop(&self) -> bool {
        matches!(self, RequestThreeDSecure::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsCard {
    /**
    * We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub request_three_d_secure: Option<RequestThreeDSecure>,
}

/**
* The funding method type to be used when there are not enough funds in the customer balance. Permitted values include: `bank_transfer`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum FundingType {
    #[serde(rename = "bank_transfer")]
    BankTransfer,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for FundingType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            FundingType::BankTransfer => "bank_transfer",
            FundingType::Noop => "",
            FundingType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for FundingType {
    fn default() -> FundingType {
        FundingType::Noop
    }
}
impl FundingType {
    pub fn is_noop(&self) -> bool {
        matches!(self, FundingType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsCustomerBalance {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bank_transfer: Option<InvoicePaymentMethodOptionsCustomerBalanceBankTransfer>,
    /**
    * The funding method type to be used when there are not enough funds in the customer balance. Permitted values include: `bank_transfer`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub funding_type: Option<FundingType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsCustomerBalanceBankTransfer {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicePaymentMethodOptionsUsBankAccount {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<VerificationMethod>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceSettingCustomField {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceSettingCustomer {
    /**
    * Custom fields displayed on the invoice.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub custom_fields: Vec<InvoiceSettingCustomField>,
    /**
    *
    */
    #[serde()]
    pub default_payment_method: Box<Option<PaymentMethodAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub footer: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceSettingQuote {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub days_until_due: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceSettingSubscriptionSchedule {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub days_until_due: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceThresholdReason {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_gte: i64,
    /**
    * Indicates which line items triggered a threshold invoice.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub item_reasons: Vec<InvoiceItemThresholdReason>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceTransferDataType {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    #[serde()]
    pub destination: Box<AccountAnyOf>,
}

/// All of the following types:
///
/// - `String`
/// - `DiscountData`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DiscountsAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    DiscountData(DiscountData),
}

impl DiscountsAnyOf {
    pub fn discount_data(&self) -> Option<&DiscountData> {
        if let DiscountsAnyOf::DiscountData(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let DiscountsAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for DiscountsAnyOf {
    fn from(f: String) -> Self {
        DiscountsAnyOf::String(f)
    }
}

impl std::convert::From<DiscountsAnyOf> for String {
    fn from(f: DiscountsAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// Sometimes you want to add a charge or credit to a customer, but actually
/// charge or credit the customer's card only at the end of a regular billing
/// cycle. This is useful for combining several charges (to minimize
/// per-transaction fees), or for having Stripe tabulate your usage-based billing
/// totals.
///
/// Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoiceItem {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    #[serde()]
    pub customer: Box<CustomerAnyOf>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub date: i64,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub discountable: bool,
    /**
    * The discounts which apply to the invoice item. Item discounts are applied before invoice discounts. Use `expand[]=discounts` to expand each discount.
    */
    #[serde()]
    pub discounts: Box<Vec<DiscountsAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde()]
    pub invoice: Box<Option<InvoiceAnyOf>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "DeletedInvoiceItemObject::is_noop")]
    pub object: DeletedInvoiceItemObject,
    /**
    *
    */
    #[serde()]
    pub period: InvoiceLineItemPeriod,
    /**
    * The price of the invoice item.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub price: Option<PriceData>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub proration: bool,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub quantity: i64,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde()]
    pub subscription: Box<Option<SubscriptionAnyOf>>,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub subscription_item: String,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub tax_rates: Vec<TaxRate>,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde()]
    pub test_clock: Box<Option<TestClockAnyOf>>,
    /**
    * Sometimes you want to add a charge or credit to a customer, but actually
    *  charge or credit the customer's card only at the end of a regular billing
    *  cycle. This is useful for combining several charges (to minimize
    *  per-transaction fees), or for having Stripe tabulate your usage-based billing
    *  totals.
    *  
    *  Related guide: [Subscription Invoices](https://stripe.com/docs/billing/invoices/subscription#adding-upcoming-invoice-items).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub unit_amount: i64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub unit_amount_decimal: f64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicesLineItemsCredited {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice: String,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub invoice_line_items: Vec<String>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicesLineItemsProrationDetails {
    /**
    * For a credit proration `line_item`, the original debit line_items to which the credit proration applies.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub credited_items: Option<InvoicesLineItemsCredited>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicesPaymentMethodOptions {
    /**
    * If paying by `acss_debit`, this sub-hash contains details about the Canadian pre-authorized debit payment method options to pass to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub acss_debit: Option<InvoicePaymentMethodOptionsAcssDebit>,
    /**
    * If paying by `bancontact`, this sub-hash contains details about the Bancontact payment method options to pass to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bancontact: Option<InvoicePaymentMethodOptionsBancontact>,
    /**
    * If paying by `card`, this sub-hash contains details about the Card payment method options to pass to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card: Option<InvoicePaymentMethodOptionsCard>,
    /**
    * If paying by `customer_balance`, this sub-hash contains details about the Bank transfer payment method options to pass to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_balance: Option<InvoicePaymentMethodOptionsCustomerBalance>,
    /**
    * If paying by `konbini`, this sub-hash contains details about the Konbini payment method options to pass to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub konbini: Option<UseStripeSdk>,
    /**
    * If paying by `us_bank_account`, this sub-hash contains details about the ACH direct debit payment method options to pass to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub us_bank_account: Option<InvoicePaymentMethodOptionsUsBankAccount>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentMethodTypes {
    #[serde(rename = "ach_credit_transfer")]
    AchCreditTransfer,
    #[serde(rename = "ach_debit")]
    AchDebit,
    #[serde(rename = "acss_debit")]
    AcssDebit,
    #[serde(rename = "au_becs_debit")]
    AuBecsDebit,
    #[serde(rename = "bacs_debit")]
    BacsDebit,
    #[serde(rename = "bancontact")]
    Bancontact,
    #[serde(rename = "boleto")]
    Boleto,
    #[serde(rename = "card")]
    Card,
    #[serde(rename = "customer_balance")]
    CustomerBalance,
    #[serde(rename = "fpx")]
    Fpx,
    #[serde(rename = "giropay")]
    Giropay,
    #[serde(rename = "grabpay")]
    Grabpay,
    #[serde(rename = "ideal")]
    Ideal,
    #[serde(rename = "konbini")]
    Konbini,
    #[serde(rename = "paynow")]
    Paynow,
    #[serde(rename = "sepa_debit")]
    SepaDebit,
    #[serde(rename = "sofort")]
    Sofort,
    #[serde(rename = "us_bank_account")]
    UsBankAccount,
    #[serde(rename = "wechat_pay")]
    WechatPay,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentMethodTypes {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentMethodTypes::AchCreditTransfer => "ach_credit_transfer",
            PaymentMethodTypes::AchDebit => "ach_debit",
            PaymentMethodTypes::AcssDebit => "acss_debit",
            PaymentMethodTypes::AuBecsDebit => "au_becs_debit",
            PaymentMethodTypes::BacsDebit => "bacs_debit",
            PaymentMethodTypes::Bancontact => "bancontact",
            PaymentMethodTypes::Boleto => "boleto",
            PaymentMethodTypes::Card => "card",
            PaymentMethodTypes::CustomerBalance => "customer_balance",
            PaymentMethodTypes::Fpx => "fpx",
            PaymentMethodTypes::Giropay => "giropay",
            PaymentMethodTypes::Grabpay => "grabpay",
            PaymentMethodTypes::Ideal => "ideal",
            PaymentMethodTypes::Konbini => "konbini",
            PaymentMethodTypes::Paynow => "paynow",
            PaymentMethodTypes::SepaDebit => "sepa_debit",
            PaymentMethodTypes::Sofort => "sofort",
            PaymentMethodTypes::UsBankAccount => "us_bank_account",
            PaymentMethodTypes::WechatPay => "wechat_pay",
            PaymentMethodTypes::Noop => "",
            PaymentMethodTypes::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentMethodTypes {
    fn default() -> PaymentMethodTypes {
        PaymentMethodTypes::Noop
    }
}
impl PaymentMethodTypes {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentMethodTypes::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicesPaymentSettings {
    /**
    * Payment-method-specific configuration to provide to the invoice’s PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_method_options: Option<InvoicesPaymentMethodOptions>,
    /**
    * The list of payment method types (e.g. card) to provide to the invoice’s PaymentIntent. If not set, Stripe attempts to automatically determine the types to use by looking at the invoice’s default payment method, the subscription’s default payment method, the customer’s default payment method, and your [invoice template settings](https://dashboard.stripe.com/settings/billing/invoice).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub payment_method_types: Vec<PaymentMethodTypes>,
}

/**
* The type of the tax ID, one of `eu_vat`, `br_cnpj`, `br_cpf`, `gb_vat`, `nz_gst`, `au_abn`, `au_arn`, `in_gst`, `no_vat`, `za_vat`, `ch_vat`, `mx_rfc`, `sg_uen`, `ru_inn`, `ru_kpp`, `ca_bn`, `hk_br`, `es_cif`, `tw_vat`, `th_vat`, `jp_cn`, `jp_rn`, `li_uid`, `my_itn`, `us_ein`, `kr_brn`, `ca_qst`, `ca_gst_hst`, `ca_pst_bc`, `ca_pst_mb`, `ca_pst_sk`, `my_sst`, `sg_gst`, `ae_trn`, `cl_tin`, `sa_vat`, `id_npwp`, `my_frp`, `il_vat`, `ge_vat`, `ua_vat`, `is_vat`, `bg_uic`, `hu_tin`, `si_tin`, or `unknown`
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum InvoicesResourceInvoiceTaxType {
    #[serde(rename = "ae_trn")]
    AeTrn,
    #[serde(rename = "au_abn")]
    AuAbn,
    #[serde(rename = "au_arn")]
    AuArn,
    #[serde(rename = "bg_uic")]
    BgUic,
    #[serde(rename = "br_cnpj")]
    BrCnpj,
    #[serde(rename = "br_cpf")]
    BrCpf,
    #[serde(rename = "ca_bn")]
    CaBn,
    #[serde(rename = "ca_gst_hst")]
    CaGstHst,
    #[serde(rename = "ca_pst_bc")]
    CaPstBc,
    #[serde(rename = "ca_pst_mb")]
    CaPstMb,
    #[serde(rename = "ca_pst_sk")]
    CaPstSk,
    #[serde(rename = "ca_qst")]
    CaQst,
    #[serde(rename = "ch_vat")]
    ChVat,
    #[serde(rename = "cl_tin")]
    ClTin,
    #[serde(rename = "es_cif")]
    EsCif,
    #[serde(rename = "eu_vat")]
    EuVat,
    #[serde(rename = "gb_vat")]
    GbVat,
    #[serde(rename = "ge_vat")]
    GeVat,
    #[serde(rename = "hk_br")]
    HkBr,
    #[serde(rename = "hu_tin")]
    HuTin,
    #[serde(rename = "id_npwp")]
    IdNpwp,
    #[serde(rename = "il_vat")]
    IlVat,
    #[serde(rename = "in_gst")]
    InGst,
    #[serde(rename = "is_vat")]
    IsVat,
    #[serde(rename = "jp_cn")]
    JpCn,
    #[serde(rename = "jp_rn")]
    JpRn,
    #[serde(rename = "kr_brn")]
    KrBrn,
    #[serde(rename = "li_uid")]
    LiUid,
    #[serde(rename = "mx_rfc")]
    MxRfc,
    #[serde(rename = "my_frp")]
    MyFrp,
    #[serde(rename = "my_itn")]
    MyItn,
    #[serde(rename = "my_sst")]
    MySst,
    #[serde(rename = "no_vat")]
    NoVat,
    #[serde(rename = "nz_gst")]
    NzGst,
    #[serde(rename = "ru_inn")]
    RuInn,
    #[serde(rename = "ru_kpp")]
    RuKpp,
    #[serde(rename = "sa_vat")]
    SaVat,
    #[serde(rename = "sg_gst")]
    SgGst,
    #[serde(rename = "sg_uen")]
    SgUen,
    #[serde(rename = "si_tin")]
    SiTin,
    #[serde(rename = "th_vat")]
    ThVat,
    #[serde(rename = "tw_vat")]
    TwVat,
    #[serde(rename = "ua_vat")]
    UaVat,
    #[serde(rename = "unknown")]
    Unknown,
    #[serde(rename = "us_ein")]
    UsEin,
    #[serde(rename = "za_vat")]
    ZaVat,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for InvoicesResourceInvoiceTaxType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            InvoicesResourceInvoiceTaxType::AeTrn => "ae_trn",
            InvoicesResourceInvoiceTaxType::AuAbn => "au_abn",
            InvoicesResourceInvoiceTaxType::AuArn => "au_arn",
            InvoicesResourceInvoiceTaxType::BgUic => "bg_uic",
            InvoicesResourceInvoiceTaxType::BrCnpj => "br_cnpj",
            InvoicesResourceInvoiceTaxType::BrCpf => "br_cpf",
            InvoicesResourceInvoiceTaxType::CaBn => "ca_bn",
            InvoicesResourceInvoiceTaxType::CaGstHst => "ca_gst_hst",
            InvoicesResourceInvoiceTaxType::CaPstBc => "ca_pst_bc",
            InvoicesResourceInvoiceTaxType::CaPstMb => "ca_pst_mb",
            InvoicesResourceInvoiceTaxType::CaPstSk => "ca_pst_sk",
            InvoicesResourceInvoiceTaxType::CaQst => "ca_qst",
            InvoicesResourceInvoiceTaxType::ChVat => "ch_vat",
            InvoicesResourceInvoiceTaxType::ClTin => "cl_tin",
            InvoicesResourceInvoiceTaxType::EsCif => "es_cif",
            InvoicesResourceInvoiceTaxType::EuVat => "eu_vat",
            InvoicesResourceInvoiceTaxType::GbVat => "gb_vat",
            InvoicesResourceInvoiceTaxType::GeVat => "ge_vat",
            InvoicesResourceInvoiceTaxType::HkBr => "hk_br",
            InvoicesResourceInvoiceTaxType::HuTin => "hu_tin",
            InvoicesResourceInvoiceTaxType::IdNpwp => "id_npwp",
            InvoicesResourceInvoiceTaxType::IlVat => "il_vat",
            InvoicesResourceInvoiceTaxType::InGst => "in_gst",
            InvoicesResourceInvoiceTaxType::IsVat => "is_vat",
            InvoicesResourceInvoiceTaxType::JpCn => "jp_cn",
            InvoicesResourceInvoiceTaxType::JpRn => "jp_rn",
            InvoicesResourceInvoiceTaxType::KrBrn => "kr_brn",
            InvoicesResourceInvoiceTaxType::LiUid => "li_uid",
            InvoicesResourceInvoiceTaxType::MxRfc => "mx_rfc",
            InvoicesResourceInvoiceTaxType::MyFrp => "my_frp",
            InvoicesResourceInvoiceTaxType::MyItn => "my_itn",
            InvoicesResourceInvoiceTaxType::MySst => "my_sst",
            InvoicesResourceInvoiceTaxType::NoVat => "no_vat",
            InvoicesResourceInvoiceTaxType::NzGst => "nz_gst",
            InvoicesResourceInvoiceTaxType::RuInn => "ru_inn",
            InvoicesResourceInvoiceTaxType::RuKpp => "ru_kpp",
            InvoicesResourceInvoiceTaxType::SaVat => "sa_vat",
            InvoicesResourceInvoiceTaxType::SgGst => "sg_gst",
            InvoicesResourceInvoiceTaxType::SgUen => "sg_uen",
            InvoicesResourceInvoiceTaxType::SiTin => "si_tin",
            InvoicesResourceInvoiceTaxType::ThVat => "th_vat",
            InvoicesResourceInvoiceTaxType::TwVat => "tw_vat",
            InvoicesResourceInvoiceTaxType::UaVat => "ua_vat",
            InvoicesResourceInvoiceTaxType::Unknown => "unknown",
            InvoicesResourceInvoiceTaxType::UsEin => "us_ein",
            InvoicesResourceInvoiceTaxType::ZaVat => "za_vat",
            InvoicesResourceInvoiceTaxType::Noop => "",
            InvoicesResourceInvoiceTaxType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for InvoicesResourceInvoiceTaxType {
    fn default() -> InvoicesResourceInvoiceTaxType {
        InvoicesResourceInvoiceTaxType::Noop
    }
}
impl InvoicesResourceInvoiceTaxType {
    pub fn is_noop(&self) -> bool {
        matches!(self, InvoicesResourceInvoiceTaxType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicesResourceInvoiceTaxId {
    /**
    * The type of the tax ID, one of `eu_vat`, `br_cnpj`, `br_cpf`, `gb_vat`, `nz_gst`, `au_abn`, `au_arn`, `in_gst`, `no_vat`, `za_vat`, `ch_vat`, `mx_rfc`, `sg_uen`, `ru_inn`, `ru_kpp`, `ca_bn`, `hk_br`, `es_cif`, `tw_vat`, `th_vat`, `jp_cn`, `jp_rn`, `li_uid`, `my_itn`, `us_ein`, `kr_brn`, `ca_qst`, `ca_gst_hst`, `ca_pst_bc`, `ca_pst_mb`, `ca_pst_sk`, `my_sst`, `sg_gst`, `ae_trn`, `cl_tin`, `sa_vat`, `id_npwp`, `my_frp`, `il_vat`, `ge_vat`, `ua_vat`, `is_vat`, `bg_uic`, `hu_tin`, `si_tin`, or `unknown`
    */
    #[serde(
        default,
        skip_serializing_if = "InvoicesResourceInvoiceTaxType::is_noop",
        rename = "type"
    )]
    pub type_: InvoicesResourceInvoiceTaxType,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub value: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct InvoicesStatusTransitions {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub finalized_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub marked_uncollectible_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub paid_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub voided_at: i64,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuerFraudRecordObject {
    #[serde(rename = "issuer_fraud_record")]
    IssuerFraudRecord,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuerFraudRecordObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuerFraudRecordObject::IssuerFraudRecord => "issuer_fraud_record",
            IssuerFraudRecordObject::Noop => "",
            IssuerFraudRecordObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuerFraudRecordObject {
    fn default() -> IssuerFraudRecordObject {
        IssuerFraudRecordObject::Noop
    }
}
impl IssuerFraudRecordObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuerFraudRecordObject::Noop)
    }
}

/// This resource has been renamed to [Early Fraud
/// Warning](#early_fraud_warning_object) and will be removed in a future API
/// version.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuerFraudRecord {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub actionable: bool,
    #[serde()]
    pub charge: Box<ChargeAnyOf>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fraud_type: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_liability_shift: bool,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuerFraudRecordObject::is_noop")]
    pub object: IssuerFraudRecordObject,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub post_date: i64,
}

/**
* How the card details were provided.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AuthorizationMethod {
    #[serde(rename = "chip")]
    Chip,
    #[serde(rename = "contactless")]
    Contactless,
    #[serde(rename = "keyed_in")]
    KeyedIn,
    #[serde(rename = "online")]
    Online,
    #[serde(rename = "swipe")]
    Swipe,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AuthorizationMethod {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AuthorizationMethod::Chip => "chip",
            AuthorizationMethod::Contactless => "contactless",
            AuthorizationMethod::KeyedIn => "keyed_in",
            AuthorizationMethod::Online => "online",
            AuthorizationMethod::Swipe => "swipe",
            AuthorizationMethod::Noop => "",
            AuthorizationMethod::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AuthorizationMethod {
    fn default() -> AuthorizationMethod {
        AuthorizationMethod::Noop
    }
}
impl AuthorizationMethod {
    pub fn is_noop(&self) -> bool {
        matches!(self, AuthorizationMethod::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `IssuingCardholder`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CardholderAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * An Issuing `Cardholder` object represents an individual or business entity who is [issued](https://stripe.com/docs/issuing) cards.
    *  
    *  Related guide: [How to create a Cardholder](https://stripe.com/docs/issuing/cards#create-cardholder)
    */
    IssuingCardholder(IssuingCardholder),
}

impl CardholderAnyOf {
    pub fn issuing_cardholder(&self) -> Option<&IssuingCardholder> {
        if let CardholderAnyOf::IssuingCardholder(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let CardholderAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for CardholderAnyOf {
    fn from(f: String) -> Self {
        CardholderAnyOf::String(f)
    }
}

impl std::convert::From<CardholderAnyOf> for String {
    fn from(f: CardholderAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingAuthorizationObject {
    #[serde(rename = "issuing.authorization")]
    IssuingAuthorization,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingAuthorizationObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingAuthorizationObject::IssuingAuthorization => "issuing.authorization",
            IssuingAuthorizationObject::Noop => "",
            IssuingAuthorizationObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingAuthorizationObject {
    fn default() -> IssuingAuthorizationObject {
        IssuingAuthorizationObject::Noop
    }
}
impl IssuingAuthorizationObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingAuthorizationObject::Noop)
    }
}

/**
* The current status of the authorization in its lifecycle.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingAuthorizationStatus {
    #[serde(rename = "closed")]
    Closed,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "reversed")]
    Reversed,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingAuthorizationStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingAuthorizationStatus::Closed => "closed",
            IssuingAuthorizationStatus::Pending => "pending",
            IssuingAuthorizationStatus::Reversed => "reversed",
            IssuingAuthorizationStatus::Noop => "",
            IssuingAuthorizationStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingAuthorizationStatus {
    fn default() -> IssuingAuthorizationStatus {
        IssuingAuthorizationStatus::Noop
    }
}
impl IssuingAuthorizationStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingAuthorizationStatus::Noop)
    }
}

/// When an [issued card](https://stripe.com/docs/issuing) is used to make a purchase, an Issuing `Authorization`
/// object is created. [Authorizations](https://stripe.com/docs/issuing/purchases/authorizations) must be approved for the
/// purchase to be completed successfully.
///
/// Related guide: [Issued Card Authorizations](https://stripe.com/docs/issuing/purchases/authorizations).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingAuthorization {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Detailed breakdown of amount components. These amounts are denominated in `currency` and in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub amount_details: Option<IssuingAuthorizationAmountDetails>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub approved: bool,
    /**
    * How the card details were provided.
    */
    #[serde(default, skip_serializing_if = "AuthorizationMethod::is_noop")]
    pub authorization_method: AuthorizationMethod,
    /**
    * List of zero, one, or two balance transactions that show funds withdrawn and reinstated to your Stripe account as a result of this dispute.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub balance_transactions: Vec<BalanceTransaction>,
    /**
    * You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
    */
    #[serde()]
    pub card: IssuingCard,
    /**
    * When an [issued card](https://stripe.com/docs/issuing) is used to make a purchase, an Issuing `Authorization`
    *  object is created. [Authorizations](https://stripe.com/docs/issuing/purchases/authorizations) must be approved for the
    *  purchase to be completed successfully.
    *  
    *  Related guide: [Issued Card Authorizations](https://stripe.com/docs/issuing/purchases/authorizations).
    */
    #[serde()]
    pub cardholder: Box<Option<CardholderAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub merchant_amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub merchant_currency: String,
    /**
    *
    */
    #[serde()]
    pub merchant_data: IssuingAuthorizationMerchantData,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuingAuthorizationObject::is_noop")]
    pub object: IssuingAuthorizationObject,
    /**
    * The pending authorization request. This field will only be non-null during an `issuing_authorization.request` webhook.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub pending_request: Option<IssuingAuthorizationPendingRequest>,
    /**
    * History of every time `pending_request` was approved/denied, either by you directly or by Stripe (e.g. based on your `spending_controls`). If the merchant changes the authorization by performing an [incremental authorization](https://stripe.com/docs/issuing/purchases/authorizations), you can look at this field to see the previous requests for the authorization.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub request_history: Vec<IssuingAuthorizationRequest>,
    /**
    * The current status of the authorization in its lifecycle.
    */
    #[serde(default, skip_serializing_if = "IssuingAuthorizationStatus::is_noop")]
    pub status: IssuingAuthorizationStatus,
    /**
    * List of [transactions](https://stripe.com/docs/api/issuing/transactions) associated with this authorization.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub transactions: Vec<IssuingTransaction>,
    /**
    *
    */
    #[serde()]
    pub verification_data: IssuingAuthorizationVerificationData,
    /**
    * When an [issued card](https://stripe.com/docs/issuing) is used to make a purchase, an Issuing `Authorization`
    *  object is created. [Authorizations](https://stripe.com/docs/issuing/purchases/authorizations) must be approved for the
    *  purchase to be completed successfully.
    *  
    *  Related guide: [Issued Card Authorizations](https://stripe.com/docs/issuing/purchases/authorizations).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub wallet: String,
}

/**
* The reason why the card was canceled.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CancellationReason {
    #[serde(rename = "lost")]
    Lost,
    #[serde(rename = "stolen")]
    Stolen,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CancellationReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CancellationReason::Lost => "lost",
            CancellationReason::Stolen => "stolen",
            CancellationReason::Noop => "",
            CancellationReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CancellationReason {
    fn default() -> CancellationReason {
        CancellationReason::Noop
    }
}
impl CancellationReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, CancellationReason::Noop)
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardObject {
    #[serde(rename = "issuing.card")]
    IssuingCard,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardObject::IssuingCard => "issuing.card",
            IssuingCardObject::Noop => "",
            IssuingCardObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardObject {
    fn default() -> IssuingCardObject {
        IssuingCardObject::Noop
    }
}
impl IssuingCardObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `IssuingCard`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CardAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
    */
    IssuingCard(IssuingCard),
}

impl CardAnyOf {
    pub fn issuing_card(&self) -> Option<&IssuingCard> {
        if let CardAnyOf::IssuingCard(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let CardAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for CardAnyOf {
    fn from(f: String) -> Self {
        CardAnyOf::String(f)
    }
}

impl std::convert::From<CardAnyOf> for String {
    fn from(f: CardAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* The reason why the previous card needed to be replaced.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ReplacementReason {
    #[serde(rename = "damaged")]
    Damaged,
    #[serde(rename = "expired")]
    Expired,
    #[serde(rename = "lost")]
    Lost,
    #[serde(rename = "stolen")]
    Stolen,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ReplacementReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ReplacementReason::Damaged => "damaged",
            ReplacementReason::Expired => "expired",
            ReplacementReason::Lost => "lost",
            ReplacementReason::Stolen => "stolen",
            ReplacementReason::Noop => "",
            ReplacementReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ReplacementReason {
    fn default() -> ReplacementReason {
        ReplacementReason::Noop
    }
}
impl ReplacementReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, ReplacementReason::Noop)
    }
}

/**
* Whether authorizations can be approved on this card.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardStatus {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "canceled")]
    Canceled,
    #[serde(rename = "inactive")]
    Inactive,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardStatus::Active => "active",
            IssuingCardStatus::Canceled => "canceled",
            IssuingCardStatus::Inactive => "inactive",
            IssuingCardStatus::Noop => "",
            IssuingCardStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardStatus {
    fn default() -> IssuingCardStatus {
        IssuingCardStatus::Noop
    }
}
impl IssuingCardStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardStatus::Noop)
    }
}

/**
* The type of the card.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardType {
    #[serde(rename = "physical")]
    Physical,
    #[serde(rename = "virtual")]
    Virtual,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardType::Physical => "physical",
            IssuingCardType::Virtual => "virtual",
            IssuingCardType::Noop => "",
            IssuingCardType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardType {
    fn default() -> IssuingCardType {
        IssuingCardType::Noop
    }
}
impl IssuingCardType {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardType::Noop)
    }
}

/// You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCard {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub brand: String,
    /**
    * The reason why the card was canceled.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub cancellation_reason: Option<CancellationReason>,
    /**
    * An Issuing `Cardholder` object represents an individual or business entity who is [issued](https://stripe.com/docs/issuing) cards.
    *  
    *  Related guide: [How to create a Cardholder](https://stripe.com/docs/issuing/cards#create-cardholder)
    */
    #[serde()]
    pub cardholder: IssuingCardholder,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cvc: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub exp_month: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub exp_year: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuingCardObject::is_noop")]
    pub object: IssuingCardObject,
    /**
    * You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
    */
    #[serde()]
    pub replaced_by: Box<Option<CardAnyOf>>,
    /**
    * You can [create physical or virtual cards](https://stripe.com/docs/issuing/cards) that are issued to cardholders.
    */
    #[serde()]
    pub replacement_for: Box<Option<CardAnyOf>>,
    /**
    * The reason why the previous card needed to be replaced.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub replacement_reason: Option<ReplacementReason>,
    /**
    * Where and how the card will be shipped.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping: Option<IssuingCardShippingData>,
    /**
    *
    */
    #[serde()]
    pub spending_controls: IssuingCardAuthorizationControls,
    /**
    * Whether authorizations can be approved on this card.
    */
    #[serde(default, skip_serializing_if = "IssuingCardStatus::is_noop")]
    pub status: IssuingCardStatus,
    /**
    * The type of the card.
    */
    #[serde(
        default,
        skip_serializing_if = "IssuingCardType::is_noop",
        rename = "type"
    )]
    pub type_: IssuingCardType,
    /**
    * Information relating to digital wallets (like Apple Pay and Google Pay).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wallets: Option<IssuingCardWallets>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardholderObject {
    #[serde(rename = "issuing.cardholder")]
    IssuingCardholder,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardholderObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardholderObject::IssuingCardholder => "issuing.cardholder",
            IssuingCardholderObject::Noop => "",
            IssuingCardholderObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardholderObject {
    fn default() -> IssuingCardholderObject {
        IssuingCardholderObject::Noop
    }
}
impl IssuingCardholderObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardholderObject::Noop)
    }
}

/**
* Specifies whether to permit authorizations on this cardholder's cards.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardholderStatus {
    #[serde(rename = "active")]
    Active,
    #[serde(rename = "blocked")]
    Blocked,
    #[serde(rename = "inactive")]
    Inactive,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardholderStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardholderStatus::Active => "active",
            IssuingCardholderStatus::Blocked => "blocked",
            IssuingCardholderStatus::Inactive => "inactive",
            IssuingCardholderStatus::Noop => "",
            IssuingCardholderStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardholderStatus {
    fn default() -> IssuingCardholderStatus {
        IssuingCardholderStatus::Noop
    }
}
impl IssuingCardholderStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardholderStatus::Noop)
    }
}

/**
* Type of entity that holds the account. This can be either `individual` or `company`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AccountHolderType {
    #[serde(rename = "company")]
    Company,
    #[serde(rename = "individual")]
    Individual,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AccountHolderType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AccountHolderType::Company => "company",
            AccountHolderType::Individual => "individual",
            AccountHolderType::Noop => "",
            AccountHolderType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AccountHolderType {
    fn default() -> AccountHolderType {
        AccountHolderType::Noop
    }
}
impl AccountHolderType {
    pub fn is_noop(&self) -> bool {
        matches!(self, AccountHolderType::Noop)
    }
}

/// An Issuing `Cardholder` object represents an individual or business entity who is [issued](https://stripe.com/docs/issuing) cards.
///
/// Related guide: [How to create a Cardholder](https://stripe.com/docs/issuing/cards#create-cardholder)
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholder {
    /**
    *
    */
    #[serde()]
    pub billing: IssuingCardholderAddress,
    /**
    * Additional information about a `company` cardholder.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub company: Option<IssuingCardholderCompany>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * An Issuing `Cardholder` object represents an individual or business entity who is [issued](https://stripe.com/docs/issuing) cards.
    *  
    *  Related guide: [How to create a Cardholder](https://stripe.com/docs/issuing/cards#create-cardholder)
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Additional information about an `individual` cardholder.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub individual: Option<IssuingCardholderIndividual>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuingCardholderObject::is_noop")]
    pub object: IssuingCardholderObject,
    /**
    * An Issuing `Cardholder` object represents an individual or business entity who is [issued](https://stripe.com/docs/issuing) cards.
    *  
    *  Related guide: [How to create a Cardholder](https://stripe.com/docs/issuing/cards#create-cardholder)
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone_number: String,
    /**
    *
    */
    #[serde()]
    pub requirements: IssuingCardholderRequirements,
    /**
    * Rules that control spending across this cardholder's cards. Refer to our [documentation](https://stripe.com/docs/issuing/controls/spending-controls) for more details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub spending_controls: Option<IssuingCardholderAuthorizationControls>,
    /**
    * Specifies whether to permit authorizations on this cardholder's cards.
    */
    #[serde(default, skip_serializing_if = "IssuingCardholderStatus::is_noop")]
    pub status: IssuingCardholderStatus,
    /**
    * Type of entity that holds the account. This can be either `individual` or `company`.
    */
    #[serde(
        default,
        skip_serializing_if = "AccountHolderType::is_noop",
        rename = "type"
    )]
    pub type_: AccountHolderType,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingDisputeObject {
    #[serde(rename = "issuing.dispute")]
    IssuingDispute,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingDisputeObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingDisputeObject::IssuingDispute => "issuing.dispute",
            IssuingDisputeObject::Noop => "",
            IssuingDisputeObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingDisputeObject {
    fn default() -> IssuingDisputeObject {
        IssuingDisputeObject::Noop
    }
}
impl IssuingDisputeObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingDisputeObject::Noop)
    }
}

/**
* Current status of the dispute.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingDisputeStatus {
    #[serde(rename = "expired")]
    Expired,
    #[serde(rename = "lost")]
    Lost,
    #[serde(rename = "submitted")]
    Submitted,
    #[serde(rename = "unsubmitted")]
    Unsubmitted,
    #[serde(rename = "won")]
    Won,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingDisputeStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingDisputeStatus::Expired => "expired",
            IssuingDisputeStatus::Lost => "lost",
            IssuingDisputeStatus::Submitted => "submitted",
            IssuingDisputeStatus::Unsubmitted => "unsubmitted",
            IssuingDisputeStatus::Won => "won",
            IssuingDisputeStatus::Noop => "",
            IssuingDisputeStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingDisputeStatus {
    fn default() -> IssuingDisputeStatus {
        IssuingDisputeStatus::Noop
    }
}
impl IssuingDisputeStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingDisputeStatus::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `IssuingTransaction`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum TransactionAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
    *  your Stripe account, such as a completed purchase or refund, is represented by an Issuing
    *  `Transaction` object.
    *  
    *  Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
    */
    IssuingTransaction(IssuingTransaction),
}

impl TransactionAnyOf {
    pub fn issuing_transaction(&self) -> Option<&IssuingTransaction> {
        if let TransactionAnyOf::IssuingTransaction(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let TransactionAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for TransactionAnyOf {
    fn from(f: String) -> Self {
        TransactionAnyOf::String(f)
    }
}

impl std::convert::From<TransactionAnyOf> for String {
    fn from(f: TransactionAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// As a [card issuer](https://stripe.com/docs/issuing), you can dispute transactions that the cardholder does not recognize, suspects to be fraudulent, or has other issues with.
///
/// Related guide: [Disputing Transactions](https://stripe.com/docs/issuing/purchases/disputes)
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDispute {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * As a [card issuer](https://stripe.com/docs/issuing), you can dispute transactions that the cardholder does not recognize, suspects to be fraudulent, or has other issues with.
    *  
    *  Related guide: [Disputing Transactions](https://stripe.com/docs/issuing/purchases/disputes)
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub balance_transactions: Vec<BalanceTransaction>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde()]
    pub evidence: IssuingDisputeEvidence,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuingDisputeObject::is_noop")]
    pub object: IssuingDisputeObject,
    /**
    * Current status of the dispute.
    */
    #[serde(default, skip_serializing_if = "IssuingDisputeStatus::is_noop")]
    pub status: IssuingDisputeStatus,
    #[serde()]
    pub transaction: Box<TransactionAnyOf>,
}

/**
* The card network for this settlement report. One of ["visa"]
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Network {
    #[serde(rename = "visa")]
    Visa,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Network {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Network::Visa => "visa",
            Network::Noop => "",
            Network::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Network {
    fn default() -> Network {
        Network::Noop
    }
}
impl Network {
    pub fn is_noop(&self) -> bool {
        matches!(self, Network::Noop)
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingSettlementObject {
    #[serde(rename = "issuing.settlement")]
    IssuingSettlement,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingSettlementObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingSettlementObject::IssuingSettlement => "issuing.settlement",
            IssuingSettlementObject::Noop => "",
            IssuingSettlementObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingSettlementObject {
    fn default() -> IssuingSettlementObject {
        IssuingSettlementObject::Noop
    }
}
impl IssuingSettlementObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingSettlementObject::Noop)
    }
}

/// When a non-stripe BIN is used, any use of an [issued card](https://stripe.com/docs/issuing) must be settled directly with the card network. The net amount owed is represented by an Issuing `Settlement` object.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingSettlement {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bin: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub clearing_date: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub interchange_fees: i64,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub net_total: i64,
    /**
    * The card network for this settlement report. One of ["visa"]
    */
    #[serde(default, skip_serializing_if = "Network::is_noop")]
    pub network: Network,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub network_fees: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub network_settlement_identifier: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuingSettlementObject::is_noop")]
    pub object: IssuingSettlementObject,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub settlement_service: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub transaction_count: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub transaction_volume: i64,
}

/// All of the following types:
///
/// - `String`
/// - `IssuingAuthorization`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AuthorizationAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * When an [issued card](https://stripe.com/docs/issuing) is used to make a purchase, an Issuing `Authorization`
    *  object is created. [Authorizations](https://stripe.com/docs/issuing/purchases/authorizations) must be approved for the
    *  purchase to be completed successfully.
    *  
    *  Related guide: [Issued Card Authorizations](https://stripe.com/docs/issuing/purchases/authorizations).
    */
    IssuingAuthorization(IssuingAuthorization),
}

impl AuthorizationAnyOf {
    pub fn issuing_authorization(&self) -> Option<&IssuingAuthorization> {
        if let AuthorizationAnyOf::IssuingAuthorization(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let AuthorizationAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for AuthorizationAnyOf {
    fn from(f: String) -> Self {
        AuthorizationAnyOf::String(f)
    }
}

impl std::convert::From<AuthorizationAnyOf> for String {
    fn from(f: AuthorizationAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// All of the following types:
///
/// - `String`
/// - `IssuingDispute`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum DisputeAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * As a [card issuer](https://stripe.com/docs/issuing), you can dispute transactions that the cardholder does not recognize, suspects to be fraudulent, or has other issues with.
    *  
    *  Related guide: [Disputing Transactions](https://stripe.com/docs/issuing/purchases/disputes)
    */
    IssuingDispute(IssuingDispute),
}

impl DisputeAnyOf {
    pub fn issuing_dispute(&self) -> Option<&IssuingDispute> {
        if let DisputeAnyOf::IssuingDispute(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let DisputeAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for DisputeAnyOf {
    fn from(f: String) -> Self {
        DisputeAnyOf::String(f)
    }
}

impl std::convert::From<DisputeAnyOf> for String {
    fn from(f: DisputeAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingTransactionObject {
    #[serde(rename = "issuing.transaction")]
    IssuingTransaction,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingTransactionObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingTransactionObject::IssuingTransaction => "issuing.transaction",
            IssuingTransactionObject::Noop => "",
            IssuingTransactionObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingTransactionObject {
    fn default() -> IssuingTransactionObject {
        IssuingTransactionObject::Noop
    }
}
impl IssuingTransactionObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingTransactionObject::Noop)
    }
}

/**
* The nature of the transaction.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingTransactionType {
    #[serde(rename = "capture")]
    Capture,
    #[serde(rename = "refund")]
    Refund,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingTransactionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingTransactionType::Capture => "capture",
            IssuingTransactionType::Refund => "refund",
            IssuingTransactionType::Noop => "",
            IssuingTransactionType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingTransactionType {
    fn default() -> IssuingTransactionType {
        IssuingTransactionType::Noop
    }
}
impl IssuingTransactionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingTransactionType::Noop)
    }
}

/**
* The digital wallet used for this transaction. One of `apple_pay`, `google_pay`, or `samsung_pay`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Wallet {
    #[serde(rename = "apple_pay")]
    ApplePay,
    #[serde(rename = "google_pay")]
    GooglePay,
    #[serde(rename = "samsung_pay")]
    SamsungPay,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Wallet {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Wallet::ApplePay => "apple_pay",
            Wallet::GooglePay => "google_pay",
            Wallet::SamsungPay => "samsung_pay",
            Wallet::Noop => "",
            Wallet::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Wallet {
    fn default() -> Wallet {
        Wallet::Noop
    }
}
impl Wallet {
    pub fn is_noop(&self) -> bool {
        matches!(self, Wallet::Noop)
    }
}

/// Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
/// your Stripe account, such as a completed purchase or refund, is represented by an Issuing
/// `Transaction` object.
///
/// Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransaction {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Detailed breakdown of amount components. These amounts are denominated in `currency` and in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub amount_details: Option<IssuingTransactionAmountDetailsData>,
    /**
    * Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
    *  your Stripe account, such as a completed purchase or refund, is represented by an Issuing
    *  `Transaction` object.
    *  
    *  Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
    */
    #[serde()]
    pub authorization: Box<Option<AuthorizationAnyOf>>,
    /**
    * Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
    *  your Stripe account, such as a completed purchase or refund, is represented by an Issuing
    *  `Transaction` object.
    *  
    *  Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
    */
    #[serde()]
    pub balance_transaction: Box<Option<BalanceTransactionAnyOf>>,
    #[serde()]
    pub card: Box<CardAnyOf>,
    /**
    * Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
    *  your Stripe account, such as a completed purchase or refund, is represented by an Issuing
    *  `Transaction` object.
    *  
    *  Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
    */
    #[serde()]
    pub cardholder: Box<Option<CardholderAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Any use of an [issued card](https://stripe.com/docs/issuing) that results in funds entering or leaving
    *  your Stripe account, such as a completed purchase or refund, is represented by an Issuing
    *  `Transaction` object.
    *  
    *  Related guide: [Issued Card Transactions](https://stripe.com/docs/issuing/purchases/transactions).
    */
    #[serde()]
    pub dispute: Box<Option<DisputeAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub merchant_amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub merchant_currency: String,
    /**
    *
    */
    #[serde()]
    pub merchant_data: IssuingAuthorizationMerchantData,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "IssuingTransactionObject::is_noop")]
    pub object: IssuingTransactionObject,
    /**
    * Additional purchase information that is optionally provided by the merchant.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub purchase_details: Option<IssuingTransactionPurchaseDetails>,
    /**
    * The nature of the transaction.
    */
    #[serde(
        default,
        skip_serializing_if = "IssuingTransactionType::is_noop",
        rename = "type"
    )]
    pub type_: IssuingTransactionType,
    /**
    * The digital wallet used for this transaction. One of `apple_pay`, `google_pay`, or `samsung_pay`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wallet: Option<Wallet>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingAuthorizationAmountDetails {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub atm_fee: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingAuthorizationMerchantData {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub category: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub category_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub network_id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub postal_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingAuthorizationPendingRequest {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Detailed breakdown of amount components. These amounts are denominated in `currency` and in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub amount_details: Option<IssuingAuthorizationAmountDetails>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub is_amount_controllable: Option<bool>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub merchant_amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub merchant_currency: String,
}

/**
* The reason for the approval or decline.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingAuthorizationRequestReason {
    #[serde(rename = "account_disabled")]
    AccountDisabled,
    #[serde(rename = "card_active")]
    CardActive,
    #[serde(rename = "card_inactive")]
    CardInactive,
    #[serde(rename = "cardholder_inactive")]
    CardholderInactive,
    #[serde(rename = "cardholder_verification_required")]
    CardholderVerificationRequired,
    #[serde(rename = "insufficient_funds")]
    InsufficientFunds,
    #[serde(rename = "not_allowed")]
    NotAllowed,
    #[serde(rename = "spending_controls")]
    SpendingControls,
    #[serde(rename = "suspected_fraud")]
    SuspectedFraud,
    #[serde(rename = "verification_failed")]
    VerificationFailed,
    #[serde(rename = "webhook_approved")]
    WebhookApproved,
    #[serde(rename = "webhook_declined")]
    WebhookDeclined,
    #[serde(rename = "webhook_timeout")]
    WebhookTimeout,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingAuthorizationRequestReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingAuthorizationRequestReason::AccountDisabled => "account_disabled",
            IssuingAuthorizationRequestReason::CardActive => "card_active",
            IssuingAuthorizationRequestReason::CardInactive => "card_inactive",
            IssuingAuthorizationRequestReason::CardholderInactive => "cardholder_inactive",
            IssuingAuthorizationRequestReason::CardholderVerificationRequired => {
                "cardholder_verification_required"
            }
            IssuingAuthorizationRequestReason::InsufficientFunds => "insufficient_funds",
            IssuingAuthorizationRequestReason::NotAllowed => "not_allowed",
            IssuingAuthorizationRequestReason::SpendingControls => "spending_controls",
            IssuingAuthorizationRequestReason::SuspectedFraud => "suspected_fraud",
            IssuingAuthorizationRequestReason::VerificationFailed => "verification_failed",
            IssuingAuthorizationRequestReason::WebhookApproved => "webhook_approved",
            IssuingAuthorizationRequestReason::WebhookDeclined => "webhook_declined",
            IssuingAuthorizationRequestReason::WebhookTimeout => "webhook_timeout",
            IssuingAuthorizationRequestReason::Noop => "",
            IssuingAuthorizationRequestReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingAuthorizationRequestReason {
    fn default() -> IssuingAuthorizationRequestReason {
        IssuingAuthorizationRequestReason::Noop
    }
}
impl IssuingAuthorizationRequestReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingAuthorizationRequestReason::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingAuthorizationRequest {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Detailed breakdown of amount components. These amounts are denominated in `currency` and in the [smallest currency unit](https://stripe.com/docs/currencies#zero-decimal).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub amount_details: Option<IssuingAuthorizationAmountDetails>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub approved: Option<bool>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub merchant_amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub merchant_currency: String,
    /**
    * The reason for the approval or decline.
    */
    #[serde(
        default,
        skip_serializing_if = "IssuingAuthorizationRequestReason::is_noop"
    )]
    pub reason: IssuingAuthorizationRequestReason,
}

/**
* Whether the cardholder provided a CVC and if it matched Stripe’s record.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CvcCheck {
    #[serde(rename = "match")]
    Match,
    #[serde(rename = "mismatch")]
    Mismatch,
    #[serde(rename = "not_provided")]
    NotProvided,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CvcCheck {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CvcCheck::Match => "match",
            CvcCheck::Mismatch => "mismatch",
            CvcCheck::NotProvided => "not_provided",
            CvcCheck::Noop => "",
            CvcCheck::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CvcCheck {
    fn default() -> CvcCheck {
        CvcCheck::Noop
    }
}
impl CvcCheck {
    pub fn is_noop(&self) -> bool {
        matches!(self, CvcCheck::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingAuthorizationVerificationData {
    /**
    * Whether the cardholder provided a CVC and if it matched Stripe’s record.
    */
    #[serde(
        default,
        skip_serializing_if = "CvcCheck::is_noop",
        rename = "address_line1_check"
    )]
    pub address_line_1_check: CvcCheck,
    /**
    * Whether the cardholder provided a CVC and if it matched Stripe’s record.
    */
    #[serde(default, skip_serializing_if = "CvcCheck::is_noop")]
    pub address_postal_code_check: CvcCheck,
    /**
    * Whether the cardholder provided a CVC and if it matched Stripe’s record.
    */
    #[serde(default, skip_serializing_if = "CvcCheck::is_noop")]
    pub cvc_check: CvcCheck,
    /**
    * Whether the cardholder provided a CVC and if it matched Stripe’s record.
    */
    #[serde(default, skip_serializing_if = "CvcCheck::is_noop")]
    pub expiry_check: CvcCheck,
}

/**
* Reason the card is ineligible for Apple Pay
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IneligibleReason {
    #[serde(rename = "missing_agreement")]
    MissingAgreement,
    #[serde(rename = "missing_cardholder_contact")]
    MissingCardholderContact,
    #[serde(rename = "unsupported_region")]
    UnsupportedRegion,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IneligibleReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IneligibleReason::MissingAgreement => "missing_agreement",
            IneligibleReason::MissingCardholderContact => "missing_cardholder_contact",
            IneligibleReason::UnsupportedRegion => "unsupported_region",
            IneligibleReason::Noop => "",
            IneligibleReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IneligibleReason {
    fn default() -> IneligibleReason {
        IneligibleReason::Noop
    }
}
impl IneligibleReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, IneligibleReason::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardApplePay {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub eligible: bool,
    /**
    * Reason the card is ineligible for Apple Pay
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ineligible_reason: Option<IneligibleReason>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Categories {
    #[serde(rename = "ac_refrigeration_repair")]
    AcRefrigerationRepair,
    #[serde(rename = "accounting_bookkeeping_services")]
    AccountingBookkeepingServices,
    #[serde(rename = "advertising_services")]
    AdvertisingServices,
    #[serde(rename = "agricultural_cooperative")]
    AgriculturalCooperative,
    #[serde(rename = "airlines_air_carriers")]
    AirlinesAirCarriers,
    #[serde(rename = "airports_flying_fields")]
    AirportsFlyingFields,
    #[serde(rename = "ambulance_services")]
    AmbulanceServices,
    #[serde(rename = "amusement_parks_carnivals")]
    AmusementParksCarnivals,
    #[serde(rename = "antique_reproductions")]
    AntiqueReproductions,
    #[serde(rename = "antique_shops")]
    AntiqueShops,
    #[serde(rename = "aquariums")]
    Aquariums,
    #[serde(rename = "architectural_surveying_services")]
    ArchitecturalSurveyingServices,
    #[serde(rename = "art_dealers_and_galleries")]
    ArtDealersAndGalleries,
    #[serde(rename = "artists_supply_and_craft_shops")]
    ArtistsSupplyAndCraftShops,
    #[serde(rename = "auto_and_home_supply_stores")]
    AutoAndHomeSupplyStores,
    #[serde(rename = "auto_body_repair_shops")]
    AutoBodyRepairShops,
    #[serde(rename = "auto_paint_shops")]
    AutoPaintShops,
    #[serde(rename = "auto_service_shops")]
    AutoServiceShops,
    #[serde(rename = "automated_cash_disburse")]
    AutomatedCashDisburse,
    #[serde(rename = "automated_fuel_dispensers")]
    AutomatedFuelDispensers,
    #[serde(rename = "automobile_associations")]
    AutomobileAssociations,
    #[serde(rename = "automotive_parts_and_accessories_stores")]
    AutomotivePartsAndAccessoriesStores,
    #[serde(rename = "automotive_tire_stores")]
    AutomotiveTireStores,
    #[serde(rename = "bail_and_bond_payments")]
    BailAndBondPayments,
    #[serde(rename = "bakeries")]
    Bakeries,
    #[serde(rename = "bands_orchestras")]
    BandsOrchestras,
    #[serde(rename = "barber_and_beauty_shops")]
    BarberAndBeautyShops,
    #[serde(rename = "betting_casino_gambling")]
    BettingCasinoGambling,
    #[serde(rename = "bicycle_shops")]
    BicycleShops,
    #[serde(rename = "billiard_pool_establishments")]
    BilliardPoolEstablishments,
    #[serde(rename = "boat_dealers")]
    BoatDealers,
    #[serde(rename = "boat_rentals_and_leases")]
    BoatRentalsAndLeases,
    #[serde(rename = "book_stores")]
    BookStores,
    #[serde(rename = "books_periodicals_and_newspapers")]
    BooksPeriodicalsAndNewspapers,
    #[serde(rename = "bowling_alleys")]
    BowlingAlleys,
    #[serde(rename = "bus_lines")]
    BusLines,
    #[serde(rename = "business_secretarial_schools")]
    BusinessSecretarialSchools,
    #[serde(rename = "buying_shopping_services")]
    BuyingShoppingServices,
    #[serde(rename = "cable_satellite_and_other_pay_television_and_radio")]
    CableSatelliteAndOtherPayTelevisionRadio,
    #[serde(rename = "camera_and_photographic_supply_stores")]
    CameraAndPhotographicSupplyStores,
    #[serde(rename = "candy_nut_and_confectionery_stores")]
    CandyNutAndConfectioneryStores,
    #[serde(rename = "car_and_truck_dealers_new_used")]
    CarAndTruckDealersNewUsed,
    #[serde(rename = "car_and_truck_dealers_used_only")]
    CarAndTruckDealersUsedOnly,
    #[serde(rename = "car_rental_agencies")]
    CarRentalAgencies,
    #[serde(rename = "car_washes")]
    CarWashes,
    #[serde(rename = "carpentry_services")]
    CarpentryServices,
    #[serde(rename = "carpet_upholstery_cleaning")]
    CarpetUpholsteryCleaning,
    #[serde(rename = "caterers")]
    Caterers,
    #[serde(rename = "charitable_and_social_service_organizations_fundraising")]
    CharitableAndSocialServiceOrganizationsFundraising,
    #[serde(rename = "chemicals_and_allied_products")]
    ChemicalsAndAlliedProducts,
    #[serde(rename = "child_care_services")]
    ChildCareServices,
    #[serde(rename = "childrens_and_infants_wear_stores")]
    ChildrensAndInfantsWearStores,
    #[serde(rename = "chiropodists_podiatrists")]
    ChiropodistsPodiatrists,
    #[serde(rename = "chiropractors")]
    Chiropractors,
    #[serde(rename = "cigar_stores_and_stands")]
    CigarStoresAndStands,
    #[serde(rename = "civic_social_fraternal_associations")]
    CivicSocialFraternalAssociations,
    #[serde(rename = "cleaning_and_maintenance")]
    CleaningAndMaintenance,
    #[serde(rename = "clothing_rental")]
    ClothingRental,
    #[serde(rename = "colleges_universities")]
    CollegesUniversities,
    #[serde(rename = "commercial_equipment")]
    CommercialEquipment,
    #[serde(rename = "commercial_footwear")]
    CommercialFootwear,
    #[serde(rename = "commercial_photography_art_and_graphics")]
    CommercialPhotographyArtAndGraphics,
    #[serde(rename = "commuter_transport_and_ferries")]
    CommuterTransportAndFerries,
    #[serde(rename = "computer_network_services")]
    ComputerNetworkServices,
    #[serde(rename = "computer_programming")]
    ComputerProgramming,
    #[serde(rename = "computer_repair")]
    ComputerRepair,
    #[serde(rename = "computer_software_stores")]
    ComputerSoftwareStores,
    #[serde(rename = "computers_peripherals_and_software")]
    ComputersPeripheralsAndSoftware,
    #[serde(rename = "concrete_work_services")]
    ConcreteWorkServices,
    #[serde(rename = "construction_materials")]
    ConstructionMaterials,
    #[serde(rename = "consulting_public_relations")]
    ConsultingPublicRelations,
    #[serde(rename = "correspondence_schools")]
    CorrespondenceSchools,
    #[serde(rename = "cosmetic_stores")]
    CosmeticStores,
    #[serde(rename = "counseling_services")]
    CounselingServices,
    #[serde(rename = "country_clubs")]
    CountryClubs,
    #[serde(rename = "courier_services")]
    CourierServices,
    #[serde(rename = "court_costs")]
    CourtCosts,
    #[serde(rename = "credit_reporting_agencies")]
    CreditReportingAgencies,
    #[serde(rename = "cruise_lines")]
    CruiseLines,
    #[serde(rename = "dairy_products_stores")]
    DairyProductsStores,
    #[serde(rename = "dance_hall_studios_schools")]
    DanceHallStudiosSchools,
    #[serde(rename = "dating_escort_services")]
    DatingEscortServices,
    #[serde(rename = "dentists_orthodontists")]
    DentistsOrthodontists,
    #[serde(rename = "department_stores")]
    DepartmentStores,
    #[serde(rename = "detective_agencies")]
    DetectiveAgencies,
    #[serde(rename = "digital_goods_applications")]
    DigitalGoodsApplications,
    #[serde(rename = "digital_goods_games")]
    DigitalGoodsGames,
    #[serde(rename = "digital_goods_large_volume")]
    DigitalGoodsLargeVolume,
    #[serde(rename = "digital_goods_media")]
    DigitalGoodsMedia,
    #[serde(rename = "direct_marketing_catalog_merchant")]
    DirectMarketingCatalogMerchant,
    #[serde(rename = "direct_marketing_combination_catalog_and_retail_merchant")]
    DirectMarketingCombinationCatalogAndRetailMerchant,
    #[serde(rename = "direct_marketing_inbound_telemarketing")]
    DirectMarketingInboundTelemarketing,
    #[serde(rename = "direct_marketing_insurance_services")]
    DirectMarketingInsuranceServices,
    #[serde(rename = "direct_marketing_other")]
    DirectMarketingOther,
    #[serde(rename = "direct_marketing_outbound_telemarketing")]
    DirectMarketingOutboundTelemarketing,
    #[serde(rename = "direct_marketing_subscription")]
    DirectMarketingSubscription,
    #[serde(rename = "direct_marketing_travel")]
    DirectMarketingTravel,
    #[serde(rename = "discount_stores")]
    DiscountStores,
    #[serde(rename = "doctors")]
    Doctors,
    #[serde(rename = "door_to_door_sales")]
    DoorToSales,
    #[serde(rename = "drapery_window_covering_and_upholstery_stores")]
    DraperyWindowCoveringAndUpholsteryStores,
    #[serde(rename = "drinking_places")]
    DrinkingPlaces,
    #[serde(rename = "drug_stores_and_pharmacies")]
    DrugStoresAndPharmacies,
    #[serde(rename = "drugs_drug_proprietaries_and_druggist_sundries")]
    DrugsDrugProprietariesAndDruggistSundries,
    #[serde(rename = "dry_cleaners")]
    DryCleaners,
    #[serde(rename = "durable_goods")]
    DurableGoods,
    #[serde(rename = "duty_free_stores")]
    DutyFreeStores,
    #[serde(rename = "eating_places_restaurants")]
    EatingPlacesRestaurants,
    #[serde(rename = "educational_services")]
    EducationalServices,
    #[serde(rename = "electric_razor_stores")]
    ElectricRazorStores,
    #[serde(rename = "electrical_parts_and_equipment")]
    ElectricalPartsAndEquipment,
    #[serde(rename = "electrical_services")]
    ElectricalServices,
    #[serde(rename = "electronics_repair_shops")]
    ElectronicsRepairShops,
    #[serde(rename = "electronics_stores")]
    ElectronicsStores,
    #[serde(rename = "elementary_secondary_schools")]
    ElementarySecondarySchools,
    #[serde(rename = "employment_temp_agencies")]
    EmploymentTempAgencies,
    #[serde(rename = "equipment_rental")]
    EquipmentRental,
    #[serde(rename = "exterminating_services")]
    ExterminatingServices,
    #[serde(rename = "family_clothing_stores")]
    FamilyClothingStores,
    #[serde(rename = "fast_food_restaurants")]
    FastFoodRestaurants,
    #[serde(rename = "financial_institutions")]
    FinancialInstitutions,
    #[serde(rename = "fines_government_administrative_entities")]
    FinesGovernmentAdministrativeEntities,
    #[serde(rename = "fireplace_fireplace_screens_and_accessories_stores")]
    FireplaceScreensAndAccessoriesStores,
    #[serde(rename = "floor_covering_stores")]
    FloorCoveringStores,
    #[serde(rename = "florists")]
    Florists,
    #[serde(rename = "florists_supplies_nursery_stock_and_flowers")]
    FloristsSuppliesNurseryStockAndFlowers,
    #[serde(rename = "freezer_and_locker_meat_provisioners")]
    FreezerAndLockerMeatProvisioners,
    #[serde(rename = "fuel_dealers_non_automotive")]
    FuelDealersNonAutomotive,
    #[serde(rename = "funeral_services_crematories")]
    FuneralServicesCrematories,
    #[serde(rename = "furniture_home_furnishings_and_equipment_stores_except_appliances")]
    FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances,
    #[serde(rename = "furniture_repair_refinishing")]
    FurnitureRepairRefinishing,
    #[serde(rename = "furriers_and_fur_shops")]
    FurriersAndFurShops,
    #[serde(rename = "general_services")]
    GeneralServices,
    #[serde(rename = "gift_card_novelty_and_souvenir_shops")]
    GiftCardNoveltyAndSouvenirShops,
    #[serde(rename = "glass_paint_and_wallpaper_stores")]
    GlassPaintAndWallpaperStores,
    #[serde(rename = "glassware_crystal_stores")]
    GlasswareCrystalStores,
    #[serde(rename = "golf_courses_public")]
    GolfCoursesPublic,
    #[serde(rename = "government_services")]
    GovernmentServices,
    #[serde(rename = "grocery_stores_supermarkets")]
    GroceryStoresSupermarkets,
    #[serde(rename = "hardware_equipment_and_supplies")]
    HardwareEquipmentAndSupplies,
    #[serde(rename = "hardware_stores")]
    HardwareStores,
    #[serde(rename = "health_and_beauty_spas")]
    HealthAndBeautySpas,
    #[serde(rename = "hearing_aids_sales_and_supplies")]
    HearingAidsSalesAndSupplies,
    #[serde(rename = "heating_plumbing_a_c")]
    HeatingPlumbingAC,
    #[serde(rename = "hobby_toy_and_game_shops")]
    HobbyToyAndGameShops,
    #[serde(rename = "home_supply_warehouse_stores")]
    HomeSupplyWarehouseStores,
    #[serde(rename = "hospitals")]
    Hospitals,
    #[serde(rename = "hotels_motels_and_resorts")]
    HotelsMotelsAndResorts,
    #[serde(rename = "household_appliance_stores")]
    HouseholdApplianceStores,
    #[serde(rename = "industrial_supplies")]
    IndustrialSupplies,
    #[serde(rename = "information_retrieval_services")]
    InformationRetrievalServices,
    #[serde(rename = "insurance_default")]
    InsuranceDefault,
    #[serde(rename = "insurance_underwriting_premiums")]
    InsuranceUnderwritingPremiums,
    #[serde(rename = "intra_company_purchases")]
    IntraCompanyPurchases,
    #[serde(rename = "jewelry_stores_watches_clocks_and_silverware_stores")]
    JewelryStoresWatchesClocksAndSilverware,
    #[serde(rename = "landscaping_services")]
    LandscapingServices,
    #[serde(rename = "laundries")]
    Laundries,
    #[serde(rename = "laundry_cleaning_services")]
    LaundryCleaningServices,
    #[serde(rename = "legal_services_attorneys")]
    LegalServicesAttorneys,
    #[serde(rename = "luggage_and_leather_goods_stores")]
    LuggageAndLeatherGoodsStores,
    #[serde(rename = "lumber_building_materials_stores")]
    LumberBuildingMaterialsStores,
    #[serde(rename = "manual_cash_disburse")]
    ManualCashDisburse,
    #[serde(rename = "marinas_service_and_supplies")]
    MarinasServiceAndSupplies,
    #[serde(rename = "masonry_stonework_and_plaster")]
    MasonryStoneworkAndPlaster,
    #[serde(rename = "massage_parlors")]
    MassageParlors,
    #[serde(rename = "medical_and_dental_labs")]
    MedicalAndDentalLabs,
    #[serde(rename = "medical_dental_ophthalmic_and_hospital_equipment_and_supplies")]
    MedicalDentalOphthalmicAndHospitalEquipmentSupplies,
    #[serde(rename = "medical_services")]
    MedicalServices,
    #[serde(rename = "membership_organizations")]
    MembershipOrganizations,
    #[serde(rename = "mens_and_boys_clothing_and_accessories_stores")]
    MensAndBoysClothingAccessoriesStores,
    #[serde(rename = "mens_womens_clothing_stores")]
    MensWomensClothingStores,
    #[serde(rename = "metal_service_centers")]
    MetalServiceCenters,
    #[serde(rename = "miscellaneous")]
    Miscellaneous,
    #[serde(rename = "miscellaneous_apparel_and_accessory_shops")]
    MiscellaneousApparelAndAccessoryShops,
    #[serde(rename = "miscellaneous_auto_dealers")]
    MiscellaneousAutoDealers,
    #[serde(rename = "miscellaneous_business_services")]
    MiscellaneousBusinessServices,
    #[serde(rename = "miscellaneous_food_stores")]
    MiscellaneousFoodStores,
    #[serde(rename = "miscellaneous_general_merchandise")]
    MiscellaneousGeneralMerchandise,
    #[serde(rename = "miscellaneous_general_services")]
    MiscellaneousGeneralServices,
    #[serde(rename = "miscellaneous_home_furnishing_specialty_stores")]
    MiscellaneousHomeFurnishingSpecialtyStores,
    #[serde(rename = "miscellaneous_publishing_and_printing")]
    MiscellaneousPublishingAndPrinting,
    #[serde(rename = "miscellaneous_recreation_services")]
    MiscellaneousRecreationServices,
    #[serde(rename = "miscellaneous_repair_shops")]
    MiscellaneousRepairShops,
    #[serde(rename = "miscellaneous_specialty_retail")]
    MiscellaneousSpecialtyRetail,
    #[serde(rename = "mobile_home_dealers")]
    MobileHomeDealers,
    #[serde(rename = "motion_picture_theaters")]
    MotionPictureTheaters,
    #[serde(rename = "motor_freight_carriers_and_trucking")]
    MotorFreightCarriersAndTrucking,
    #[serde(rename = "motor_homes_dealers")]
    MotorHomesDealers,
    #[serde(rename = "motor_vehicle_supplies_and_new_parts")]
    MotorVehicleSuppliesAndNewParts,
    #[serde(rename = "motorcycle_shops_and_dealers")]
    MotorcycleShopsAndDealers,
    #[serde(rename = "motorcycle_shops_dealers")]
    MotorcycleShopsDealers,
    #[serde(rename = "music_stores_musical_instruments_pianos_and_sheet_music")]
    MusicStoresMusicalInstrumentsPianosAndSheet,
    #[serde(rename = "news_dealers_and_newsstands")]
    NewsDealersAndNewsstands,
    #[serde(rename = "non_fi_money_orders")]
    NonFiMoneyOrders,
    #[serde(rename = "non_fi_stored_value_card_purchase_load")]
    NonFiStoredValueCardPurchaseLoad,
    #[serde(rename = "nondurable_goods")]
    NondurableGoods,
    #[serde(rename = "nurseries_lawn_and_garden_supply_stores")]
    NurseriesLawnAndGardenSupplyStores,
    #[serde(rename = "nursing_personal_care")]
    NursingPersonalCare,
    #[serde(rename = "office_and_commercial_furniture")]
    OfficeAndCommercialFurniture,
    #[serde(rename = "opticians_eyeglasses")]
    OpticiansEyeglasses,
    #[serde(rename = "optometrists_ophthalmologist")]
    OptometristsOphthalmologist,
    #[serde(rename = "orthopedic_goods_prosthetic_devices")]
    OrthopedicGoodsProstheticDevices,
    #[serde(rename = "osteopaths")]
    Osteopaths,
    #[serde(rename = "package_stores_beer_wine_and_liquor")]
    PackageStoresBeerWineAndLiquor,
    #[serde(rename = "paints_varnishes_and_supplies")]
    PaintsVarnishesAndSupplies,
    #[serde(rename = "parking_lots_garages")]
    ParkingLotsGarages,
    #[serde(rename = "passenger_railways")]
    PassengerRailways,
    #[serde(rename = "pawn_shops")]
    PawnShops,
    #[serde(rename = "pet_shops_pet_food_and_supplies")]
    PetShopsFoodAndSupplies,
    #[serde(rename = "petroleum_and_petroleum_products")]
    PetroleumAndProducts,
    #[serde(rename = "photo_developing")]
    PhotoDeveloping,
    #[serde(rename = "photographic_photocopy_microfilm_equipment_and_supplies")]
    PhotographicPhotocopyMicrofilmEquipmentAndSupplies,
    #[serde(rename = "photographic_studios")]
    PhotographicStudios,
    #[serde(rename = "picture_video_production")]
    PictureVideoProduction,
    #[serde(rename = "piece_goods_notions_and_other_dry_goods")]
    PieceGoodsNotionsAndOtherDry,
    #[serde(rename = "plumbing_heating_equipment_and_supplies")]
    PlumbingHeatingEquipmentAndSupplies,
    #[serde(rename = "political_organizations")]
    PoliticalOrganizations,
    #[serde(rename = "postal_services_government_only")]
    PostalServicesGovernmentOnly,
    #[serde(rename = "precious_stones_and_metals_watches_and_jewelry")]
    PreciousStonesAndMetalsWatchesJewelry,
    #[serde(rename = "professional_services")]
    ProfessionalServices,
    #[serde(rename = "public_warehousing_and_storage")]
    PublicWarehousingAndStorage,
    #[serde(rename = "quick_copy_repro_and_blueprint")]
    QuickCopyReproAndBlueprint,
    #[serde(rename = "railroads")]
    Railroads,
    #[serde(rename = "real_estate_agents_and_managers_rentals")]
    RealEstateAgentsAndManagersRentals,
    #[serde(rename = "record_stores")]
    RecordStores,
    #[serde(rename = "recreational_vehicle_rentals")]
    RecreationalVehicleRentals,
    #[serde(rename = "religious_goods_stores")]
    ReligiousGoodsStores,
    #[serde(rename = "religious_organizations")]
    ReligiousOrganizations,
    #[serde(rename = "roofing_siding_sheet_metal")]
    RoofingSidingSheetMetal,
    #[serde(rename = "secretarial_support_services")]
    SecretarialSupportServices,
    #[serde(rename = "security_brokers_dealers")]
    SecurityBrokersDealers,
    #[serde(rename = "service_stations")]
    ServiceStations,
    #[serde(rename = "sewing_needlework_fabric_and_piece_goods_stores")]
    SewingNeedleworkFabricAndPieceGoodsStores,
    #[serde(rename = "shoe_repair_hat_cleaning")]
    ShoeRepairHatCleaning,
    #[serde(rename = "shoe_stores")]
    ShoeStores,
    #[serde(rename = "small_appliance_repair")]
    SmallApplianceRepair,
    #[serde(rename = "snowmobile_dealers")]
    SnowmobileDealers,
    #[serde(rename = "special_trade_services")]
    SpecialTradeServices,
    #[serde(rename = "specialty_cleaning")]
    SpecialtyCleaning,
    #[serde(rename = "sporting_goods_stores")]
    SportingGoodsStores,
    #[serde(rename = "sporting_recreation_camps")]
    SportingRecreationCamps,
    #[serde(rename = "sports_and_riding_apparel_stores")]
    SportsAndRidingApparelStores,
    #[serde(rename = "sports_clubs_fields")]
    SportsClubsFields,
    #[serde(rename = "stamp_and_coin_stores")]
    StampAndCoinStores,
    #[serde(rename = "stationary_office_supplies_printing_and_writing_paper")]
    StationaryOfficeSuppliesPrintingAndWritingPaper,
    #[serde(rename = "stationery_stores_office_and_school_supply_stores")]
    StationeryStoresOfficeAndSchoolSupply,
    #[serde(rename = "swimming_pools_sales")]
    SwimmingPoolsSales,
    #[serde(rename = "t_ui_travel_germany")]
    TUiTravelGermany,
    #[serde(rename = "tailors_alterations")]
    TailorsAlterations,
    #[serde(rename = "tax_payments_government_agencies")]
    TaxPaymentsGovernmentAgencies,
    #[serde(rename = "tax_preparation_services")]
    TaxPreparationServices,
    #[serde(rename = "taxicabs_limousines")]
    TaxicabsLimousines,
    #[serde(rename = "telecommunication_equipment_and_telephone_sales")]
    TelecommunicationEquipmentAndTelephoneSales,
    #[serde(rename = "telecommunication_services")]
    TelecommunicationServices,
    #[serde(rename = "telegraph_services")]
    TelegraphServices,
    #[serde(rename = "tent_and_awning_shops")]
    TentAndAwningShops,
    #[serde(rename = "testing_laboratories")]
    TestingLaboratories,
    #[serde(rename = "theatrical_ticket_agencies")]
    TheatricalTicketAgencies,
    #[serde(rename = "timeshares")]
    Timeshares,
    #[serde(rename = "tire_retreading_and_repair")]
    TireRetreadingAndRepair,
    #[serde(rename = "tolls_bridge_fees")]
    TollsBridgeFees,
    #[serde(rename = "tourist_attractions_and_exhibits")]
    TouristAttractionsAndExhibits,
    #[serde(rename = "towing_services")]
    TowingServices,
    #[serde(rename = "trailer_parks_campgrounds")]
    TrailerParksCampgrounds,
    #[serde(rename = "transportation_services")]
    TransportationServices,
    #[serde(rename = "travel_agencies_tour_operators")]
    TravelAgenciesTourOperators,
    #[serde(rename = "truck_stop_iteration")]
    TruckStopIteration,
    #[serde(rename = "truck_utility_trailer_rentals")]
    TruckUtilityTrailerRentals,
    #[serde(rename = "typesetting_plate_making_and_related_services")]
    TypesettingPlateMakingAndRelatedServices,
    #[serde(rename = "typewriter_stores")]
    TypewriterStores,
    #[serde(rename = "u_s_federal_government_agencies_or_departments")]
    USFederalGovernmentAgenciesOrDepartments,
    #[serde(rename = "uniforms_commercial_clothing")]
    UniformsCommercialClothing,
    #[serde(rename = "used_merchandise_and_secondhand_stores")]
    UsedMerchandiseAndSecondhandStores,
    #[serde(rename = "utilities")]
    Utilities,
    #[serde(rename = "variety_stores")]
    VarietyStores,
    #[serde(rename = "veterinary_services")]
    VeterinaryServices,
    #[serde(rename = "video_amusement_game_supplies")]
    VideoAmusementGameSupplies,
    #[serde(rename = "video_game_arcades")]
    VideoGameArcades,
    #[serde(rename = "video_tape_rental_stores")]
    VideoTapeRentalStores,
    #[serde(rename = "vocational_trade_schools")]
    VocationalTradeSchools,
    #[serde(rename = "watch_jewelry_repair")]
    WatchJewelryRepair,
    #[serde(rename = "welding_repair")]
    WeldingRepair,
    #[serde(rename = "wholesale_clubs")]
    WholesaleClubs,
    #[serde(rename = "wig_and_toupee_stores")]
    WigAndToupeeStores,
    #[serde(rename = "wires_money_orders")]
    WiresMoneyOrders,
    #[serde(rename = "womens_accessory_and_specialty_shops")]
    WomensAccessoryAndSpecialtyShops,
    #[serde(rename = "womens_ready_to_wear_stores")]
    WomensReadyToWearStores,
    #[serde(rename = "wrecking_and_salvage_yards")]
    WreckingAndSalvageYards,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Categories {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Categories::AcRefrigerationRepair => "ac_refrigeration_repair",
            Categories::AccountingBookkeepingServices => "accounting_bookkeeping_services",
            Categories::AdvertisingServices => "advertising_services",
            Categories::AgriculturalCooperative => "agricultural_cooperative",
            Categories::AirlinesAirCarriers => "airlines_air_carriers",
            Categories::AirportsFlyingFields => "airports_flying_fields",
            Categories::AmbulanceServices => "ambulance_services",
            Categories::AmusementParksCarnivals => "amusement_parks_carnivals",
            Categories::AntiqueReproductions => "antique_reproductions",
            Categories::AntiqueShops => "antique_shops",
            Categories::Aquariums => "aquariums",
            Categories::ArchitecturalSurveyingServices => "architectural_surveying_services",
            Categories::ArtDealersAndGalleries => "art_dealers_and_galleries",
            Categories::ArtistsSupplyAndCraftShops => "artists_supply_and_craft_shops",
            Categories::AutoAndHomeSupplyStores => "auto_and_home_supply_stores",
            Categories::AutoBodyRepairShops => "auto_body_repair_shops",
            Categories::AutoPaintShops => "auto_paint_shops",
            Categories::AutoServiceShops => "auto_service_shops",
            Categories::AutomatedCashDisburse => "automated_cash_disburse",
            Categories::AutomatedFuelDispensers => "automated_fuel_dispensers",
            Categories::AutomobileAssociations => "automobile_associations",
            Categories::AutomotivePartsAndAccessoriesStores => {
                "automotive_parts_and_accessories_stores"
            }
            Categories::AutomotiveTireStores => "automotive_tire_stores",
            Categories::BailAndBondPayments => "bail_and_bond_payments",
            Categories::Bakeries => "bakeries",
            Categories::BandsOrchestras => "bands_orchestras",
            Categories::BarberAndBeautyShops => "barber_and_beauty_shops",
            Categories::BettingCasinoGambling => "betting_casino_gambling",
            Categories::BicycleShops => "bicycle_shops",
            Categories::BilliardPoolEstablishments => "billiard_pool_establishments",
            Categories::BoatDealers => "boat_dealers",
            Categories::BoatRentalsAndLeases => "boat_rentals_and_leases",
            Categories::BookStores => "book_stores",
            Categories::BooksPeriodicalsAndNewspapers => "books_periodicals_and_newspapers",
            Categories::BowlingAlleys => "bowling_alleys",
            Categories::BusLines => "bus_lines",
            Categories::BusinessSecretarialSchools => "business_secretarial_schools",
            Categories::BuyingShoppingServices => "buying_shopping_services",
            Categories::CableSatelliteAndOtherPayTelevisionRadio => {
                "cable_satellite_and_other_pay_television_and_radio"
            }
            Categories::CameraAndPhotographicSupplyStores => {
                "camera_and_photographic_supply_stores"
            }
            Categories::CandyNutAndConfectioneryStores => "candy_nut_and_confectionery_stores",
            Categories::CarAndTruckDealersNewUsed => "car_and_truck_dealers_new_used",
            Categories::CarAndTruckDealersUsedOnly => "car_and_truck_dealers_used_only",
            Categories::CarRentalAgencies => "car_rental_agencies",
            Categories::CarWashes => "car_washes",
            Categories::CarpentryServices => "carpentry_services",
            Categories::CarpetUpholsteryCleaning => "carpet_upholstery_cleaning",
            Categories::Caterers => "caterers",
            Categories::CharitableAndSocialServiceOrganizationsFundraising => {
                "charitable_and_social_service_organizations_fundraising"
            }
            Categories::ChemicalsAndAlliedProducts => "chemicals_and_allied_products",
            Categories::ChildCareServices => "child_care_services",
            Categories::ChildrensAndInfantsWearStores => "childrens_and_infants_wear_stores",
            Categories::ChiropodistsPodiatrists => "chiropodists_podiatrists",
            Categories::Chiropractors => "chiropractors",
            Categories::CigarStoresAndStands => "cigar_stores_and_stands",
            Categories::CivicSocialFraternalAssociations => "civic_social_fraternal_associations",
            Categories::CleaningAndMaintenance => "cleaning_and_maintenance",
            Categories::ClothingRental => "clothing_rental",
            Categories::CollegesUniversities => "colleges_universities",
            Categories::CommercialEquipment => "commercial_equipment",
            Categories::CommercialFootwear => "commercial_footwear",
            Categories::CommercialPhotographyArtAndGraphics => {
                "commercial_photography_art_and_graphics"
            }
            Categories::CommuterTransportAndFerries => "commuter_transport_and_ferries",
            Categories::ComputerNetworkServices => "computer_network_services",
            Categories::ComputerProgramming => "computer_programming",
            Categories::ComputerRepair => "computer_repair",
            Categories::ComputerSoftwareStores => "computer_software_stores",
            Categories::ComputersPeripheralsAndSoftware => "computers_peripherals_and_software",
            Categories::ConcreteWorkServices => "concrete_work_services",
            Categories::ConstructionMaterials => "construction_materials",
            Categories::ConsultingPublicRelations => "consulting_public_relations",
            Categories::CorrespondenceSchools => "correspondence_schools",
            Categories::CosmeticStores => "cosmetic_stores",
            Categories::CounselingServices => "counseling_services",
            Categories::CountryClubs => "country_clubs",
            Categories::CourierServices => "courier_services",
            Categories::CourtCosts => "court_costs",
            Categories::CreditReportingAgencies => "credit_reporting_agencies",
            Categories::CruiseLines => "cruise_lines",
            Categories::DairyProductsStores => "dairy_products_stores",
            Categories::DanceHallStudiosSchools => "dance_hall_studios_schools",
            Categories::DatingEscortServices => "dating_escort_services",
            Categories::DentistsOrthodontists => "dentists_orthodontists",
            Categories::DepartmentStores => "department_stores",
            Categories::DetectiveAgencies => "detective_agencies",
            Categories::DigitalGoodsApplications => "digital_goods_applications",
            Categories::DigitalGoodsGames => "digital_goods_games",
            Categories::DigitalGoodsLargeVolume => "digital_goods_large_volume",
            Categories::DigitalGoodsMedia => "digital_goods_media",
            Categories::DirectMarketingCatalogMerchant => "direct_marketing_catalog_merchant",
            Categories::DirectMarketingCombinationCatalogAndRetailMerchant => {
                "direct_marketing_combination_catalog_and_retail_merchant"
            }
            Categories::DirectMarketingInboundTelemarketing => {
                "direct_marketing_inbound_telemarketing"
            }
            Categories::DirectMarketingInsuranceServices => "direct_marketing_insurance_services",
            Categories::DirectMarketingOther => "direct_marketing_other",
            Categories::DirectMarketingOutboundTelemarketing => {
                "direct_marketing_outbound_telemarketing"
            }
            Categories::DirectMarketingSubscription => "direct_marketing_subscription",
            Categories::DirectMarketingTravel => "direct_marketing_travel",
            Categories::DiscountStores => "discount_stores",
            Categories::Doctors => "doctors",
            Categories::DoorToSales => "door_to_door_sales",
            Categories::DraperyWindowCoveringAndUpholsteryStores => {
                "drapery_window_covering_and_upholstery_stores"
            }
            Categories::DrinkingPlaces => "drinking_places",
            Categories::DrugStoresAndPharmacies => "drug_stores_and_pharmacies",
            Categories::DrugsDrugProprietariesAndDruggistSundries => {
                "drugs_drug_proprietaries_and_druggist_sundries"
            }
            Categories::DryCleaners => "dry_cleaners",
            Categories::DurableGoods => "durable_goods",
            Categories::DutyFreeStores => "duty_free_stores",
            Categories::EatingPlacesRestaurants => "eating_places_restaurants",
            Categories::EducationalServices => "educational_services",
            Categories::ElectricRazorStores => "electric_razor_stores",
            Categories::ElectricalPartsAndEquipment => "electrical_parts_and_equipment",
            Categories::ElectricalServices => "electrical_services",
            Categories::ElectronicsRepairShops => "electronics_repair_shops",
            Categories::ElectronicsStores => "electronics_stores",
            Categories::ElementarySecondarySchools => "elementary_secondary_schools",
            Categories::EmploymentTempAgencies => "employment_temp_agencies",
            Categories::EquipmentRental => "equipment_rental",
            Categories::ExterminatingServices => "exterminating_services",
            Categories::FamilyClothingStores => "family_clothing_stores",
            Categories::FastFoodRestaurants => "fast_food_restaurants",
            Categories::FinancialInstitutions => "financial_institutions",
            Categories::FinesGovernmentAdministrativeEntities => {
                "fines_government_administrative_entities"
            }
            Categories::FireplaceScreensAndAccessoriesStores => {
                "fireplace_fireplace_screens_and_accessories_stores"
            }
            Categories::FloorCoveringStores => "floor_covering_stores",
            Categories::Florists => "florists",
            Categories::FloristsSuppliesNurseryStockAndFlowers => {
                "florists_supplies_nursery_stock_and_flowers"
            }
            Categories::FreezerAndLockerMeatProvisioners => "freezer_and_locker_meat_provisioners",
            Categories::FuelDealersNonAutomotive => "fuel_dealers_non_automotive",
            Categories::FuneralServicesCrematories => "funeral_services_crematories",
            Categories::FurnitureHomeFurnishingsAndEquipmentStoresExceptAppliances => {
                "furniture_home_furnishings_and_equipment_stores_except_appliances"
            }
            Categories::FurnitureRepairRefinishing => "furniture_repair_refinishing",
            Categories::FurriersAndFurShops => "furriers_and_fur_shops",
            Categories::GeneralServices => "general_services",
            Categories::GiftCardNoveltyAndSouvenirShops => "gift_card_novelty_and_souvenir_shops",
            Categories::GlassPaintAndWallpaperStores => "glass_paint_and_wallpaper_stores",
            Categories::GlasswareCrystalStores => "glassware_crystal_stores",
            Categories::GolfCoursesPublic => "golf_courses_public",
            Categories::GovernmentServices => "government_services",
            Categories::GroceryStoresSupermarkets => "grocery_stores_supermarkets",
            Categories::HardwareEquipmentAndSupplies => "hardware_equipment_and_supplies",
            Categories::HardwareStores => "hardware_stores",
            Categories::HealthAndBeautySpas => "health_and_beauty_spas",
            Categories::HearingAidsSalesAndSupplies => "hearing_aids_sales_and_supplies",
            Categories::HeatingPlumbingAC => "heating_plumbing_a_c",
            Categories::HobbyToyAndGameShops => "hobby_toy_and_game_shops",
            Categories::HomeSupplyWarehouseStores => "home_supply_warehouse_stores",
            Categories::Hospitals => "hospitals",
            Categories::HotelsMotelsAndResorts => "hotels_motels_and_resorts",
            Categories::HouseholdApplianceStores => "household_appliance_stores",
            Categories::IndustrialSupplies => "industrial_supplies",
            Categories::InformationRetrievalServices => "information_retrieval_services",
            Categories::InsuranceDefault => "insurance_default",
            Categories::InsuranceUnderwritingPremiums => "insurance_underwriting_premiums",
            Categories::IntraCompanyPurchases => "intra_company_purchases",
            Categories::JewelryStoresWatchesClocksAndSilverware => {
                "jewelry_stores_watches_clocks_and_silverware_stores"
            }
            Categories::LandscapingServices => "landscaping_services",
            Categories::Laundries => "laundries",
            Categories::LaundryCleaningServices => "laundry_cleaning_services",
            Categories::LegalServicesAttorneys => "legal_services_attorneys",
            Categories::LuggageAndLeatherGoodsStores => "luggage_and_leather_goods_stores",
            Categories::LumberBuildingMaterialsStores => "lumber_building_materials_stores",
            Categories::ManualCashDisburse => "manual_cash_disburse",
            Categories::MarinasServiceAndSupplies => "marinas_service_and_supplies",
            Categories::MasonryStoneworkAndPlaster => "masonry_stonework_and_plaster",
            Categories::MassageParlors => "massage_parlors",
            Categories::MedicalAndDentalLabs => "medical_and_dental_labs",
            Categories::MedicalDentalOphthalmicAndHospitalEquipmentSupplies => {
                "medical_dental_ophthalmic_and_hospital_equipment_and_supplies"
            }
            Categories::MedicalServices => "medical_services",
            Categories::MembershipOrganizations => "membership_organizations",
            Categories::MensAndBoysClothingAccessoriesStores => {
                "mens_and_boys_clothing_and_accessories_stores"
            }
            Categories::MensWomensClothingStores => "mens_womens_clothing_stores",
            Categories::MetalServiceCenters => "metal_service_centers",
            Categories::Miscellaneous => "miscellaneous",
            Categories::MiscellaneousApparelAndAccessoryShops => {
                "miscellaneous_apparel_and_accessory_shops"
            }
            Categories::MiscellaneousAutoDealers => "miscellaneous_auto_dealers",
            Categories::MiscellaneousBusinessServices => "miscellaneous_business_services",
            Categories::MiscellaneousFoodStores => "miscellaneous_food_stores",
            Categories::MiscellaneousGeneralMerchandise => "miscellaneous_general_merchandise",
            Categories::MiscellaneousGeneralServices => "miscellaneous_general_services",
            Categories::MiscellaneousHomeFurnishingSpecialtyStores => {
                "miscellaneous_home_furnishing_specialty_stores"
            }
            Categories::MiscellaneousPublishingAndPrinting => {
                "miscellaneous_publishing_and_printing"
            }
            Categories::MiscellaneousRecreationServices => "miscellaneous_recreation_services",
            Categories::MiscellaneousRepairShops => "miscellaneous_repair_shops",
            Categories::MiscellaneousSpecialtyRetail => "miscellaneous_specialty_retail",
            Categories::MobileHomeDealers => "mobile_home_dealers",
            Categories::MotionPictureTheaters => "motion_picture_theaters",
            Categories::MotorFreightCarriersAndTrucking => "motor_freight_carriers_and_trucking",
            Categories::MotorHomesDealers => "motor_homes_dealers",
            Categories::MotorVehicleSuppliesAndNewParts => "motor_vehicle_supplies_and_new_parts",
            Categories::MotorcycleShopsAndDealers => "motorcycle_shops_and_dealers",
            Categories::MotorcycleShopsDealers => "motorcycle_shops_dealers",
            Categories::MusicStoresMusicalInstrumentsPianosAndSheet => {
                "music_stores_musical_instruments_pianos_and_sheet_music"
            }
            Categories::NewsDealersAndNewsstands => "news_dealers_and_newsstands",
            Categories::NonFiMoneyOrders => "non_fi_money_orders",
            Categories::NonFiStoredValueCardPurchaseLoad => {
                "non_fi_stored_value_card_purchase_load"
            }
            Categories::NondurableGoods => "nondurable_goods",
            Categories::NurseriesLawnAndGardenSupplyStores => {
                "nurseries_lawn_and_garden_supply_stores"
            }
            Categories::NursingPersonalCare => "nursing_personal_care",
            Categories::OfficeAndCommercialFurniture => "office_and_commercial_furniture",
            Categories::OpticiansEyeglasses => "opticians_eyeglasses",
            Categories::OptometristsOphthalmologist => "optometrists_ophthalmologist",
            Categories::OrthopedicGoodsProstheticDevices => "orthopedic_goods_prosthetic_devices",
            Categories::Osteopaths => "osteopaths",
            Categories::PackageStoresBeerWineAndLiquor => "package_stores_beer_wine_and_liquor",
            Categories::PaintsVarnishesAndSupplies => "paints_varnishes_and_supplies",
            Categories::ParkingLotsGarages => "parking_lots_garages",
            Categories::PassengerRailways => "passenger_railways",
            Categories::PawnShops => "pawn_shops",
            Categories::PetShopsFoodAndSupplies => "pet_shops_pet_food_and_supplies",
            Categories::PetroleumAndProducts => "petroleum_and_petroleum_products",
            Categories::PhotoDeveloping => "photo_developing",
            Categories::PhotographicPhotocopyMicrofilmEquipmentAndSupplies => {
                "photographic_photocopy_microfilm_equipment_and_supplies"
            }
            Categories::PhotographicStudios => "photographic_studios",
            Categories::PictureVideoProduction => "picture_video_production",
            Categories::PieceGoodsNotionsAndOtherDry => "piece_goods_notions_and_other_dry_goods",
            Categories::PlumbingHeatingEquipmentAndSupplies => {
                "plumbing_heating_equipment_and_supplies"
            }
            Categories::PoliticalOrganizations => "political_organizations",
            Categories::PostalServicesGovernmentOnly => "postal_services_government_only",
            Categories::PreciousStonesAndMetalsWatchesJewelry => {
                "precious_stones_and_metals_watches_and_jewelry"
            }
            Categories::ProfessionalServices => "professional_services",
            Categories::PublicWarehousingAndStorage => "public_warehousing_and_storage",
            Categories::QuickCopyReproAndBlueprint => "quick_copy_repro_and_blueprint",
            Categories::Railroads => "railroads",
            Categories::RealEstateAgentsAndManagersRentals => {
                "real_estate_agents_and_managers_rentals"
            }
            Categories::RecordStores => "record_stores",
            Categories::RecreationalVehicleRentals => "recreational_vehicle_rentals",
            Categories::ReligiousGoodsStores => "religious_goods_stores",
            Categories::ReligiousOrganizations => "religious_organizations",
            Categories::RoofingSidingSheetMetal => "roofing_siding_sheet_metal",
            Categories::SecretarialSupportServices => "secretarial_support_services",
            Categories::SecurityBrokersDealers => "security_brokers_dealers",
            Categories::ServiceStations => "service_stations",
            Categories::SewingNeedleworkFabricAndPieceGoodsStores => {
                "sewing_needlework_fabric_and_piece_goods_stores"
            }
            Categories::ShoeRepairHatCleaning => "shoe_repair_hat_cleaning",
            Categories::ShoeStores => "shoe_stores",
            Categories::SmallApplianceRepair => "small_appliance_repair",
            Categories::SnowmobileDealers => "snowmobile_dealers",
            Categories::SpecialTradeServices => "special_trade_services",
            Categories::SpecialtyCleaning => "specialty_cleaning",
            Categories::SportingGoodsStores => "sporting_goods_stores",
            Categories::SportingRecreationCamps => "sporting_recreation_camps",
            Categories::SportsAndRidingApparelStores => "sports_and_riding_apparel_stores",
            Categories::SportsClubsFields => "sports_clubs_fields",
            Categories::StampAndCoinStores => "stamp_and_coin_stores",
            Categories::StationaryOfficeSuppliesPrintingAndWritingPaper => {
                "stationary_office_supplies_printing_and_writing_paper"
            }
            Categories::StationeryStoresOfficeAndSchoolSupply => {
                "stationery_stores_office_and_school_supply_stores"
            }
            Categories::SwimmingPoolsSales => "swimming_pools_sales",
            Categories::TUiTravelGermany => "t_ui_travel_germany",
            Categories::TailorsAlterations => "tailors_alterations",
            Categories::TaxPaymentsGovernmentAgencies => "tax_payments_government_agencies",
            Categories::TaxPreparationServices => "tax_preparation_services",
            Categories::TaxicabsLimousines => "taxicabs_limousines",
            Categories::TelecommunicationEquipmentAndTelephoneSales => {
                "telecommunication_equipment_and_telephone_sales"
            }
            Categories::TelecommunicationServices => "telecommunication_services",
            Categories::TelegraphServices => "telegraph_services",
            Categories::TentAndAwningShops => "tent_and_awning_shops",
            Categories::TestingLaboratories => "testing_laboratories",
            Categories::TheatricalTicketAgencies => "theatrical_ticket_agencies",
            Categories::Timeshares => "timeshares",
            Categories::TireRetreadingAndRepair => "tire_retreading_and_repair",
            Categories::TollsBridgeFees => "tolls_bridge_fees",
            Categories::TouristAttractionsAndExhibits => "tourist_attractions_and_exhibits",
            Categories::TowingServices => "towing_services",
            Categories::TrailerParksCampgrounds => "trailer_parks_campgrounds",
            Categories::TransportationServices => "transportation_services",
            Categories::TravelAgenciesTourOperators => "travel_agencies_tour_operators",
            Categories::TruckStopIteration => "truck_stop_iteration",
            Categories::TruckUtilityTrailerRentals => "truck_utility_trailer_rentals",
            Categories::TypesettingPlateMakingAndRelatedServices => {
                "typesetting_plate_making_and_related_services"
            }
            Categories::TypewriterStores => "typewriter_stores",
            Categories::USFederalGovernmentAgenciesOrDepartments => {
                "u_s_federal_government_agencies_or_departments"
            }
            Categories::UniformsCommercialClothing => "uniforms_commercial_clothing",
            Categories::UsedMerchandiseAndSecondhandStores => {
                "used_merchandise_and_secondhand_stores"
            }
            Categories::Utilities => "utilities",
            Categories::VarietyStores => "variety_stores",
            Categories::VeterinaryServices => "veterinary_services",
            Categories::VideoAmusementGameSupplies => "video_amusement_game_supplies",
            Categories::VideoGameArcades => "video_game_arcades",
            Categories::VideoTapeRentalStores => "video_tape_rental_stores",
            Categories::VocationalTradeSchools => "vocational_trade_schools",
            Categories::WatchJewelryRepair => "watch_jewelry_repair",
            Categories::WeldingRepair => "welding_repair",
            Categories::WholesaleClubs => "wholesale_clubs",
            Categories::WigAndToupeeStores => "wig_and_toupee_stores",
            Categories::WiresMoneyOrders => "wires_money_orders",
            Categories::WomensAccessoryAndSpecialtyShops => "womens_accessory_and_specialty_shops",
            Categories::WomensReadyToWearStores => "womens_ready_to_wear_stores",
            Categories::WreckingAndSalvageYards => "wrecking_and_salvage_yards",
            Categories::Noop => "",
            Categories::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Categories {
    fn default() -> Categories {
        Categories::Noop
    }
}
impl Categories {
    pub fn is_noop(&self) -> bool {
        matches!(self, Categories::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardAuthorizationControls {
    /**
    * Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub allowed_categories: Vec<Categories>,
    /**
    * Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub blocked_categories: Vec<Categories>,
    /**
    * Limit spending with amount-based rules that apply across any cards this card replaced (i.e., its `replacement_for` card and _that_ card's `replacement_for` card, up the chain).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub spending_limits: Vec<IssuingCardSpendingLimit>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub spending_limits_currency: String,
}

/**
* The delivery company that shipped a card.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Carrier {
    #[serde(rename = "dhl")]
    Dhl,
    #[serde(rename = "fedex")]
    Fedex,
    #[serde(rename = "royal_mail")]
    RoyalMail,
    #[serde(rename = "usps")]
    Usps,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Carrier {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Carrier::Dhl => "dhl",
            Carrier::Fedex => "fedex",
            Carrier::RoyalMail => "royal_mail",
            Carrier::Usps => "usps",
            Carrier::Noop => "",
            Carrier::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Carrier {
    fn default() -> Carrier {
        Carrier::Noop
    }
}
impl Carrier {
    pub fn is_noop(&self) -> bool {
        matches!(self, Carrier::Noop)
    }
}

/**
* Shipment service, such as `standard` or `express`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Service {
    #[serde(rename = "express")]
    Express,
    #[serde(rename = "priority")]
    Priority,
    #[serde(rename = "standard")]
    Standard,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Service {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Service::Express => "express",
            Service::Priority => "priority",
            Service::Standard => "standard",
            Service::Noop => "",
            Service::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Service {
    fn default() -> Service {
        Service::Noop
    }
}
impl Service {
    pub fn is_noop(&self) -> bool {
        matches!(self, Service::Noop)
    }
}

/**
* The delivery status of the card.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardShippingStatus {
    #[serde(rename = "canceled")]
    Canceled,
    #[serde(rename = "delivered")]
    Delivered,
    #[serde(rename = "failure")]
    Failure,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "returned")]
    Returned,
    #[serde(rename = "shipped")]
    Shipped,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardShippingStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardShippingStatus::Canceled => "canceled",
            IssuingCardShippingStatus::Delivered => "delivered",
            IssuingCardShippingStatus::Failure => "failure",
            IssuingCardShippingStatus::Pending => "pending",
            IssuingCardShippingStatus::Returned => "returned",
            IssuingCardShippingStatus::Shipped => "shipped",
            IssuingCardShippingStatus::Noop => "",
            IssuingCardShippingStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardShippingStatus {
    fn default() -> IssuingCardShippingStatus {
        IssuingCardShippingStatus::Noop
    }
}
impl IssuingCardShippingStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardShippingStatus::Noop)
    }
}

/**
* Packaging options.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingCardShippingType {
    #[serde(rename = "bulk")]
    Bulk,
    #[serde(rename = "individual")]
    Individual,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingCardShippingType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingCardShippingType::Bulk => "bulk",
            IssuingCardShippingType::Individual => "individual",
            IssuingCardShippingType::Noop => "",
            IssuingCardShippingType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingCardShippingType {
    fn default() -> IssuingCardShippingType {
        IssuingCardShippingType::Noop
    }
}
impl IssuingCardShippingType {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingCardShippingType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardShippingData {
    /**
    *
    */
    #[serde()]
    pub address: Address,
    /**
    * The delivery company that shipped a card.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub carrier: Option<Carrier>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub eta: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    * Shipment service, such as `standard` or `express`.
    */
    #[serde(default, skip_serializing_if = "Service::is_noop")]
    pub service: Service,
    /**
    * The delivery status of the card.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status: Option<IssuingCardShippingStatus>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tracking_number: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tracking_url: String,
    /**
    * Packaging options.
    */
    #[serde(
        default,
        skip_serializing_if = "IssuingCardShippingType::is_noop",
        rename = "type"
    )]
    pub type_: IssuingCardShippingType,
}

/**
* Interval (or event) to which the amount applies.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Interval {
    #[serde(rename = "all_time")]
    AllTime,
    #[serde(rename = "daily")]
    Daily,
    #[serde(rename = "monthly")]
    Monthly,
    #[serde(rename = "per_authorization")]
    PerAuthorization,
    #[serde(rename = "weekly")]
    Weekly,
    #[serde(rename = "yearly")]
    Yearly,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Interval {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Interval::AllTime => "all_time",
            Interval::Daily => "daily",
            Interval::Monthly => "monthly",
            Interval::PerAuthorization => "per_authorization",
            Interval::Weekly => "weekly",
            Interval::Yearly => "yearly",
            Interval::Noop => "",
            Interval::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Interval {
    fn default() -> Interval {
        Interval::Noop
    }
}
impl Interval {
    pub fn is_noop(&self) -> bool {
        matches!(self, Interval::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardSpendingLimit {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub categories: Vec<Categories>,
    /**
    * Interval (or event) to which the amount applies.
    */
    #[serde(default, skip_serializing_if = "Interval::is_noop")]
    pub interval: Interval,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardWallets {
    /**
    *
    */
    #[serde()]
    pub apple_pay: IssuingCardApplePay,
    /**
    *
    */
    #[serde()]
    pub google_pay: IssuingCardApplePay,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub primary_account_identifier: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderAddress {
    /**
    *
    */
    #[serde()]
    pub address: Address,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderAuthorizationControls {
    /**
    * Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub allowed_categories: Vec<Categories>,
    /**
    * Array of strings containing [categories](https://stripe.com/docs/api#issuing_authorization_object-merchant_data-category) of authorizations to allow. All other categories will be blocked. Cannot be set with `blocked_categories`.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub blocked_categories: Vec<Categories>,
    /**
    * Limit spending with amount-based rules that apply across this cardholder's cards.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub spending_limits: Vec<IssuingCardSpendingLimit>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub spending_limits_currency: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderCompany {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub tax_id_provided: bool,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderDocument {
    /**
    *
    */
    #[serde()]
    pub back: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub front: Box<Option<IconAnyOf>>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderIndividual {
    /**
    * The date of birth of this cardholder.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub dob: Option<IssuingCardholderIndividualDobData>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub first_name: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub last_name: String,
    /**
    * Government-issued ID document for this cardholder.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification: Option<IssuingCardholderVerification>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderIndividualDobData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/**
* If `disabled_reason` is present, all cards will decline authorizations with `cardholder_verification_required` reason.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum DisabledReason {
    #[serde(rename = "listed")]
    Listed,
    #[serde(rename = "rejected.listed")]
    RejectedListed,
    #[serde(rename = "under_review")]
    UnderReview,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for DisabledReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            DisabledReason::Listed => "listed",
            DisabledReason::RejectedListed => "rejected.listed",
            DisabledReason::UnderReview => "under_review",
            DisabledReason::Noop => "",
            DisabledReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for DisabledReason {
    fn default() -> DisabledReason {
        DisabledReason::Noop
    }
}
impl DisabledReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, DisabledReason::Noop)
    }
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PastDue {
    #[serde(rename = "company.tax_id")]
    CompanyTaxId,
    #[serde(rename = "individual.dob.day")]
    IndividualDobDay,
    #[serde(rename = "individual.dob.month")]
    IndividualDobMonth,
    #[serde(rename = "individual.dob.year")]
    IndividualDobYear,
    #[serde(rename = "individual.first_name")]
    IndividualFirstName,
    #[serde(rename = "individual.last_name")]
    IndividualLastName,
    #[serde(rename = "individual.verification.document")]
    IndividualVerificationDocument,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PastDue {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PastDue::CompanyTaxId => "company.tax_id",
            PastDue::IndividualDobDay => "individual.dob.day",
            PastDue::IndividualDobMonth => "individual.dob.month",
            PastDue::IndividualDobYear => "individual.dob.year",
            PastDue::IndividualFirstName => "individual.first_name",
            PastDue::IndividualLastName => "individual.last_name",
            PastDue::IndividualVerificationDocument => "individual.verification.document",
            PastDue::Noop => "",
            PastDue::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PastDue {
    fn default() -> PastDue {
        PastDue::Noop
    }
}
impl PastDue {
    pub fn is_noop(&self) -> bool {
        matches!(self, PastDue::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderRequirements {
    /**
    * If `disabled_reason` is present, all cards will decline authorizations with `cardholder_verification_required` reason.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub disabled_reason: Option<DisabledReason>,
    /**
    * Array of fields that need to be collected in order to verify and re-enable the cardholder.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub past_due: Vec<PastDue>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingCardholderVerification {
    /**
    * An identifying document, either a passport or local ID card.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub document: Option<IssuingCardholderDocument>,
}

/**
* Whether the product was a merchandise or service.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ProductType {
    #[serde(rename = "merchandise")]
    Merchandise,
    #[serde(rename = "service")]
    Service,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ProductType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ProductType::Merchandise => "merchandise",
            ProductType::Service => "service",
            ProductType::Noop => "",
            ProductType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ProductType {
    fn default() -> ProductType {
        ProductType::Noop
    }
}
impl ProductType {
    pub fn is_noop(&self) -> bool {
        matches!(self, ProductType::Noop)
    }
}

/**
* Result of cardholder's attempt to return the product.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ReturnStatus {
    #[serde(rename = "merchant_rejected")]
    MerchantRejected,
    #[serde(rename = "successful")]
    Successful,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ReturnStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ReturnStatus::MerchantRejected => "merchant_rejected",
            ReturnStatus::Successful => "successful",
            ReturnStatus::Noop => "",
            ReturnStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ReturnStatus {
    fn default() -> ReturnStatus {
        ReturnStatus::Noop
    }
}
impl ReturnStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, ReturnStatus::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeCanceledEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub canceled_at: i64,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub cancellation_policy_provided: bool,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cancellation_reason: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expected_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub product_description: String,
    /**
    * Whether the product was a merchandise or service.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub product_type: Option<ProductType>,
    /**
    * Result of cardholder's attempt to return the product.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub return_status: Option<ReturnStatus>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub returned_at: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeDuplicateEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub card_statement: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub cash_receipt: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub check_image: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub original_transaction: String,
}

/**
* The reason for filing the dispute. Its value will match the field containing the evidence.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum IssuingDisputeEvidenceReason {
    #[serde(rename = "canceled")]
    Canceled,
    #[serde(rename = "duplicate")]
    Duplicate,
    #[serde(rename = "fraudulent")]
    Fraudulent,
    #[serde(rename = "merchandise_not_as_described")]
    MerchandiseNotAsDescribed,
    #[serde(rename = "not_received")]
    NotReceived,
    #[serde(rename = "other")]
    Other,
    #[serde(rename = "service_not_as_described")]
    ServiceNotAsDescribed,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for IssuingDisputeEvidenceReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            IssuingDisputeEvidenceReason::Canceled => "canceled",
            IssuingDisputeEvidenceReason::Duplicate => "duplicate",
            IssuingDisputeEvidenceReason::Fraudulent => "fraudulent",
            IssuingDisputeEvidenceReason::MerchandiseNotAsDescribed => {
                "merchandise_not_as_described"
            }
            IssuingDisputeEvidenceReason::NotReceived => "not_received",
            IssuingDisputeEvidenceReason::Other => "other",
            IssuingDisputeEvidenceReason::ServiceNotAsDescribed => "service_not_as_described",
            IssuingDisputeEvidenceReason::Noop => "",
            IssuingDisputeEvidenceReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for IssuingDisputeEvidenceReason {
    fn default() -> IssuingDisputeEvidenceReason {
        IssuingDisputeEvidenceReason::Noop
    }
}
impl IssuingDisputeEvidenceReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, IssuingDisputeEvidenceReason::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeEvidence {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub canceled: Option<IssuingDisputeCanceledEvidence>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub duplicate: Option<IssuingDisputeDuplicateEvidence>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub fraudulent: Option<IssuingDisputeFraudulentEvidence>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub merchandise_not_as_described: Option<IssuingDisputeMerchandiseNotAsDescribedEvidence>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub not_received: Option<IssuingDisputeNotReceivedEvidence>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub other: Option<IssuingDisputeOtherEvidence>,
    /**
    * The reason for filing the dispute. Its value will match the field containing the evidence.
    */
    #[serde(default, skip_serializing_if = "IssuingDisputeEvidenceReason::is_noop")]
    pub reason: IssuingDisputeEvidenceReason,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub service_not_as_described: Option<IssuingDisputeServiceNotAsDescribedEvidence>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeFraudulentEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeMerchandiseNotAsDescribedEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub received_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub return_description: String,
    /**
    * Result of cardholder's attempt to return the product.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub return_status: Option<ReturnStatus>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub returned_at: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeNotReceivedEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expected_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub product_description: String,
    /**
    * Whether the product was a merchandise or service.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub product_type: Option<ProductType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeOtherEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub product_description: String,
    /**
    * Whether the product was a merchandise or service.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub product_type: Option<ProductType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingDisputeServiceNotAsDescribedEvidence {
    /**
    *
    */
    #[serde()]
    pub additional_documentation: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub canceled_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cancellation_reason: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub explanation: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub received_at: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionAmountDetailsData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub atm_fee: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionFlightData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub departure_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub passenger_name: String,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub refundable: bool,
    /**
    * The legs of the trip.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub segments: Vec<IssuingTransactionFlightDataLeg>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub travel_agency: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionFlightDataLeg {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub arrival_airport_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub carrier: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub departure_airport_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub flight_number: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub service_class: String,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub stopover_allowed: bool,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionFuelData {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub unit: String,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub unit_cost_decimal: f64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub volume_decimal: f64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionLodgingData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub check_in_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub nights: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionPurchaseDetails {
    /**
    * Information about the flight that was purchased with this transaction.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub flight: Option<IssuingTransactionFlightData>,
    /**
    * Information about fuel that was purchased with this transaction.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub fuel: Option<IssuingTransactionFuelData>,
    /**
    * Information about lodging that was purchased with this transaction.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub lodging: Option<IssuingTransactionLodgingData>,
    /**
    * The line items in the purchase.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub receipt: Vec<IssuingTransactionReceiptData>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reference: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct IssuingTransactionReceiptData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub quantity: f64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub total: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub unit_cost: i64,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum ItemObject {
    #[serde(rename = "item")]
    Item,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for ItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            ItemObject::Item => "item",
            ItemObject::Noop => "",
            ItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for ItemObject {
    fn default() -> ItemObject {
        ItemObject::Noop
    }
}
impl ItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, ItemObject::Noop)
    }
}

/// A line item.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Item {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_subtotal: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_total: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * A line item.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub discounts: Vec<LineItemsDiscountAmount>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "ItemObject::is_noop")]
    pub object: ItemObject,
    /**
    * The price of the invoice item.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub price: Option<PriceData>,
    /**
    * A line item.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub quantity: i64,
    /**
    * A line item.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub taxes: Vec<LineItemsTaxAmount>,
}

/**
* The category identifying the legal structure of the company or legal entity. See [Business structure](https://stripe.com/docs/connect/identity-verification#business-structure) for more details.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum Structure {
    #[serde(rename = "free_zone_establishment")]
    FreeZoneEstablishment,
    #[serde(rename = "free_zone_llc")]
    FreeZoneLlc,
    #[serde(rename = "government_instrumentality")]
    GovernmentInstrumentality,
    #[serde(rename = "governmental_unit")]
    GovernmentalUnit,
    #[serde(rename = "incorporated_non_profit")]
    IncorporatedNonProfit,
    #[serde(rename = "limited_liability_partnership")]
    LimitedLiabilityPartnership,
    #[serde(rename = "llc")]
    Llc,
    #[serde(rename = "multi_member_llc")]
    MultiMemberLlc,
    #[serde(rename = "private_company")]
    PrivateCompany,
    #[serde(rename = "private_corporation")]
    PrivateCorporation,
    #[serde(rename = "private_partnership")]
    PrivatePartnership,
    #[serde(rename = "public_company")]
    PublicCompany,
    #[serde(rename = "public_corporation")]
    PublicCorporation,
    #[serde(rename = "public_partnership")]
    PublicPartnership,
    #[serde(rename = "single_member_llc")]
    SingleMemberLlc,
    #[serde(rename = "sole_establishment")]
    SoleEstablishment,
    #[serde(rename = "sole_proprietorship")]
    SoleProprietorship,
    #[serde(rename = "tax_exempt_government_instrumentality")]
    TaxExemptGovernmentInstrumentality,
    #[serde(rename = "unincorporated_association")]
    UnincorporatedAssociation,
    #[serde(rename = "unincorporated_non_profit")]
    UnincorporatedNonProfit,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for Structure {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            Structure::FreeZoneEstablishment => "free_zone_establishment",
            Structure::FreeZoneLlc => "free_zone_llc",
            Structure::GovernmentInstrumentality => "government_instrumentality",
            Structure::GovernmentalUnit => "governmental_unit",
            Structure::IncorporatedNonProfit => "incorporated_non_profit",
            Structure::LimitedLiabilityPartnership => "limited_liability_partnership",
            Structure::Llc => "llc",
            Structure::MultiMemberLlc => "multi_member_llc",
            Structure::PrivateCompany => "private_company",
            Structure::PrivateCorporation => "private_corporation",
            Structure::PrivatePartnership => "private_partnership",
            Structure::PublicCompany => "public_company",
            Structure::PublicCorporation => "public_corporation",
            Structure::PublicPartnership => "public_partnership",
            Structure::SingleMemberLlc => "single_member_llc",
            Structure::SoleEstablishment => "sole_establishment",
            Structure::SoleProprietorship => "sole_proprietorship",
            Structure::TaxExemptGovernmentInstrumentality => {
                "tax_exempt_government_instrumentality"
            }
            Structure::UnincorporatedAssociation => "unincorporated_association",
            Structure::UnincorporatedNonProfit => "unincorporated_non_profit",
            Structure::Noop => "",
            Structure::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for Structure {
    fn default() -> Structure {
        Structure::Noop
    }
}
impl Structure {
    pub fn is_noop(&self) -> bool {
        matches!(self, Structure::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityCompany {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address: Option<Address>,
    /**
    * The Kana variation of the company's primary address (Japan only).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address_kana: Option<LegalEntityJapanAddress>,
    /**
    * The Kana variation of the company's primary address (Japan only).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub address_kanji: Option<LegalEntityJapanAddress>,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub directors_provided: bool,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub executives_provided: bool,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name_kana: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub name_kanji: String,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub owners_provided: bool,
    /**
    * This hash is used to attest that the beneficial owner information provided to Stripe is both current and correct.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ownership_declaration: Option<LegalEntityUboDeclaration>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub phone: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub structure: Option<Structure>,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub tax_id_provided: bool,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tax_id_registrar: String,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub vat_id_provided: bool,
    /**
    * Information on the verification state of the company.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification: Option<LegalEntityCompanyVerificationData>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityCompanyVerificationData {
    /**
    *
    */
    #[serde()]
    pub document: LegalEntityCompanyVerificationDocument,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityCompanyVerificationDocument {
    /**
    *
    */
    #[serde()]
    pub back: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub details: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub details_code: String,
    /**
    *
    */
    #[serde()]
    pub front: Box<Option<IconAnyOf>>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityDob {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityJapanAddress {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub city: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "line1"
    )]
    pub line_1: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "line2"
    )]
    pub line_2: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub postal_code: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub state: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub town: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityPersonVerification {
    /**
    * A document showing address, either a passport, local ID card, or utility bill from a well-known utility company.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub additional_document: Option<LegalEntityPersonVerificationDocument>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub details: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub details_code: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub document: Option<LegalEntityPersonVerificationDocument>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityPersonVerificationDocument {
    /**
    *
    */
    #[serde()]
    pub back: Box<Option<IconAnyOf>>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub details: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub details_code: String,
    /**
    *
    */
    #[serde()]
    pub front: Box<Option<IconAnyOf>>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LegalEntityUboDeclaration {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub date: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_agent: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum LineItemObject {
    #[serde(rename = "line_item")]
    LineItem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for LineItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            LineItemObject::LineItem => "line_item",
            LineItemObject::Noop => "",
            LineItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for LineItemObject {
    fn default() -> LineItemObject {
        LineItemObject::Noop
    }
}
impl LineItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, LineItemObject::Noop)
    }
}

/**
* A string identifying the type of the source of this line item, either an `invoiceitem` or a `subscription`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum LineItemType {
    #[serde(rename = "invoiceitem")]
    Invoiceitem,
    #[serde(rename = "subscription")]
    Subscription,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for LineItemType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            LineItemType::Invoiceitem => "invoiceitem",
            LineItemType::Subscription => "subscription",
            LineItemType::Noop => "",
            LineItemType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for LineItemType {
    fn default() -> LineItemType {
        LineItemType::Noop
    }
}
impl LineItemType {
    pub fn is_noop(&self) -> bool {
        matches!(self, LineItemType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LineItem {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub discount_amounts: Vec<DiscountsResourceDiscountAmount>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub discountable: bool,
    /**
    * The discounts which apply to the invoice item. Item discounts are applied before invoice discounts. Use `expand[]=discounts` to expand each discount.
    */
    #[serde()]
    pub discounts: Box<Vec<DiscountsAnyOf>>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub invoice_item: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "LineItemObject::is_noop")]
    pub object: LineItemObject,
    /**
    *
    */
    #[serde()]
    pub period: InvoiceLineItemPeriod,
    /**
    * The price of the invoice item.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub price: Option<PriceData>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub proration: bool,
    /**
    * Additional details for proration line items
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub proration_details: Option<InvoicesLineItemsProrationDetails>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub quantity: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub subscription: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub subscription_item: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub tax_amounts: Vec<InvoiceTaxAmount>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub tax_rates: Vec<TaxRate>,
    /**
    * A string identifying the type of the source of this line item, either an `invoiceitem` or a `subscription`.
    */
    #[serde(
        default,
        skip_serializing_if = "LineItemType::is_noop",
        rename = "type"
    )]
    pub type_: LineItemType,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LineItemsDiscountAmount {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * A discount represents the actual application of a coupon to a particular
    *  customer. It contains information about when the discount began and when it
    *  will end.
    *  
    *  Related guide: [Applying Discounts to Subscriptions](https://stripe.com/docs/billing/subscriptions/discounts).
    */
    #[serde()]
    pub discount: DiscountData,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LineItemsTaxAmount {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Tax rates can be applied to [invoices](https://stripe.com/docs/billing/invoices/tax-rates), [subscriptions](https://stripe.com/docs/billing/subscriptions/taxes) and [Checkout Sessions](https://stripe.com/docs/payments/checkout/set-up-a-subscription#tax-rates) to collect tax.
    *  
    *  Related guide: [Tax Rates](https://stripe.com/docs/billing/taxes/tax-rates).
    */
    #[serde()]
    pub rate: TaxRate,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum LoginLinkObject {
    #[serde(rename = "login_link")]
    LoginLink,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for LoginLinkObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            LoginLinkObject::LoginLink => "login_link",
            LoginLinkObject::Noop => "",
            LoginLinkObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for LoginLinkObject {
    fn default() -> LoginLinkObject {
        LoginLinkObject::Noop
    }
}
impl LoginLinkObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, LoginLinkObject::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct LoginLink {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "LoginLinkObject::is_noop")]
    pub object: LoginLinkObject,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum MandateObject {
    #[serde(rename = "mandate")]
    Mandate,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for MandateObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MandateObject::Mandate => "mandate",
            MandateObject::Noop => "",
            MandateObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for MandateObject {
    fn default() -> MandateObject {
        MandateObject::Noop
    }
}
impl MandateObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, MandateObject::Noop)
    }
}

/**
* The type of the mandate.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum MandateType {
    #[serde(rename = "multi_use")]
    MultiUse,
    #[serde(rename = "single_use")]
    SingleUse,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for MandateType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MandateType::MultiUse => "multi_use",
            MandateType::SingleUse => "single_use",
            MandateType::Noop => "",
            MandateType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for MandateType {
    fn default() -> MandateType {
        MandateType::Noop
    }
}
impl MandateType {
    pub fn is_noop(&self) -> bool {
        matches!(self, MandateType::Noop)
    }
}

/// A Mandate is a record of the permission a customer has given you to debit their payment method.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Mandate {
    /**
    *
    */
    #[serde()]
    pub customer_acceptance: CustomerAcceptance,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * A Mandate is a record of the permission a customer has given you to debit their payment method.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub multi_use: Option<UseStripeSdk>,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "MandateObject::is_noop")]
    pub object: MandateObject,
    #[serde()]
    pub payment_method: Box<PaymentMethodAnyOf>,
    /**
    *
    */
    #[serde()]
    pub payment_method_details: MandatePaymentMethodDetails,
    /**
    * A Mandate is a record of the permission a customer has given you to debit their payment method.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub single_use: Option<MandateSingleUse>,
    /**
    * The status of the transfers capability of the account, or whether your platform can transfer funds to the account.
    */
    #[serde(default, skip_serializing_if = "Transfers::is_noop")]
    pub status: Transfers,
    /**
    * The type of the mandate.
    */
    #[serde(default, skip_serializing_if = "MandateType::is_noop", rename = "type")]
    pub type_: MandateType,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct MandateAcssDebit {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub default_for: Vec<DefaultFor>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub interval_description: String,
    /**
    * Payment schedule for the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_schedule: Option<PaymentSchedule>,
    /**
    * Transaction type of the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transaction_type: Option<TransactionType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct MandateAuBecsDebit {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* The status of the mandate on the Bacs network. Can be one of `pending`, `revoked`, `refused`, or `accepted`.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum NetworkStatus {
    #[serde(rename = "accepted")]
    Accepted,
    #[serde(rename = "pending")]
    Pending,
    #[serde(rename = "refused")]
    Refused,
    #[serde(rename = "revoked")]
    Revoked,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for NetworkStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            NetworkStatus::Accepted => "accepted",
            NetworkStatus::Pending => "pending",
            NetworkStatus::Refused => "refused",
            NetworkStatus::Revoked => "revoked",
            NetworkStatus::Noop => "",
            NetworkStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for NetworkStatus {
    fn default() -> NetworkStatus {
        NetworkStatus::Noop
    }
}
impl NetworkStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, NetworkStatus::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct MandateBacsDebit {
    /**
    * The status of the mandate on the Bacs network. Can be one of `pending`, `revoked`, `refused`, or `accepted`.
    */
    #[serde(default, skip_serializing_if = "NetworkStatus::is_noop")]
    pub network_status: NetworkStatus,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reference: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct MandatePaymentMethodDetails {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub acss_debit: Option<MandateAcssDebit>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub au_becs_debit: Option<MandateAuBecsDebit>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bacs_debit: Option<MandateBacsDebit>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sepa_debit: Option<MandateSepaDebit>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub us_bank_account: Option<UseStripeSdk>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct MandateSepaDebit {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub reference: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct MandateSingleUse {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Networks {
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub available: Vec<String>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub preferred: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct NotificationEventData {
    /**
    * When confirming a PaymentIntent with Stripe.js, Stripe.js depends on the contents of this dictionary to invoke authentication flows. The shape of the contents is subject to change and is only intended to be used by Stripe.js.
    */
    #[serde()]
    pub object: UseStripeSdk,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub previous_attributes: Option<UseStripeSdk>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct NotificationEventRequest {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub idempotency_key: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct OnlineAcceptance {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub ip_address: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub user_agent: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum OrderObject {
    #[serde(rename = "order")]
    Order,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for OrderObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            OrderObject::Order => "order",
            OrderObject::Noop => "",
            OrderObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for OrderObject {
    fn default() -> OrderObject {
        OrderObject::Noop
    }
}
impl OrderObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, OrderObject::Noop)
    }
}

/// A list of returns that have taken place for this order.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Returns {
    /**
    * Details about each object.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<OrderReturn>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/// Order objects are created to handle end customers' purchases of previously
/// defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
/// as list all orders. Orders are identified by a unique, random ID.
///
/// Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Order {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_returned: i64,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub application: String,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub application_fee: i64,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde()]
    pub charge: Box<Option<ChargeAnyOf>>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub email: String,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub external_coupon_code: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * List of items constituting the order. An order can have up to 25 items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub items: Vec<OrderItem>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "OrderObject::is_noop")]
    pub object: OrderObject,
    /**
    * A list of returns that have taken place for this order.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub returns: Option<Returns>,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub selected_shipping_method: String,
    /**
    * Shipping information for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping: Option<ShippingData>,
    /**
    * A list of supported shipping methods for this order. The desired shipping method can be specified either by updating the order, or when paying it.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub shipping_methods: Vec<ShippingMethod>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub status: String,
    /**
    * The timestamps at which the order status was updated.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub status_transitions: Option<StatusTransitions>,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub updated: i64,
    /**
    * Order objects are created to handle end customers' purchases of previously
    *  defined [products](https://stripe.com/docs/api#products). You can create, retrieve, and pay individual orders, as well
    *  as list all orders. Orders are identified by a unique, random ID.
    *  
    *  Related guide: [Tax, Shipping, and Inventory](https://stripe.com/docs/orders-legacy).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub upstream_id: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum OrderItemObject {
    #[serde(rename = "order_item")]
    OrderItem,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for OrderItemObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            OrderItemObject::OrderItem => "order_item",
            OrderItemObject::Noop => "",
            OrderItemObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for OrderItemObject {
    fn default() -> OrderItemObject {
        OrderItemObject::Noop
    }
}
impl OrderItemObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, OrderItemObject::Noop)
    }
}

/// All of the following types:
///
/// - `String`
/// - `Sku`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum ParentAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * Stores representations of [stock keeping units](http://en.wikipedia.org/wiki/Stock_keeping_unit).
    *  SKUs describe specific product variations, taking into account any combination of: attributes,
    *  currency, and cost. For example, a product may be a T-shirt, whereas a specific SKU represents
    *  the `size: large`, `color: red` version of that shirt.
    *  
    *  Can also be used to manage inventory.
    */
    Sku(Sku),
}

impl ParentAnyOf {
    pub fn sku(&self) -> Option<&Sku> {
        if let ParentAnyOf::Sku(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let ParentAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for ParentAnyOf {
    fn from(f: String) -> Self {
        ParentAnyOf::String(f)
    }
}

impl std::convert::From<ParentAnyOf> for String {
    fn from(f: ParentAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

/// A representation of the constituent items of any given order. Can be used to
/// represent [SKUs](https://stripe.com/docs/api#skus), shipping costs, or taxes owed on the order.
///
/// Related guide: [Orders](https://stripe.com/docs/orders/guide).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct OrderItem {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "OrderItemObject::is_noop")]
    pub object: OrderItemObject,
    /**
    * A representation of the constituent items of any given order. Can be used to
    *  represent [SKUs](https://stripe.com/docs/api#skus), shipping costs, or taxes owed on the order.
    *  
    *  Related guide: [Orders](https://stripe.com/docs/orders/guide).
    */
    #[serde()]
    pub parent: Box<Option<ParentAnyOf>>,
    /**
    * A representation of the constituent items of any given order. Can be used to
    *  represent [SKUs](https://stripe.com/docs/api#skus), shipping costs, or taxes owed on the order.
    *  
    *  Related guide: [Orders](https://stripe.com/docs/orders/guide).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub quantity: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum OrderReturnObject {
    #[serde(rename = "order_return")]
    OrderReturn,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for OrderReturnObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            OrderReturnObject::OrderReturn => "order_return",
            OrderReturnObject::Noop => "",
            OrderReturnObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for OrderReturnObject {
    fn default() -> OrderReturnObject {
        OrderReturnObject::Noop
    }
}
impl OrderReturnObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, OrderReturnObject::Noop)
    }
}

/// A return represents the full or partial return of a number of [order items](https://stripe.com/docs/api#order_items).
/// Returns always belong to an order, and may optionally contain a refund.
///
/// Related guide: [Handling Returns](https://stripe.com/docs/orders/guide#handling-returns).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct OrderReturn {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * List of items constituting the order. An order can have up to 25 items.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub items: Vec<OrderItem>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "OrderReturnObject::is_noop")]
    pub object: OrderReturnObject,
    /**
    * A return represents the full or partial return of a number of [order items](https://stripe.com/docs/api#order_items).
    *  Returns always belong to an order, and may optionally contain a refund.
    *  
    *  Related guide: [Handling Returns](https://stripe.com/docs/orders/guide#handling-returns).
    */
    #[serde()]
    pub order: Box<Option<OrderAnyOf>>,
    /**
    * A return represents the full or partial return of a number of [order items](https://stripe.com/docs/api#order_items).
    *  Returns always belong to an order, and may optionally contain a refund.
    *  
    *  Related guide: [Handling Returns](https://stripe.com/docs/orders/guide#handling-returns).
    */
    #[serde()]
    pub refund: Box<Option<RefundAnyOf>>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PackageDimensions {
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub height: f64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub length: f64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub weight: f64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub width: f64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PortalInvoiceList {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub enabled: bool,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentFlowsPrivateMethodsAlipayDetails {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub buyer_id: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub transaction_id: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentFlowsPrivateMethodsKlarnaDob {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub day: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub month: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub year: i64,
}

/**
* Reason for cancellation of this PaymentIntent, either user-provided (`duplicate`, `fraudulent`, `requested_by_customer`, or `abandoned`) or generated by Stripe internally (`failed_invoice`, `void_invoice`, or `automatic`).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentCancellationReason {
    #[serde(rename = "abandoned")]
    Abandoned,
    #[serde(rename = "automatic")]
    Automatic,
    #[serde(rename = "duplicate")]
    Duplicate,
    #[serde(rename = "failed_invoice")]
    FailedInvoice,
    #[serde(rename = "fraudulent")]
    Fraudulent,
    #[serde(rename = "requested_by_customer")]
    RequestedByCustomer,
    #[serde(rename = "void_invoice")]
    VoidInvoice,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentCancellationReason {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentCancellationReason::Abandoned => "abandoned",
            PaymentIntentCancellationReason::Automatic => "automatic",
            PaymentIntentCancellationReason::Duplicate => "duplicate",
            PaymentIntentCancellationReason::FailedInvoice => "failed_invoice",
            PaymentIntentCancellationReason::Fraudulent => "fraudulent",
            PaymentIntentCancellationReason::RequestedByCustomer => "requested_by_customer",
            PaymentIntentCancellationReason::VoidInvoice => "void_invoice",
            PaymentIntentCancellationReason::Noop => "",
            PaymentIntentCancellationReason::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentCancellationReason {
    fn default() -> PaymentIntentCancellationReason {
        PaymentIntentCancellationReason::Noop
    }
}
impl PaymentIntentCancellationReason {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentIntentCancellationReason::Noop)
    }
}

/**
* Controls when the funds will be captured from the customer's account.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum CaptureMethod {
    #[serde(rename = "automatic")]
    Automatic,
    #[serde(rename = "manual")]
    Manual,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for CaptureMethod {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            CaptureMethod::Automatic => "automatic",
            CaptureMethod::Manual => "manual",
            CaptureMethod::Noop => "",
            CaptureMethod::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for CaptureMethod {
    fn default() -> CaptureMethod {
        CaptureMethod::Noop
    }
}
impl CaptureMethod {
    pub fn is_noop(&self) -> bool {
        matches!(self, CaptureMethod::Noop)
    }
}

/// Charges that were created by this PaymentIntent, if any.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct Charges {
    /**
    * This list only contains the latest charge, even if there were previously multiple unsuccessful charges. To view all previous charges for a PaymentIntent, you can filter the charges list using the `payment_intent` [parameter](https://stripe.com/docs/api/charges/list#list_charges-payment_intent).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub data: Vec<Charge>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub has_more: bool,
    /**
    * String representing the object's type. Objects of the same type share the same value. Always has the value `list`.
    */
    #[serde(default, skip_serializing_if = "Object::is_noop")]
    pub object: Object,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentObject {
    #[serde(rename = "payment_intent")]
    PaymentIntent,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentObject::PaymentIntent => "payment_intent",
            PaymentIntentObject::Noop => "",
            PaymentIntentObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentObject {
    fn default() -> PaymentIntentObject {
        PaymentIntentObject::Noop
    }
}
impl PaymentIntentObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentIntentObject::Noop)
    }
}

/**
* Indicates that you intend to make future payments with this PaymentIntent's payment method.
*   
*   Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
*   
*   When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum SetupFutureUsage {
    #[serde(rename = "off_session")]
    OffSession,
    #[serde(rename = "on_session")]
    OnSession,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for SetupFutureUsage {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            SetupFutureUsage::OffSession => "off_session",
            SetupFutureUsage::OnSession => "on_session",
            SetupFutureUsage::Noop => "",
            SetupFutureUsage::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for SetupFutureUsage {
    fn default() -> SetupFutureUsage {
        SetupFutureUsage::Noop
    }
}
impl SetupFutureUsage {
    pub fn is_noop(&self) -> bool {
        matches!(self, SetupFutureUsage::Noop)
    }
}

/**
* Status of this PaymentIntent, one of `requires_payment_method`, `requires_confirmation`, `requires_action`, `processing`, `requires_capture`, `canceled`, or `succeeded`. Read more about each PaymentIntent [status](https://stripe.com/docs/payments/intents#intent-statuses).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentStatus {
    #[serde(rename = "canceled")]
    Canceled,
    #[serde(rename = "processing")]
    Processing,
    #[serde(rename = "requires_action")]
    RequiresAction,
    #[serde(rename = "requires_capture")]
    RequiresCapture,
    #[serde(rename = "requires_confirmation")]
    RequiresConfirmation,
    #[serde(rename = "requires_payment_method")]
    RequiresPaymentMethod,
    #[serde(rename = "succeeded")]
    Succeeded,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentStatus {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentStatus::Canceled => "canceled",
            PaymentIntentStatus::Processing => "processing",
            PaymentIntentStatus::RequiresAction => "requires_action",
            PaymentIntentStatus::RequiresCapture => "requires_capture",
            PaymentIntentStatus::RequiresConfirmation => "requires_confirmation",
            PaymentIntentStatus::RequiresPaymentMethod => "requires_payment_method",
            PaymentIntentStatus::Succeeded => "succeeded",
            PaymentIntentStatus::Noop => "",
            PaymentIntentStatus::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentStatus {
    fn default() -> PaymentIntentStatus {
        PaymentIntentStatus::Noop
    }
}
impl PaymentIntentStatus {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentIntentStatus::Noop)
    }
}

/// A PaymentIntent guides you through the process of collecting a payment from your customer.
/// We recommend that you create exactly one PaymentIntent for each order or
/// customer session in your system. You can reference the PaymentIntent later to
/// see the history of payment attempts for a particular session.
///
/// A PaymentIntent transitions through
/// [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
/// throughout its lifetime as it interfaces with Stripe.js to perform
/// authentication flows and ultimately creates at most one successful charge.
///
/// Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntent {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_capturable: i64,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount_received: i64,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde()]
    pub application: Box<Option<ApplicationAnyOf>>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub application_fee_amount: i64,
    /**
    * Settings to configure compatible payment methods from the [Stripe Dashboard](https://dashboard.stripe.com/settings/payment_methods)
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub automatic_payment_methods: Option<PortalInvoiceList>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub canceled_at: i64,
    /**
    * Reason for cancellation of this PaymentIntent, either user-provided (`duplicate`, `fraudulent`, `requested_by_customer`, or `abandoned`) or generated by Stripe internally (`failed_invoice`, `void_invoice`, or `automatic`).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub cancellation_reason: Option<PaymentIntentCancellationReason>,
    /**
    * Controls when the funds will be captured from the customer's account.
    */
    #[serde(default, skip_serializing_if = "CaptureMethod::is_noop")]
    pub capture_method: CaptureMethod,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub charges: Option<Charges>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub client_secret: String,
    /**
    * Controls when the funds will be captured from the customer's account.
    */
    #[serde(default, skip_serializing_if = "CaptureMethod::is_noop")]
    pub confirmation_method: CaptureMethod,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub currency: String,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde()]
    pub customer: Box<Option<CustomerAnyOf>>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub description: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde()]
    pub invoice: Box<Option<InvoiceAnyOf>>,
    /**
    * The error encountered during the previous attempt to finalize the invoice. This field is cleared when the invoice is successfully finalized.
    */
    #[serde()]
    pub last_payment_error: Box<Option<ApiErrors>>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * If present, this property tells you what actions you need to take in order for your customer to fulfill a payment using the provided source.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub next_action: Option<PaymentIntentNextAction>,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "PaymentIntentObject::is_noop")]
    pub object: PaymentIntentObject,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde()]
    pub on_behalf_of: Box<Option<AccountAnyOf>>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde()]
    pub payment_method: Box<Option<PaymentMethodAnyOf>>,
    /**
    * Payment-method-specific configuration for this PaymentIntent.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_method_options: Option<PaymentIntentMethodOptionsData>,
    /**
    * Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub payment_method_types: Vec<String>,
    /**
    * If present, this property tells you about the processing state of the payment.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub processing: Option<PaymentIntentProcessing>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub receipt_email: String,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde()]
    pub review: Box<Option<ReviewAnyOf>>,
    /**
    * Indicates that you intend to make future payments with this PaymentIntent's payment method.
    *  
    *  Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
    *  
    *  When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<SetupFutureUsage>,
    /**
    * Shipping information for the charge.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping: Option<ShippingData>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor: String,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub statement_descriptor_suffix: String,
    /**
    * Status of this PaymentIntent, one of `requires_payment_method`, `requires_confirmation`, `requires_action`, `processing`, `requires_capture`, `canceled`, or `succeeded`. Read more about each PaymentIntent [status](https://stripe.com/docs/payments/intents#intent-statuses).
    */
    #[serde(default, skip_serializing_if = "PaymentIntentStatus::is_noop")]
    pub status: PaymentIntentStatus,
    /**
    * The data with which to automatically create a Transfer when the payment is finalized. See the PaymentIntents [use case for connected accounts](https://stripe.com/docs/payments/connected-accounts) for details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transfer_data: Option<TransferData>,
    /**
    * A PaymentIntent guides you through the process of collecting a payment from your customer.
    *  We recommend that you create exactly one PaymentIntent for each order or
    *  customer session in your system. You can reference the PaymentIntent later to
    *  see the history of payment attempts for a particular session.
    *  
    *  A PaymentIntent transitions through
    *  [multiple statuses](https://stripe.com/docs/payments/intents#intent-statuses)
    *  throughout its lifetime as it interfaces with Stripe.js to perform
    *  authentication flows and ultimately creates at most one successful charge.
    *  
    *  Related guide: [Payment Intents API](https://stripe.com/docs/payments/payment-intents).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub transfer_group: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentCardProcessing {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer_notification: Option<PaymentIntentProcessingCustomerNotification>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextAction {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub alipay_handle_redirect: Option<PaymentIntentNextActionAlipayHandleRedirect>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub boleto_display_details: Option<PaymentIntentNextActionBoleto>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card_await_notification: Option<PaymentIntentNextActionCardAwaitNotification>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub konbini_display_details: Option<PaymentIntentNextActionKonbini>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub oxxo_display_details: Option<PaymentIntentNextActionDisplayOxxoDetails>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub paynow_display_qr_code: Option<PaymentIntentNextActionPaynowDisplayQrCode>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub redirect_to_url: Option<PaymentIntentNextActionRedirectUrl>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "type"
    )]
    pub type_: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub use_stripe_sdk: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verify_with_microdeposits: Option<SetupIntentNextActionVerifyWithMicrodeposits>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wechat_pay_display_qr_code: Option<PaymentIntentNextActionWechatPayDisplayQrCode>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wechat_pay_redirect_to_android_app:
        Option<PaymentIntentNextActionWechatPayRedirectAndroidApp>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wechat_pay_redirect_to_ios_app: Option<PaymentIntentNextActionWechatPayRedirectIosApp>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionAlipayHandleRedirect {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub native_data: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub native_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub return_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionBoleto {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub hosted_voucher_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub pdf: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionCardAwaitNotification {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub charge_attempt_at: i64,
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub customer_approval_required: bool,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionDisplayOxxoDetails {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_after: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub hosted_voucher_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub number: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionKonbini {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub expires_at: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub hosted_voucher_url: String,
    /**
    *
    */
    #[serde()]
    pub stores: PaymentIntentNextActionKonbiniStores,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionKonbiniFamilymart {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub confirmation_number: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_code: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionKonbiniLawson {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub confirmation_number: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_code: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionKonbiniMinistop {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub confirmation_number: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_code: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionKonbiniSeicomart {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub confirmation_number: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub payment_code: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionKonbiniStores {
    /**
    * FamilyMart instruction details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub familymart: Option<PaymentIntentNextActionKonbiniFamilymart>,
    /**
    * Lawson instruction details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub lawson: Option<PaymentIntentNextActionKonbiniLawson>,
    /**
    * Ministop instruction details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ministop: Option<PaymentIntentNextActionKonbiniMinistop>,
    /**
    * Seicomart instruction details.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub seicomart: Option<PaymentIntentNextActionKonbiniSeicomart>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionPaynowDisplayQrCode {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub data: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub image_url_png: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub image_url_svg: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionRedirectUrl {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub return_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* The type of the microdeposit sent to the customer. Used to distinguish between different verification methods.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum MicrodepositType {
    #[serde(rename = "amounts")]
    Amounts,
    #[serde(rename = "descriptor_code")]
    DescriptorCode,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for MicrodepositType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            MicrodepositType::Amounts => "amounts",
            MicrodepositType::DescriptorCode => "descriptor_code",
            MicrodepositType::Noop => "",
            MicrodepositType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for MicrodepositType {
    fn default() -> MicrodepositType {
        MicrodepositType::Noop
    }
}
impl MicrodepositType {
    pub fn is_noop(&self) -> bool {
        matches!(self, MicrodepositType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct SetupIntentNextActionVerifyWithMicrodeposits {
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub arrival_date: i64,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub hosted_verification_url: String,
    /**
    * The type of the microdeposit sent to the customer. Used to distinguish between different verification methods.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub microdeposit_type: Option<MicrodepositType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionWechatPayDisplayQrCode {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub data: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub image_data_url: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub image_url_png: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub image_url_svg: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionWechatPayRedirectAndroidApp {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub app_id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub nonce_str: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub package: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub partner_id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub prepay_id: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sign: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub timestamp: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentNextActionWechatPayRedirectIosApp {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub native_url: String,
}

/// All of the following types:
///
/// - `PaymentIntentMethodOptionsAcssDebit`
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AcssDebitAnyOf {
    /**
    *
    */
    PaymentIntentMethodOptionsAcssDebit(PaymentIntentMethodOptionsAcssDebit),
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
}

impl AcssDebitAnyOf {
    pub fn payment_intent_method_options_acss_debit(
        &self,
    ) -> Option<&PaymentIntentMethodOptionsAcssDebit> {
        if let AcssDebitAnyOf::PaymentIntentMethodOptionsAcssDebit(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let AcssDebitAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsAfterpayClearpay`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AfterpayClearpayAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsAfterpayClearpay(PaymentMethodOptionsAfterpayClearpay),
}

impl AfterpayClearpayAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let AfterpayClearpayAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_afterpay_clearpay(
        &self,
    ) -> Option<&PaymentMethodOptionsAfterpayClearpay> {
        if let AfterpayClearpayAnyOf::PaymentMethodOptionsAfterpayClearpay(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsAlipay`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AlipayAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsAlipay(PaymentMethodOptionsAlipay),
}

impl AlipayAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let AlipayAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_alipay(&self) -> Option<&PaymentMethodOptionsAlipay> {
        if let AlipayAnyOf::PaymentMethodOptionsAlipay(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentIntentMethodOptionsAuBecsDebit`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum AuBecsDebitAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentIntentMethodOptionsAuBecsDebit(PaymentIntentMethodOptionsAuBecsDebit),
}

impl AuBecsDebitAnyOf {
    pub fn payment_intent_method_options_au_becs_debit(
        &self,
    ) -> Option<&PaymentIntentMethodOptionsAuBecsDebit> {
        if let AuBecsDebitAnyOf::PaymentIntentMethodOptionsAuBecsDebit(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let AuBecsDebitAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsBacsDebit`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum BacsDebitAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsBacsDebit(PaymentMethodOptionsBacsDebit),
}

impl BacsDebitAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let BacsDebitAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_bacs_debit(&self) -> Option<&PaymentMethodOptionsBacsDebit> {
        if let BacsDebitAnyOf::PaymentMethodOptionsBacsDebit(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsBancontact`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum BancontactAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsBancontact(PaymentMethodOptionsBancontact),
}

impl BancontactAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let BancontactAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_bancontact(&self) -> Option<&PaymentMethodOptionsBancontact> {
        if let BancontactAnyOf::PaymentMethodOptionsBancontact(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsBoleto`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum BoletoAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsBoleto(PaymentMethodOptionsBoleto),
}

impl BoletoAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let BoletoAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_boleto(&self) -> Option<&PaymentMethodOptionsBoleto> {
        if let BoletoAnyOf::PaymentMethodOptionsBoleto(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentIntentMethodOptionsCard`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PaymentIntentMethodOptionsCardAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentIntentMethodOptionsCard(PaymentIntentMethodOptionsCard),
}

impl PaymentIntentMethodOptionsCardAnyOf {
    pub fn payment_intent_method_options_card(&self) -> Option<&PaymentIntentMethodOptionsCard> {
        if let PaymentIntentMethodOptionsCardAnyOf::PaymentIntentMethodOptionsCard(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let PaymentIntentMethodOptionsCardAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(
            ref_,
        ) = self
        {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsCardPresent`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum CardPresentAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsCardPresent(PaymentMethodOptionsCardPresent),
}

impl CardPresentAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let CardPresentAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_card_present(&self) -> Option<&PaymentMethodOptionsCardPresent> {
        if let CardPresentAnyOf::PaymentMethodOptionsCardPresent(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentIntentMethodOptionsEps`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum EpsAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentIntentMethodOptionsEps(PaymentIntentMethodOptionsEps),
}

impl EpsAnyOf {
    pub fn payment_intent_method_options_eps(&self) -> Option<&PaymentIntentMethodOptionsEps> {
        if let EpsAnyOf::PaymentIntentMethodOptionsEps(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let EpsAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsFpx`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum FpxAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsFpx(PaymentMethodOptionsFpx),
}

impl FpxAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let FpxAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_fpx(&self) -> Option<&PaymentMethodOptionsFpx> {
        if let FpxAnyOf::PaymentMethodOptionsFpx(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsGiropay`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GiropayAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsGiropay(PaymentMethodOptionsGiropay),
}

impl GiropayAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let GiropayAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_giropay(&self) -> Option<&PaymentMethodOptionsGiropay> {
        if let GiropayAnyOf::PaymentMethodOptionsGiropay(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsGrabpay`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum GrabpayAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsGrabpay(PaymentMethodOptionsGrabpay),
}

impl GrabpayAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let GrabpayAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_grabpay(&self) -> Option<&PaymentMethodOptionsGrabpay> {
        if let GrabpayAnyOf::PaymentMethodOptionsGrabpay(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsIdeal`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum IdealAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsIdeal(PaymentMethodOptionsIdeal),
}

impl IdealAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let IdealAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_ideal(&self) -> Option<&PaymentMethodOptionsIdeal> {
        if let IdealAnyOf::PaymentMethodOptionsIdeal(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `UseStripeSdk`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum InteracPresentAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    * When confirming a PaymentIntent with Stripe.js, Stripe.js depends on the contents of this dictionary to invoke authentication flows. The shape of the contents is subject to change and is only intended to be used by Stripe.js.
    */
    UseStripeSdk(UseStripeSdk),
}

impl InteracPresentAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let InteracPresentAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn use_stripe_sdk(&self) -> Option<&UseStripeSdk> {
        if let InteracPresentAnyOf::UseStripeSdk(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsKlarna`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum KlarnaAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsKlarna(PaymentMethodOptionsKlarna),
}

impl KlarnaAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let KlarnaAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_klarna(&self) -> Option<&PaymentMethodOptionsKlarna> {
        if let KlarnaAnyOf::PaymentMethodOptionsKlarna(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsKonbini`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum KonbiniAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsKonbini(PaymentMethodOptionsKonbini),
}

impl KonbiniAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let KonbiniAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_konbini(&self) -> Option<&PaymentMethodOptionsKonbini> {
        if let KonbiniAnyOf::PaymentMethodOptionsKonbini(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsOxxo`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum OxxoAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsOxxo(PaymentMethodOptionsOxxo),
}

impl OxxoAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let OxxoAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_oxxo(&self) -> Option<&PaymentMethodOptionsOxxo> {
        if let OxxoAnyOf::PaymentMethodOptionsOxxo(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsP24`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum P24AnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsP24(PaymentMethodOptionsP24),
}

impl P24AnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let P24AnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_p24(&self) -> Option<&PaymentMethodOptionsP24> {
        if let P24AnyOf::PaymentMethodOptionsP24(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsPaynow`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum PaynowAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsPaynow(PaymentMethodOptionsPaynow),
}

impl PaynowAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let PaynowAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_paynow(&self) -> Option<&PaymentMethodOptionsPaynow> {
        if let PaynowAnyOf::PaymentMethodOptionsPaynow(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentIntentMethodOptionsSepaDebit`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SepaDebitAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentIntentMethodOptionsSepaDebit(PaymentIntentMethodOptionsSepaDebit),
}

impl SepaDebitAnyOf {
    pub fn payment_intent_method_options_sepa_debit(
        &self,
    ) -> Option<&PaymentIntentMethodOptionsSepaDebit> {
        if let SepaDebitAnyOf::PaymentIntentMethodOptionsSepaDebit(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let SepaDebitAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsSofort`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SofortAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsSofort(PaymentMethodOptionsSofort),
}

impl SofortAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let SofortAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_sofort(&self) -> Option<&PaymentMethodOptionsSofort> {
        if let SofortAnyOf::PaymentMethodOptionsSofort(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentIntentMethodOptionsUsBankAccount`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum UsBankAccountAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentIntentMethodOptionsUsBankAccount(PaymentIntentMethodOptionsUsBankAccount),
}

impl UsBankAccountAnyOf {
    pub fn payment_intent_method_options_us_bank_account(
        &self,
    ) -> Option<&PaymentIntentMethodOptionsUsBankAccount> {
        if let UsBankAccountAnyOf::PaymentIntentMethodOptionsUsBankAccount(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let UsBankAccountAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

/// All of the following types:
///
/// - `PaymentIntentTypeSpecificMethodOptionsClient`
/// - `PaymentMethodOptionsWechatPay`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum WechatPayAnyOf {
    /**
    *
    */
    PaymentIntentTypeSpecificMethodOptionsClient(PaymentIntentTypeSpecificMethodOptionsClient),
    /**
    *
    */
    PaymentMethodOptionsWechatPay(PaymentMethodOptionsWechatPay),
}

impl WechatPayAnyOf {
    pub fn payment_intent_type_specific_method_options_client(
        &self,
    ) -> Option<&PaymentIntentTypeSpecificMethodOptionsClient> {
        if let WechatPayAnyOf::PaymentIntentTypeSpecificMethodOptionsClient(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn payment_method_options_wechat_pay(&self) -> Option<&PaymentMethodOptionsWechatPay> {
        if let WechatPayAnyOf::PaymentMethodOptionsWechatPay(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsData {
    /**
    *
    */
    #[serde()]
    pub acss_debit: Box<Option<AcssDebitAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub afterpay_clearpay: Box<Option<AfterpayClearpayAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub alipay: Box<Option<AlipayAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub au_becs_debit: Box<Option<AuBecsDebitAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub bacs_debit: Box<Option<BacsDebitAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub bancontact: Box<Option<BancontactAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub boleto: Box<Option<BoletoAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub card: Box<Option<PaymentIntentMethodOptionsCardAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub card_present: Box<Option<CardPresentAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub eps: Box<Option<EpsAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub fpx: Box<Option<FpxAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub giropay: Box<Option<GiropayAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub grabpay: Box<Option<GrabpayAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub ideal: Box<Option<IdealAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub interac_present: Box<Option<InteracPresentAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub klarna: Box<Option<KlarnaAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub konbini: Box<Option<KonbiniAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub oxxo: Box<Option<OxxoAnyOf>>,
    /**
    *
    */
    #[serde(rename = "p24")]
    pub p_24: Box<Option<P24AnyOf>>,
    /**
    *
    */
    #[serde()]
    pub paynow: Box<Option<PaynowAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub sepa_debit: Box<Option<SepaDebitAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub sofort: Box<Option<SofortAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub us_bank_account: Box<Option<UsBankAccountAnyOf>>,
    /**
    *
    */
    #[serde()]
    pub wechat_pay: Box<Option<WechatPayAnyOf>>,
}

/**
* Indicates that you intend to make future payments with this PaymentIntent's payment method.
*   
*   Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
*   
*   When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentMethodOptionsAcssDebitSetupFutureUsage {
    #[serde(rename = "none")]
    None,
    #[serde(rename = "off_session")]
    OffSession,
    #[serde(rename = "on_session")]
    OnSession,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentMethodOptionsAcssDebitSetupFutureUsage {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::None => "none",
            PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::OffSession => "off_session",
            PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::OnSession => "on_session",
            PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::Noop => "",
            PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentMethodOptionsAcssDebitSetupFutureUsage {
    fn default() -> PaymentIntentMethodOptionsAcssDebitSetupFutureUsage {
        PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::Noop
    }
}
impl PaymentIntentMethodOptionsAcssDebitSetupFutureUsage {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            PaymentIntentMethodOptionsAcssDebitSetupFutureUsage::Noop
        )
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsAcssDebit {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub mandate_options: Option<PaymentIntentMethodOptionsMandateAcssDebit>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsAcssDebitSetupFutureUsage>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<VerificationMethod>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsAuBecsDebit {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsAcssDebitSetupFutureUsage>,
}

/**
* Controls when the funds will be captured from the customer's account.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentMethodOptionsCardCapture {
    #[serde(rename = "manual")]
    Manual,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentMethodOptionsCardCapture {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentMethodOptionsCardCapture::Manual => "manual",
            PaymentIntentMethodOptionsCardCapture::Noop => "",
            PaymentIntentMethodOptionsCardCapture::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentMethodOptionsCardCapture {
    fn default() -> PaymentIntentMethodOptionsCardCapture {
        PaymentIntentMethodOptionsCardCapture::Noop
    }
}
impl PaymentIntentMethodOptionsCardCapture {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentIntentMethodOptionsCardCapture::Noop)
    }
}

/**
* Selected network to process this payment intent on. Depends on the available networks of the card attached to the payment intent. Can be only set confirm-time.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentMethodOptionsCardNetwork {
    #[serde(rename = "amex")]
    Amex,
    #[serde(rename = "cartes_bancaires")]
    CartesBancaires,
    #[serde(rename = "diners")]
    Diners,
    #[serde(rename = "discover")]
    Discover,
    #[serde(rename = "interac")]
    Interac,
    #[serde(rename = "jcb")]
    Jcb,
    #[serde(rename = "mastercard")]
    Mastercard,
    #[serde(rename = "unionpay")]
    Unionpay,
    #[serde(rename = "unknown")]
    Unknown,
    #[serde(rename = "visa")]
    Visa,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentMethodOptionsCardNetwork {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentMethodOptionsCardNetwork::Amex => "amex",
            PaymentIntentMethodOptionsCardNetwork::CartesBancaires => "cartes_bancaires",
            PaymentIntentMethodOptionsCardNetwork::Diners => "diners",
            PaymentIntentMethodOptionsCardNetwork::Discover => "discover",
            PaymentIntentMethodOptionsCardNetwork::Interac => "interac",
            PaymentIntentMethodOptionsCardNetwork::Jcb => "jcb",
            PaymentIntentMethodOptionsCardNetwork::Mastercard => "mastercard",
            PaymentIntentMethodOptionsCardNetwork::Unionpay => "unionpay",
            PaymentIntentMethodOptionsCardNetwork::Unknown => "unknown",
            PaymentIntentMethodOptionsCardNetwork::Visa => "visa",
            PaymentIntentMethodOptionsCardNetwork::Noop => "",
            PaymentIntentMethodOptionsCardNetwork::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentMethodOptionsCardNetwork {
    fn default() -> PaymentIntentMethodOptionsCardNetwork {
        PaymentIntentMethodOptionsCardNetwork::Noop
    }
}
impl PaymentIntentMethodOptionsCardNetwork {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentIntentMethodOptionsCardNetwork::Noop)
    }
}

/**
* We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Permitted values include: `automatic` or `any`. If not provided, defaults to `automatic`. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentMethodOptionsCardRequestThreeDSecure {
    #[serde(rename = "any")]
    Any,
    #[serde(rename = "automatic")]
    Automatic,
    #[serde(rename = "challenge_only")]
    ChallengeOnly,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentMethodOptionsCardRequestThreeDSecure {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentMethodOptionsCardRequestThreeDSecure::Any => "any",
            PaymentIntentMethodOptionsCardRequestThreeDSecure::Automatic => "automatic",
            PaymentIntentMethodOptionsCardRequestThreeDSecure::ChallengeOnly => "challenge_only",
            PaymentIntentMethodOptionsCardRequestThreeDSecure::Noop => "",
            PaymentIntentMethodOptionsCardRequestThreeDSecure::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentMethodOptionsCardRequestThreeDSecure {
    fn default() -> PaymentIntentMethodOptionsCardRequestThreeDSecure {
        PaymentIntentMethodOptionsCardRequestThreeDSecure::Noop
    }
}
impl PaymentIntentMethodOptionsCardRequestThreeDSecure {
    pub fn is_noop(&self) -> bool {
        matches!(
            self,
            PaymentIntentMethodOptionsCardRequestThreeDSecure::Noop
        )
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsCard {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub capture_method: Option<PaymentIntentMethodOptionsCardCapture>,
    /**
    * Installment details for this payment (Mexico only).
    *  
    *  For more information, see the [installments integration guide](https://stripe.com/docs/payments/installments).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub installments: Option<PaymentMethodOptionsCardInstallments>,
    /**
    * Configuration options for setting up an eMandate for cards issued in India.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub mandate_options: Option<PaymentMethodOptionsCardMandate>,
    /**
    * Selected network to process this payment intent on. Depends on the available networks of the card attached to the payment intent. Can be only set confirm-time.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub network: Option<PaymentIntentMethodOptionsCardNetwork>,
    /**
    * We strongly recommend that you rely on our SCA Engine to automatically prompt your customers for authentication based on risk level and [other requirements](https://stripe.com/docs/strong-customer-authentication). However, if you wish to request 3D Secure based on logic from your own fraud engine, provide this option. Permitted values include: `automatic` or `any`. If not provided, defaults to `automatic`. Read our guide on [manually requesting 3D Secure](https://stripe.com/docs/payments/3d-secure#manual-three-ds) for more information on how this configuration interacts with Radar and our SCA Engine.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub request_three_d_secure: Option<PaymentIntentMethodOptionsCardRequestThreeDSecure>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsAcssDebitSetupFutureUsage>,
}

/**
* Indicates that you intend to make future payments with this PaymentIntent's payment method.
*   
*   Providing this parameter will [attach the payment method](https://stripe.com/docs/payments/save-during-payment) to the PaymentIntent's Customer, if present, after the PaymentIntent is confirmed and any required actions from the user are complete. If no Customer was provided, the payment method can still be [attached](https://stripe.com/docs/api/payment_methods/attach) to a Customer after the transaction completes.
*   
*   When processing card payments, Stripe also uses `setup_future_usage` to dynamically optimize your payment flow and comply with regional legislation and network rules, such as [SCA](https://stripe.com/docs/strong-customer-authentication).
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentIntentMethodOptionsEpsSetupFutureUsage {
    #[serde(rename = "none")]
    None,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentIntentMethodOptionsEpsSetupFutureUsage {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentIntentMethodOptionsEpsSetupFutureUsage::None => "none",
            PaymentIntentMethodOptionsEpsSetupFutureUsage::Noop => "",
            PaymentIntentMethodOptionsEpsSetupFutureUsage::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentIntentMethodOptionsEpsSetupFutureUsage {
    fn default() -> PaymentIntentMethodOptionsEpsSetupFutureUsage {
        PaymentIntentMethodOptionsEpsSetupFutureUsage::Noop
    }
}
impl PaymentIntentMethodOptionsEpsSetupFutureUsage {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentIntentMethodOptionsEpsSetupFutureUsage::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsEps {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsEpsSetupFutureUsage>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsMandateAcssDebit {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub custom_mandate_url: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub interval_description: String,
    /**
    * Payment schedule for the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_schedule: Option<PaymentSchedule>,
    /**
    * Transaction type of the mandate.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transaction_type: Option<TransactionType>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsSepaDebit {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub mandate_options: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsAcssDebitSetupFutureUsage>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentMethodOptionsUsBankAccount {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsAcssDebitSetupFutureUsage>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<VerificationMethod>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentProcessing {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card: Option<PaymentIntentCardProcessing>,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "CardObject::is_noop", rename = "type")]
    pub type_: CardObject,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentProcessingCustomerNotification {
    /**
    *
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub approval_requested: bool,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub completes_at: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentIntentTypeSpecificMethodOptionsClient {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub capture_method: Option<PaymentIntentMethodOptionsCardCapture>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub setup_future_usage: Option<PaymentIntentMethodOptionsAcssDebitSetupFutureUsage>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub verification_method: Option<VerificationMethod>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentLinkObject {
    #[serde(rename = "payment_link")]
    PaymentLink,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentLinkObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentLinkObject::PaymentLink => "payment_link",
            PaymentLinkObject::Noop => "",
            PaymentLinkObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentLinkObject {
    fn default() -> PaymentLinkObject {
        PaymentLinkObject::Noop
    }
}
impl PaymentLinkObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentLinkObject::Noop)
    }
}

/// A payment link is a shareable URL that will take your customers to a hosted payment page. A payment link can be shared and used multiple times.
///
/// When a customer opens a payment link it will open a new [checkout session](https://stripe.com/docs/api/checkout/sessions) to render the payment page. You can use [checkout session events](https://stripe.com/docs/api/events/types#event_types-checkout.session.completed) to track payments through payment links.
///
/// Related guide: [Payment Links API](https://stripe.com/docs/payments/payment-links/api)
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentLink {
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub active: bool,
    /**
    *
    */
    #[serde()]
    pub after_completion: PaymentLinksResourceAfterCompletion,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub allow_promotion_codes: bool,
    /**
    * A payment link is a shareable URL that will take your customers to a hosted payment page. A payment link can be shared and used multiple times.
    *  
    *  When a customer opens a payment link it will open a new [checkout session](https://stripe.com/docs/api/checkout/sessions) to render the payment page. You can use [checkout session events](https://stripe.com/docs/api/events/types#event_types-checkout.session.completed) to track payments through payment links.
    *  
    *  Related guide: [Payment Links API](https://stripe.com/docs/payments/payment-links/api)
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub application_fee_amount: i64,
    /**
    * The exchange rate used, if applicable, for this transaction. Specifically, if money was converted from currency A to currency B, then the `amount` in currency A, times `exchange_rate`, would be the `amount` in currency B. For example, suppose you charged a customer 10.00 EUR. Then the PaymentIntent's `amount` would be `1000` and `currency` would be `eur`. Suppose this was converted into 12.34 USD in your Stripe account. Then the BalanceTransaction's `amount` would be `1234`, `currency` would be `usd`, and `exchange_rate` would be `1.234`.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_f64",
        deserialize_with = "crate::utils::deserialize_null_f64::deserialize"
    )]
    pub application_fee_percent: f64,
    /**
    *
    */
    #[serde()]
    pub automatic_tax: PortalInvoiceList,
    /**
    * Describes whether Checkout should collect the customer's billing address.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub billing_address_collection: Option<BillingAddressCollection>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * A payment link is a shareable URL that will take your customers to a hosted payment page. A payment link can be shared and used multiple times.
    *  
    *  When a customer opens a payment link it will open a new [checkout session](https://stripe.com/docs/api/checkout/sessions) to render the payment page. You can use [checkout session events](https://stripe.com/docs/api/events/types#event_types-checkout.session.completed) to track payments through payment links.
    *  
    *  Related guide: [Payment Links API](https://stripe.com/docs/payments/payment-links/api)
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub line_items: Option<LineItems>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * Set of [key-value pairs](https://stripe.com/docs/api/metadata) that you can attach to an object. This can be useful for storing additional information about the object in a structured format.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "PaymentLinkObject::is_noop")]
    pub object: PaymentLinkObject,
    /**
    * A payment link is a shareable URL that will take your customers to a hosted payment page. A payment link can be shared and used multiple times.
    *  
    *  When a customer opens a payment link it will open a new [checkout session](https://stripe.com/docs/api/checkout/sessions) to render the payment page. You can use [checkout session events](https://stripe.com/docs/api/events/types#event_types-checkout.session.completed) to track payments through payment links.
    *  
    *  Related guide: [Payment Links API](https://stripe.com/docs/payments/payment-links/api)
    */
    #[serde()]
    pub on_behalf_of: Box<Option<AccountAnyOf>>,
    /**
    * The list of payment method types that customers can use. When `null`, Stripe will dynamically show relevant payment methods you've enabled in your [payment method settings](https://dashboard.stripe.com/settings/payment_methods).
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub payment_method_types: Vec<CardObject>,
    /**
    *
    */
    #[serde()]
    pub phone_number_collection: PortalInvoiceList,
    /**
    * Configuration for collecting the customer's shipping address.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub shipping_address_collection: Option<PaymentLinksResourceShippingAddressCollection>,
    /**
    * When creating a subscription, the specified configuration data will be used. There must be at least one line item with a recurring price to use `subscription_data`.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub subscription_data: Option<PaymentLinksResourceSubscriptionData>,
    /**
    * The account (if any) the payments will be attributed to for tax reporting, and where funds from each payment will be transferred to.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub transfer_data: Option<PaymentLinksResourceTransferData>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub url: String,
}

/**
* The specified behavior after the purchase is complete.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentLinksResourceAfterCompletionType {
    #[serde(rename = "hosted_confirmation")]
    HostedConfirmation,
    #[serde(rename = "redirect")]
    Redirect,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentLinksResourceAfterCompletionType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentLinksResourceAfterCompletionType::HostedConfirmation => "hosted_confirmation",
            PaymentLinksResourceAfterCompletionType::Redirect => "redirect",
            PaymentLinksResourceAfterCompletionType::Noop => "",
            PaymentLinksResourceAfterCompletionType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentLinksResourceAfterCompletionType {
    fn default() -> PaymentLinksResourceAfterCompletionType {
        PaymentLinksResourceAfterCompletionType::Noop
    }
}
impl PaymentLinksResourceAfterCompletionType {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentLinksResourceAfterCompletionType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentLinksResourceAfterCompletion {
    /**
    *
    */
    #[serde(default)]
    pub hosted_confirmation: PaymentLinksResourceCompletionBehaviorConfirmationPage,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub redirect: Option<MandateAuBecsDebit>,
    /**
    * The specified behavior after the purchase is complete.
    */
    #[serde(
        default,
        skip_serializing_if = "PaymentLinksResourceAfterCompletionType::is_noop",
        rename = "type"
    )]
    pub type_: PaymentLinksResourceAfterCompletionType,
}

///
#[derive(Serialize, Default, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentLinksResourceCompletionBehaviorConfirmationPage {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub custom_message: String,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum AllowedCountries {
    #[serde(rename = "AC")]
    Ac,
    #[serde(rename = "AD")]
    Ad,
    #[serde(rename = "AE")]
    Ae,
    #[serde(rename = "AF")]
    Af,
    #[serde(rename = "AG")]
    Ag,
    #[serde(rename = "AI")]
    Ai,
    #[serde(rename = "AL")]
    Al,
    #[serde(rename = "AM")]
    Am,
    #[serde(rename = "AO")]
    Ao,
    #[serde(rename = "AQ")]
    Aq,
    #[serde(rename = "AR")]
    Ar,
    #[serde(rename = "AT")]
    At,
    #[serde(rename = "AU")]
    Au,
    #[serde(rename = "AW")]
    Aw,
    #[serde(rename = "AX")]
    Ax,
    #[serde(rename = "AZ")]
    Az,
    #[serde(rename = "BA")]
    Ba,
    #[serde(rename = "BB")]
    Bb,
    #[serde(rename = "BD")]
    Bd,
    #[serde(rename = "BE")]
    Be,
    #[serde(rename = "BF")]
    Bf,
    #[serde(rename = "BG")]
    Bg,
    #[serde(rename = "BH")]
    Bh,
    #[serde(rename = "BI")]
    Bi,
    #[serde(rename = "BJ")]
    Bj,
    #[serde(rename = "BL")]
    Bl,
    #[serde(rename = "BM")]
    Bm,
    #[serde(rename = "BN")]
    Bn,
    #[serde(rename = "BO")]
    Bo,
    #[serde(rename = "BQ")]
    Bq,
    #[serde(rename = "BR")]
    Br,
    #[serde(rename = "BS")]
    Bs,
    #[serde(rename = "BT")]
    Bt,
    #[serde(rename = "BV")]
    Bv,
    #[serde(rename = "BW")]
    Bw,
    #[serde(rename = "BY")]
    By,
    #[serde(rename = "BZ")]
    Bz,
    #[serde(rename = "CA")]
    Ca,
    #[serde(rename = "CD")]
    Cd,
    #[serde(rename = "CF")]
    Cf,
    #[serde(rename = "CG")]
    Cg,
    #[serde(rename = "CH")]
    Ch,
    #[serde(rename = "CI")]
    Ci,
    #[serde(rename = "CK")]
    Ck,
    #[serde(rename = "CL")]
    Cl,
    #[serde(rename = "CM")]
    Cm,
    #[serde(rename = "CN")]
    Cn,
    #[serde(rename = "CO")]
    Co,
    #[serde(rename = "CR")]
    Cr,
    #[serde(rename = "CV")]
    Cv,
    #[serde(rename = "CW")]
    Cw,
    #[serde(rename = "CY")]
    Cy,
    #[serde(rename = "CZ")]
    Cz,
    #[serde(rename = "DE")]
    De,
    #[serde(rename = "DJ")]
    Dj,
    #[serde(rename = "DK")]
    Dk,
    #[serde(rename = "DM")]
    Dm,
    #[serde(rename = "DO")]
    Do,
    #[serde(rename = "DZ")]
    Dz,
    #[serde(rename = "EC")]
    Ec,
    #[serde(rename = "EE")]
    Ee,
    #[serde(rename = "EG")]
    Eg,
    #[serde(rename = "EH")]
    Eh,
    #[serde(rename = "ER")]
    Er,
    #[serde(rename = "ES")]
    Es,
    #[serde(rename = "ET")]
    Et,
    #[serde(rename = "FI")]
    Fi,
    #[serde(rename = "FJ")]
    Fj,
    #[serde(rename = "FK")]
    Fk,
    #[serde(rename = "FO")]
    Fo,
    #[serde(rename = "FR")]
    Fr,
    #[serde(rename = "GA")]
    Ga,
    #[serde(rename = "GB")]
    Gb,
    #[serde(rename = "GD")]
    Gd,
    #[serde(rename = "GE")]
    Ge,
    #[serde(rename = "GF")]
    Gf,
    #[serde(rename = "GG")]
    Gg,
    #[serde(rename = "GH")]
    Gh,
    #[serde(rename = "GI")]
    Gi,
    #[serde(rename = "GL")]
    Gl,
    #[serde(rename = "GM")]
    Gm,
    #[serde(rename = "GN")]
    Gn,
    #[serde(rename = "GP")]
    Gp,
    #[serde(rename = "GQ")]
    Gq,
    #[serde(rename = "GR")]
    Gr,
    #[serde(rename = "GS")]
    Gs,
    #[serde(rename = "GT")]
    Gt,
    #[serde(rename = "GU")]
    Gu,
    #[serde(rename = "GW")]
    Gw,
    #[serde(rename = "GY")]
    Gy,
    #[serde(rename = "HK")]
    Hk,
    #[serde(rename = "HN")]
    Hn,
    #[serde(rename = "HR")]
    Hr,
    #[serde(rename = "HT")]
    Ht,
    #[serde(rename = "HU")]
    Hu,
    #[serde(rename = "ID")]
    Id,
    #[serde(rename = "IE")]
    Ie,
    #[serde(rename = "IL")]
    Il,
    #[serde(rename = "IM")]
    Im,
    #[serde(rename = "IN")]
    In,
    #[serde(rename = "IO")]
    Io,
    #[serde(rename = "IQ")]
    Iq,
    #[serde(rename = "IS")]
    Is,
    #[serde(rename = "IT")]
    It,
    #[serde(rename = "JE")]
    Je,
    #[serde(rename = "JM")]
    Jm,
    #[serde(rename = "JO")]
    Jo,
    #[serde(rename = "JP")]
    Jp,
    #[serde(rename = "KE")]
    Ke,
    #[serde(rename = "KG")]
    Kg,
    #[serde(rename = "KH")]
    Kh,
    #[serde(rename = "KI")]
    Ki,
    #[serde(rename = "KM")]
    Km,
    #[serde(rename = "KN")]
    Kn,
    #[serde(rename = "KR")]
    Kr,
    #[serde(rename = "KW")]
    Kw,
    #[serde(rename = "KY")]
    Ky,
    #[serde(rename = "KZ")]
    Kz,
    #[serde(rename = "LA")]
    La,
    #[serde(rename = "LB")]
    Lb,
    #[serde(rename = "LC")]
    Lc,
    #[serde(rename = "LI")]
    Li,
    #[serde(rename = "LK")]
    Lk,
    #[serde(rename = "LR")]
    Lr,
    #[serde(rename = "LS")]
    Ls,
    #[serde(rename = "LT")]
    Lt,
    #[serde(rename = "LU")]
    Lu,
    #[serde(rename = "LV")]
    Lv,
    #[serde(rename = "LY")]
    Ly,
    #[serde(rename = "MA")]
    Ma,
    #[serde(rename = "MC")]
    Mc,
    #[serde(rename = "MD")]
    Md,
    #[serde(rename = "ME")]
    Me,
    #[serde(rename = "MF")]
    Mf,
    #[serde(rename = "MG")]
    Mg,
    #[serde(rename = "MK")]
    Mk,
    #[serde(rename = "ML")]
    Ml,
    #[serde(rename = "MM")]
    Mm,
    #[serde(rename = "MN")]
    Mn,
    #[serde(rename = "MO")]
    Mo,
    #[serde(rename = "MQ")]
    Mq,
    #[serde(rename = "MR")]
    Mr,
    #[serde(rename = "MS")]
    Ms,
    #[serde(rename = "MT")]
    Mt,
    #[serde(rename = "MU")]
    Mu,
    #[serde(rename = "MV")]
    Mv,
    #[serde(rename = "MW")]
    Mw,
    #[serde(rename = "MX")]
    Mx,
    #[serde(rename = "MY")]
    My,
    #[serde(rename = "MZ")]
    Mz,
    #[serde(rename = "NA")]
    Na,
    #[serde(rename = "NC")]
    Nc,
    #[serde(rename = "NE")]
    Ne,
    #[serde(rename = "NG")]
    Ng,
    #[serde(rename = "NI")]
    Ni,
    #[serde(rename = "NL")]
    Nl,
    #[serde(rename = "NO")]
    No,
    #[serde(rename = "NP")]
    Np,
    #[serde(rename = "NR")]
    Nr,
    #[serde(rename = "NU")]
    Nu,
    #[serde(rename = "NZ")]
    Nz,
    #[serde(rename = "OM")]
    Om,
    #[serde(rename = "PA")]
    Pa,
    #[serde(rename = "PE")]
    Pe,
    #[serde(rename = "PF")]
    Pf,
    #[serde(rename = "PG")]
    Pg,
    #[serde(rename = "PH")]
    Ph,
    #[serde(rename = "PK")]
    Pk,
    #[serde(rename = "PL")]
    Pl,
    #[serde(rename = "PM")]
    Pm,
    #[serde(rename = "PN")]
    Pn,
    #[serde(rename = "PR")]
    Pr,
    #[serde(rename = "PS")]
    Ps,
    #[serde(rename = "PT")]
    Pt,
    #[serde(rename = "PY")]
    Py,
    #[serde(rename = "QA")]
    Qa,
    #[serde(rename = "RE")]
    Re,
    #[serde(rename = "RO")]
    Ro,
    #[serde(rename = "RS")]
    Rs,
    #[serde(rename = "RU")]
    Ru,
    #[serde(rename = "RW")]
    Rw,
    #[serde(rename = "SA")]
    Sa,
    #[serde(rename = "SB")]
    Sb,
    #[serde(rename = "SC")]
    Sc,
    #[serde(rename = "SE")]
    Se,
    #[serde(rename = "SG")]
    Sg,
    #[serde(rename = "SH")]
    Sh,
    #[serde(rename = "SI")]
    Si,
    #[serde(rename = "SJ")]
    Sj,
    #[serde(rename = "SK")]
    Sk,
    #[serde(rename = "SL")]
    Sl,
    #[serde(rename = "SM")]
    Sm,
    #[serde(rename = "SN")]
    Sn,
    #[serde(rename = "SO")]
    So,
    #[serde(rename = "SR")]
    Sr,
    #[serde(rename = "SS")]
    Ss,
    #[serde(rename = "ST")]
    St,
    #[serde(rename = "SV")]
    Sv,
    #[serde(rename = "SX")]
    Sx,
    #[serde(rename = "SZ")]
    Sz,
    #[serde(rename = "TA")]
    Ta,
    #[serde(rename = "TC")]
    Tc,
    #[serde(rename = "TD")]
    Td,
    #[serde(rename = "TF")]
    Tf,
    #[serde(rename = "TG")]
    Tg,
    #[serde(rename = "TH")]
    Th,
    #[serde(rename = "TJ")]
    Tj,
    #[serde(rename = "TK")]
    Tk,
    #[serde(rename = "TL")]
    Tl,
    #[serde(rename = "TM")]
    Tm,
    #[serde(rename = "TN")]
    Tn,
    #[serde(rename = "TO")]
    To,
    #[serde(rename = "TR")]
    Tr,
    #[serde(rename = "TT")]
    Tt,
    #[serde(rename = "TV")]
    Tv,
    #[serde(rename = "TW")]
    Tw,
    #[serde(rename = "TZ")]
    Tz,
    #[serde(rename = "UA")]
    Ua,
    #[serde(rename = "UG")]
    Ug,
    #[serde(rename = "US")]
    Us,
    #[serde(rename = "UY")]
    Uy,
    #[serde(rename = "UZ")]
    Uz,
    #[serde(rename = "VA")]
    Va,
    #[serde(rename = "VC")]
    Vc,
    #[serde(rename = "VE")]
    Ve,
    #[serde(rename = "VG")]
    Vg,
    #[serde(rename = "VN")]
    Vn,
    #[serde(rename = "VU")]
    Vu,
    #[serde(rename = "WF")]
    Wf,
    #[serde(rename = "WS")]
    Ws,
    #[serde(rename = "XK")]
    Xk,
    #[serde(rename = "YE")]
    Ye,
    #[serde(rename = "YT")]
    Yt,
    #[serde(rename = "ZA")]
    Za,
    #[serde(rename = "ZM")]
    Zm,
    #[serde(rename = "ZW")]
    Zw,
    #[serde(rename = "ZZ")]
    Zz,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for AllowedCountries {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            AllowedCountries::Ac => "AC",
            AllowedCountries::Ad => "AD",
            AllowedCountries::Ae => "AE",
            AllowedCountries::Af => "AF",
            AllowedCountries::Ag => "AG",
            AllowedCountries::Ai => "AI",
            AllowedCountries::Al => "AL",
            AllowedCountries::Am => "AM",
            AllowedCountries::Ao => "AO",
            AllowedCountries::Aq => "AQ",
            AllowedCountries::Ar => "AR",
            AllowedCountries::At => "AT",
            AllowedCountries::Au => "AU",
            AllowedCountries::Aw => "AW",
            AllowedCountries::Ax => "AX",
            AllowedCountries::Az => "AZ",
            AllowedCountries::Ba => "BA",
            AllowedCountries::Bb => "BB",
            AllowedCountries::Bd => "BD",
            AllowedCountries::Be => "BE",
            AllowedCountries::Bf => "BF",
            AllowedCountries::Bg => "BG",
            AllowedCountries::Bh => "BH",
            AllowedCountries::Bi => "BI",
            AllowedCountries::Bj => "BJ",
            AllowedCountries::Bl => "BL",
            AllowedCountries::Bm => "BM",
            AllowedCountries::Bn => "BN",
            AllowedCountries::Bo => "BO",
            AllowedCountries::Bq => "BQ",
            AllowedCountries::Br => "BR",
            AllowedCountries::Bs => "BS",
            AllowedCountries::Bt => "BT",
            AllowedCountries::Bv => "BV",
            AllowedCountries::Bw => "BW",
            AllowedCountries::By => "BY",
            AllowedCountries::Bz => "BZ",
            AllowedCountries::Ca => "CA",
            AllowedCountries::Cd => "CD",
            AllowedCountries::Cf => "CF",
            AllowedCountries::Cg => "CG",
            AllowedCountries::Ch => "CH",
            AllowedCountries::Ci => "CI",
            AllowedCountries::Ck => "CK",
            AllowedCountries::Cl => "CL",
            AllowedCountries::Cm => "CM",
            AllowedCountries::Cn => "CN",
            AllowedCountries::Co => "CO",
            AllowedCountries::Cr => "CR",
            AllowedCountries::Cv => "CV",
            AllowedCountries::Cw => "CW",
            AllowedCountries::Cy => "CY",
            AllowedCountries::Cz => "CZ",
            AllowedCountries::De => "DE",
            AllowedCountries::Dj => "DJ",
            AllowedCountries::Dk => "DK",
            AllowedCountries::Dm => "DM",
            AllowedCountries::Do => "DO",
            AllowedCountries::Dz => "DZ",
            AllowedCountries::Ec => "EC",
            AllowedCountries::Ee => "EE",
            AllowedCountries::Eg => "EG",
            AllowedCountries::Eh => "EH",
            AllowedCountries::Er => "ER",
            AllowedCountries::Es => "ES",
            AllowedCountries::Et => "ET",
            AllowedCountries::Fi => "FI",
            AllowedCountries::Fj => "FJ",
            AllowedCountries::Fk => "FK",
            AllowedCountries::Fo => "FO",
            AllowedCountries::Fr => "FR",
            AllowedCountries::Ga => "GA",
            AllowedCountries::Gb => "GB",
            AllowedCountries::Gd => "GD",
            AllowedCountries::Ge => "GE",
            AllowedCountries::Gf => "GF",
            AllowedCountries::Gg => "GG",
            AllowedCountries::Gh => "GH",
            AllowedCountries::Gi => "GI",
            AllowedCountries::Gl => "GL",
            AllowedCountries::Gm => "GM",
            AllowedCountries::Gn => "GN",
            AllowedCountries::Gp => "GP",
            AllowedCountries::Gq => "GQ",
            AllowedCountries::Gr => "GR",
            AllowedCountries::Gs => "GS",
            AllowedCountries::Gt => "GT",
            AllowedCountries::Gu => "GU",
            AllowedCountries::Gw => "GW",
            AllowedCountries::Gy => "GY",
            AllowedCountries::Hk => "HK",
            AllowedCountries::Hn => "HN",
            AllowedCountries::Hr => "HR",
            AllowedCountries::Ht => "HT",
            AllowedCountries::Hu => "HU",
            AllowedCountries::Id => "ID",
            AllowedCountries::Ie => "IE",
            AllowedCountries::Il => "IL",
            AllowedCountries::Im => "IM",
            AllowedCountries::In => "IN",
            AllowedCountries::Io => "IO",
            AllowedCountries::Iq => "IQ",
            AllowedCountries::Is => "IS",
            AllowedCountries::It => "IT",
            AllowedCountries::Je => "JE",
            AllowedCountries::Jm => "JM",
            AllowedCountries::Jo => "JO",
            AllowedCountries::Jp => "JP",
            AllowedCountries::Ke => "KE",
            AllowedCountries::Kg => "KG",
            AllowedCountries::Kh => "KH",
            AllowedCountries::Ki => "KI",
            AllowedCountries::Km => "KM",
            AllowedCountries::Kn => "KN",
            AllowedCountries::Kr => "KR",
            AllowedCountries::Kw => "KW",
            AllowedCountries::Ky => "KY",
            AllowedCountries::Kz => "KZ",
            AllowedCountries::La => "LA",
            AllowedCountries::Lb => "LB",
            AllowedCountries::Lc => "LC",
            AllowedCountries::Li => "LI",
            AllowedCountries::Lk => "LK",
            AllowedCountries::Lr => "LR",
            AllowedCountries::Ls => "LS",
            AllowedCountries::Lt => "LT",
            AllowedCountries::Lu => "LU",
            AllowedCountries::Lv => "LV",
            AllowedCountries::Ly => "LY",
            AllowedCountries::Ma => "MA",
            AllowedCountries::Mc => "MC",
            AllowedCountries::Md => "MD",
            AllowedCountries::Me => "ME",
            AllowedCountries::Mf => "MF",
            AllowedCountries::Mg => "MG",
            AllowedCountries::Mk => "MK",
            AllowedCountries::Ml => "ML",
            AllowedCountries::Mm => "MM",
            AllowedCountries::Mn => "MN",
            AllowedCountries::Mo => "MO",
            AllowedCountries::Mq => "MQ",
            AllowedCountries::Mr => "MR",
            AllowedCountries::Ms => "MS",
            AllowedCountries::Mt => "MT",
            AllowedCountries::Mu => "MU",
            AllowedCountries::Mv => "MV",
            AllowedCountries::Mw => "MW",
            AllowedCountries::Mx => "MX",
            AllowedCountries::My => "MY",
            AllowedCountries::Mz => "MZ",
            AllowedCountries::Na => "NA",
            AllowedCountries::Nc => "NC",
            AllowedCountries::Ne => "NE",
            AllowedCountries::Ng => "NG",
            AllowedCountries::Ni => "NI",
            AllowedCountries::Nl => "NL",
            AllowedCountries::No => "NO",
            AllowedCountries::Np => "NP",
            AllowedCountries::Nr => "NR",
            AllowedCountries::Nu => "NU",
            AllowedCountries::Nz => "NZ",
            AllowedCountries::Om => "OM",
            AllowedCountries::Pa => "PA",
            AllowedCountries::Pe => "PE",
            AllowedCountries::Pf => "PF",
            AllowedCountries::Pg => "PG",
            AllowedCountries::Ph => "PH",
            AllowedCountries::Pk => "PK",
            AllowedCountries::Pl => "PL",
            AllowedCountries::Pm => "PM",
            AllowedCountries::Pn => "PN",
            AllowedCountries::Pr => "PR",
            AllowedCountries::Ps => "PS",
            AllowedCountries::Pt => "PT",
            AllowedCountries::Py => "PY",
            AllowedCountries::Qa => "QA",
            AllowedCountries::Re => "RE",
            AllowedCountries::Ro => "RO",
            AllowedCountries::Rs => "RS",
            AllowedCountries::Ru => "RU",
            AllowedCountries::Rw => "RW",
            AllowedCountries::Sa => "SA",
            AllowedCountries::Sb => "SB",
            AllowedCountries::Sc => "SC",
            AllowedCountries::Se => "SE",
            AllowedCountries::Sg => "SG",
            AllowedCountries::Sh => "SH",
            AllowedCountries::Si => "SI",
            AllowedCountries::Sj => "SJ",
            AllowedCountries::Sk => "SK",
            AllowedCountries::Sl => "SL",
            AllowedCountries::Sm => "SM",
            AllowedCountries::Sn => "SN",
            AllowedCountries::So => "SO",
            AllowedCountries::Sr => "SR",
            AllowedCountries::Ss => "SS",
            AllowedCountries::St => "ST",
            AllowedCountries::Sv => "SV",
            AllowedCountries::Sx => "SX",
            AllowedCountries::Sz => "SZ",
            AllowedCountries::Ta => "TA",
            AllowedCountries::Tc => "TC",
            AllowedCountries::Td => "TD",
            AllowedCountries::Tf => "TF",
            AllowedCountries::Tg => "TG",
            AllowedCountries::Th => "TH",
            AllowedCountries::Tj => "TJ",
            AllowedCountries::Tk => "TK",
            AllowedCountries::Tl => "TL",
            AllowedCountries::Tm => "TM",
            AllowedCountries::Tn => "TN",
            AllowedCountries::To => "TO",
            AllowedCountries::Tr => "TR",
            AllowedCountries::Tt => "TT",
            AllowedCountries::Tv => "TV",
            AllowedCountries::Tw => "TW",
            AllowedCountries::Tz => "TZ",
            AllowedCountries::Ua => "UA",
            AllowedCountries::Ug => "UG",
            AllowedCountries::Us => "US",
            AllowedCountries::Uy => "UY",
            AllowedCountries::Uz => "UZ",
            AllowedCountries::Va => "VA",
            AllowedCountries::Vc => "VC",
            AllowedCountries::Ve => "VE",
            AllowedCountries::Vg => "VG",
            AllowedCountries::Vn => "VN",
            AllowedCountries::Vu => "VU",
            AllowedCountries::Wf => "WF",
            AllowedCountries::Ws => "WS",
            AllowedCountries::Xk => "XK",
            AllowedCountries::Ye => "YE",
            AllowedCountries::Yt => "YT",
            AllowedCountries::Za => "ZA",
            AllowedCountries::Zm => "ZM",
            AllowedCountries::Zw => "ZW",
            AllowedCountries::Zz => "ZZ",
            AllowedCountries::Noop => "",
            AllowedCountries::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for AllowedCountries {
    fn default() -> AllowedCountries {
        AllowedCountries::Noop
    }
}
impl AllowedCountries {
    pub fn is_noop(&self) -> bool {
        matches!(self, AllowedCountries::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentLinksResourceShippingAddressCollection {
    /**
    * An array of two-letter ISO country codes representing which countries Checkout should provide as options for shipping locations. Unsupported country codes: `AS, CX, CC, CU, HM, IR, KP, MH, FM, NF, MP, PW, SD, SY, UM, VI`.
    */
    #[serde(
        default,
        skip_serializing_if = "Vec::is_empty",
        deserialize_with = "crate::utils::deserialize_null_vector::deserialize"
    )]
    pub allowed_countries: Vec<AllowedCountries>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentLinksResourceSubscriptionData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub trial_period_days: i64,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentLinksResourceTransferData {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub amount: i64,
    #[serde()]
    pub destination: Box<AccountAnyOf>,
}

/**
* String representing the object's type. Objects of the same type share the same value.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentMethodObject {
    #[serde(rename = "payment_method")]
    PaymentMethod,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentMethodObject {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentMethodObject::PaymentMethod => "payment_method",
            PaymentMethodObject::Noop => "",
            PaymentMethodObject::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentMethodObject {
    fn default() -> PaymentMethodObject {
        PaymentMethodObject::Noop
    }
}
impl PaymentMethodObject {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentMethodObject::Noop)
    }
}

/**
* The type of the PaymentMethod. An additional hash is included on the PaymentMethod with a name matching this value. It contains additional information specific to the PaymentMethod type.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentMethodType {
    #[serde(rename = "acss_debit")]
    AcssDebit,
    #[serde(rename = "afterpay_clearpay")]
    AfterpayClearpay,
    #[serde(rename = "alipay")]
    Alipay,
    #[serde(rename = "au_becs_debit")]
    AuBecsDebit,
    #[serde(rename = "bacs_debit")]
    BacsDebit,
    #[serde(rename = "bancontact")]
    Bancontact,
    #[serde(rename = "boleto")]
    Boleto,
    #[serde(rename = "card")]
    Card,
    #[serde(rename = "card_present")]
    CardPresent,
    #[serde(rename = "eps")]
    Eps,
    #[serde(rename = "fpx")]
    Fpx,
    #[serde(rename = "giropay")]
    Giropay,
    #[serde(rename = "grabpay")]
    Grabpay,
    #[serde(rename = "ideal")]
    Ideal,
    #[serde(rename = "interac_present")]
    InteracPresent,
    #[serde(rename = "klarna")]
    Klarna,
    #[serde(rename = "konbini")]
    Konbini,
    #[serde(rename = "oxxo")]
    Oxxo,
    #[serde(rename = "p24")]
    P24,
    #[serde(rename = "paynow")]
    Paynow,
    #[serde(rename = "sepa_debit")]
    SepaDebit,
    #[serde(rename = "sofort")]
    Sofort,
    #[serde(rename = "us_bank_account")]
    UsBankAccount,
    #[serde(rename = "wechat_pay")]
    WechatPay,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentMethodType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentMethodType::AcssDebit => "acss_debit",
            PaymentMethodType::AfterpayClearpay => "afterpay_clearpay",
            PaymentMethodType::Alipay => "alipay",
            PaymentMethodType::AuBecsDebit => "au_becs_debit",
            PaymentMethodType::BacsDebit => "bacs_debit",
            PaymentMethodType::Bancontact => "bancontact",
            PaymentMethodType::Boleto => "boleto",
            PaymentMethodType::Card => "card",
            PaymentMethodType::CardPresent => "card_present",
            PaymentMethodType::Eps => "eps",
            PaymentMethodType::Fpx => "fpx",
            PaymentMethodType::Giropay => "giropay",
            PaymentMethodType::Grabpay => "grabpay",
            PaymentMethodType::Ideal => "ideal",
            PaymentMethodType::InteracPresent => "interac_present",
            PaymentMethodType::Klarna => "klarna",
            PaymentMethodType::Konbini => "konbini",
            PaymentMethodType::Oxxo => "oxxo",
            PaymentMethodType::P24 => "p24",
            PaymentMethodType::Paynow => "paynow",
            PaymentMethodType::SepaDebit => "sepa_debit",
            PaymentMethodType::Sofort => "sofort",
            PaymentMethodType::UsBankAccount => "us_bank_account",
            PaymentMethodType::WechatPay => "wechat_pay",
            PaymentMethodType::Noop => "",
            PaymentMethodType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentMethodType {
    fn default() -> PaymentMethodType {
        PaymentMethodType::Noop
    }
}
impl PaymentMethodType {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentMethodType::Noop)
    }
}

/// PaymentMethod objects represent your customer's payment instruments.
/// They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
/// Customer objects to store instrument details for future payments.
///
/// Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethod {
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub acss_debit: Option<PaymentMethodAcssDebit>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub afterpay_clearpay: Option<UseStripeSdk>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub alipay: Option<UseStripeSdk>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub au_becs_debit: Option<PaymentMethodAuBecsDebit>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bacs_debit: Option<PaymentMethodBacsDebit>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub bancontact: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde()]
    pub billing_details: BillingDetails,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub boleto: Option<PaymentMethodBoleto>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card: Option<PaymentMethodCard>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub card_present: Option<UseStripeSdk>,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub created: i64,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub customer: Option<CustomerAnyOfData>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub eps: Option<PaymentMethodEps>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub fpx: Option<PaymentMethodFpx>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub giropay: Option<UseStripeSdk>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub grabpay: Option<UseStripeSdk>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub id: String,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub ideal: Option<PaymentMethodIdeal>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub interac_present: Option<UseStripeSdk>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub klarna: Option<PaymentMethodKlarna>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub konbini: Option<UseStripeSdk>,
    /**
    * Whether the account can create live charges.
    */
    #[serde(
        default,
        deserialize_with = "crate::utils::deserialize_null_boolean::deserialize"
    )]
    pub livemode: bool,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub metadata: String,
    /**
    * String representing the object's type. Objects of the same type share the same value.
    */
    #[serde(default, skip_serializing_if = "PaymentMethodObject::is_noop")]
    pub object: PaymentMethodObject,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub oxxo: Option<UseStripeSdk>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none", rename = "p24")]
    pub p_24: Option<PaymentMethodP24>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub paynow: Option<UseStripeSdk>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sepa_debit: Option<PaymentMethodSepaDebit>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub sofort: Option<PaymentMethodSofort>,
    /**
    * The type of the PaymentMethod. An additional hash is included on the PaymentMethod with a name matching this value. It contains additional information specific to the PaymentMethod type.
    */
    #[serde(
        default,
        skip_serializing_if = "PaymentMethodType::is_noop",
        rename = "type"
    )]
    pub type_: PaymentMethodType,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub us_bank_account: Option<PaymentMethodUsBankAccount>,
    /**
    * PaymentMethod objects represent your customer's payment instruments.
    *  They can be used with [PaymentIntents](https://stripe.com/docs/payments/payment-intents) to collect payments or saved to
    *  Customer objects to store instrument details for future payments.
    *  
    *  Related guides: [Payment Methods](https://stripe.com/docs/payments/payment-methods) and [More Payment Scenarios](https://stripe.com/docs/payments/more-payment-scenarios).
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wechat_pay: Option<UseStripeSdk>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodAcssDebit {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bank_name: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub institution_number: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub transit_number: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodAuBecsDebit {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub bsb_number: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodBacsDebit {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub sort_code: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodBoleto {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub tax_id: String,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodCard {
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub brand: String,
    /**
    * Checks on Card address and CVC if provided.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub checks: Option<PaymentMethodCardChecks>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub country: String,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub exp_month: i64,
    /**
    * Time at which the account was connected. Measured in seconds since the Unix epoch.
    */
    #[serde(
        default,
        skip_serializing_if = "crate::utils::zero_i64",
        deserialize_with = "crate::utils::deserialize_null_i64::deserialize"
    )]
    pub exp_year: i64,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub fingerprint: String,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub funding: String,
    /**
    * Details of the original PaymentMethod that created this object.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub generated_from: Option<PaymentMethodCardGenerated>,
    /**
    * The account's country.
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "last4"
    )]
    pub last_4: String,
    /**
    * Contains information about card networks that can be used to process the payment.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub networks: Option<Networks>,
    /**
    * Contains details on how this Card maybe be used for 3D Secure authentication.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub three_d_secure_usage: Option<ThreeDSecureUsage>,
    /**
    * If this Card is part of a card wallet, this contains the details of the card wallet.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub wallet: Option<PaymentMethodCardWalletData>,
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodCardChecks {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "address_line1_check"
    )]
    pub address_line_1_check: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub address_postal_code_check: String,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub cvc_check: String,
}

/// All of the following types:
///
/// - `String`
/// - `SetupAttempt`
///
/// You can easily convert this enum to the inner value with `From` and `Into`, as both are implemented for each type.
///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
#[serde(untagged)]
pub enum SetupAttemptAnyOf {
    /**
    * The account's country.
    */
    String(String),
    /**
    * A SetupAttempt describes one attempted confirmation of a SetupIntent,
    *  whether that confirmation was successful or unsuccessful. You can use
    *  SetupAttempts to inspect details of a specific attempt at setting up a
    *  payment method using a SetupIntent.
    */
    SetupAttempt(SetupAttempt),
}

impl SetupAttemptAnyOf {
    pub fn setup_attempt(&self) -> Option<&SetupAttempt> {
        if let SetupAttemptAnyOf::SetupAttempt(ref_) = self {
            return Some(ref_);
        }
        None
    }

    pub fn string(&self) -> Option<&String> {
        if let SetupAttemptAnyOf::String(ref_) = self {
            return Some(ref_);
        }
        None
    }
}

impl std::convert::From<String> for SetupAttemptAnyOf {
    fn from(f: String) -> Self {
        SetupAttemptAnyOf::String(f)
    }
}

impl std::convert::From<SetupAttemptAnyOf> for String {
    fn from(f: SetupAttemptAnyOf) -> Self {
        f.string().unwrap().clone()
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodCardGenerated {
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize"
    )]
    pub charge: String,
    /**
    * Transaction-specific details of the payment method used in the payment.
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub payment_method_details: Option<CardGeneratedFromPaymentMethodDetails>,
    /**
    *
    */
    #[serde()]
    pub setup_attempt: Box<Option<SetupAttemptAnyOf>>,
}

/**
* The type of the card wallet, one of `amex_express_checkout`, `apple_pay`, `google_pay`, `masterpass`, `samsung_pay`, or `visa_checkout`. An additional hash is included on the Wallet subhash with a name matching this value. It contains additional information specific to the card wallet type.
*/
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub enum PaymentMethodCardWalletType {
    #[serde(rename = "amex_express_checkout")]
    AmexExpressCheckout,
    #[serde(rename = "apple_pay")]
    ApplePay,
    #[serde(rename = "google_pay")]
    GooglePay,
    #[serde(rename = "masterpass")]
    Masterpass,
    #[serde(rename = "samsung_pay")]
    SamsungPay,
    #[serde(rename = "visa_checkout")]
    VisaCheckout,
    #[serde(rename = "")]
    Noop,
    #[serde(other)]
    FallthroughString,
}

impl std::fmt::Display for PaymentMethodCardWalletType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match &*self {
            PaymentMethodCardWalletType::AmexExpressCheckout => "amex_express_checkout",
            PaymentMethodCardWalletType::ApplePay => "apple_pay",
            PaymentMethodCardWalletType::GooglePay => "google_pay",
            PaymentMethodCardWalletType::Masterpass => "masterpass",
            PaymentMethodCardWalletType::SamsungPay => "samsung_pay",
            PaymentMethodCardWalletType::VisaCheckout => "visa_checkout",
            PaymentMethodCardWalletType::Noop => "",
            PaymentMethodCardWalletType::FallthroughString => "*",
        }
        .fmt(f)
    }
}

impl Default for PaymentMethodCardWalletType {
    fn default() -> PaymentMethodCardWalletType {
        PaymentMethodCardWalletType::Noop
    }
}
impl PaymentMethodCardWalletType {
    pub fn is_noop(&self) -> bool {
        matches!(self, PaymentMethodCardWalletType::Noop)
    }
}

///
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone, JsonSchema)]
pub struct PaymentMethodCardWalletData {
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub amex_express_checkout: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub apple_pay: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(
        default,
        skip_serializing_if = "String::is_empty",
        deserialize_with = "crate::utils::deserialize_null_string::deserialize",
        rename = "dynamic_last4"
    )]
    pub dynamic_last_4: String,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub google_pay: Option<UseStripeSdk>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub masterpass: Option<PaymentMethodCardWalletMasterpass>,
    /**
    *
    */
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub samsung_pay: Option<UseStripeSdk>,
    /**
    * The type of the card wallet, one of `amex_express_checkout`, `apple_pay`, `google_pay`, `masterpass`, `samsung_pay`, or `visa_checkout`. An additional hash is included on the Wallet subhash with a name matching this value. It c