use anyhow::Result;

use crate::Client;

pub struct Reviews {
    pub client: Client,
}

impl Reviews {
    #[doc(hidden)]
    pub fn new(client: Client) -> Self {
        Reviews { client }
    }

    /**
    * This function performs a `GET` to the `/v1/reviews` endpoint.
    *
    * <p>Returns a list of <code>Review</code> objects that have <code>open</code> set to <code>true</code>. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
    *
    * **Parameters:**
    *
    * * `created: &str`
    * * `ending_before: &str` -- A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
    * * `expand: &[String]` -- Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    * * `limit: i64` -- A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
    * * `starting_after: &str` -- A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
    */
    pub async fn get_page(
        &self,
        created: &str,
        ending_before: &str,
        limit: i64,
        starting_after: &str,
    ) -> Result<Vec<crate::types::Review>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if !ending_before.is_empty() {
            query_args.push(("ending_before".to_string(), ending_before.to_string()));
        }
        if limit > 0 {
            query_args.push(("limit".to_string(), limit.to_string()));
        }
        if !starting_after.is_empty() {
            query_args.push(("starting_after".to_string(), starting_after.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!("/v1/reviews?{}", query_);

        let resp: crate::types::GetReviewsResponse = self.client.get(&url, None).await?;

        // Return our response data.
        Ok(resp.data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/reviews` endpoint.
    *
    * As opposed to `get`, this function returns all the pages of the request at once.
    *
    * <p>Returns a list of <code>Review</code> objects that have <code>open</code> set to <code>true</code>. The objects are sorted in descending order by creation date, with the most recently created object appearing first.</p>
    */
    pub async fn get_all(&self, created: &str) -> Result<Vec<crate::types::Review>> {
        let url = "/v1/reviews".to_string();
        let mut resp: crate::types::GetReviewsResponse = self.client.get(&url, None).await?;

        let mut data = resp.data;
        let mut has_more = resp.has_more;
        let mut page = "".to_string();

        // Paginate if we should.
        while has_more {
            if !data.is_empty() {
                let last = data.last().unwrap();
                let j = serde_json::json!(last);
                if let serde_json::Value::Object(o) = j {
                    if let Some(p) = o.get("id") {
                        if let serde_json::Value::String(s) = p {
                            page = s.to_string();
                        }
                    }
                }
            }

            if !url.contains('?') {
                resp = self
                    .client
                    .get(&format!("{}?startng_after={}", url, page), None)
                    .await?;
            } else {
                resp = self
                    .client
                    .get(&format!("{}&starting_after={}", url, page), None)
                    .await?;
            }

            data.append(&mut resp.data);

            has_more = resp.has_more;
        }

        // Return our response data.
        Ok(data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/reviews/{review}` endpoint.
    *
    * <p>Retrieves a <code>Review</code> object.</p>
    *
    * **Parameters:**
    *
    * * `expand: &[String]` -- Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    * * `review: &str` -- The account's country.
    */
    pub async fn get(&self, review: &str) -> Result<crate::types::Review> {
        let url = format!(
            "/v1/reviews/{}",
            crate::progenitor_support::encode_path(&review.to_string()),
        );

        self.client.get(&url, None).await
    }

    /**
    * This function performs a `POST` to the `/v1/reviews/{review}/approve` endpoint.
    *
    * <p>Approves a <code>Review</code> object, closing it and removing it from the list of reviews.</p>
    *
    * **Parameters:**
    *
    * * `review: &str` -- The account's country.
    */
    pub async fn post_approve(&self, review: &str) -> Result<crate::types::Review> {
        let url = format!(
            "/v1/reviews/{}/approve",
            crate::progenitor_support::encode_path(&review.to_string()),
        );

        self.client.post(&url, None).await
    }
}
