use anyhow::Result;

use crate::Client;

pub struct Bitcoin {
    pub client: Client,
}

impl Bitcoin {
    #[doc(hidden)]
    pub fn new(client: Client) -> Self {
        Bitcoin { client }
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/receivers` endpoint.
    *
    * <p>Returns a list of your receivers. Receivers are returned sorted by creation date, with the most recently created receivers appearing first.</p>
    *
    * **Parameters:**
    *
    * * `active: bool` -- Whether the account can create live charges.
    * * `ending_before: &str` -- A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
    * * `expand: &[String]` -- Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    * * `filled: bool` -- Whether the account can create live charges.
    * * `limit: i64` -- A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
    * * `starting_after: &str` -- A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
    * * `uncaptured_funds: bool` -- Whether the account can create live charges.
    */
    pub async fn get_receivers(
        &self,
        active: bool,
        ending_before: &str,
        filled: bool,
        limit: i64,
        starting_after: &str,
        uncaptured_funds: bool,
    ) -> Result<Vec<crate::types::BitcoinReceiver>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if active {
            query_args.push(("active".to_string(), active.to_string()));
        }
        if !ending_before.is_empty() {
            query_args.push(("ending_before".to_string(), ending_before.to_string()));
        }
        if filled {
            query_args.push(("filled".to_string(), filled.to_string()));
        }
        if limit > 0 {
            query_args.push(("limit".to_string(), limit.to_string()));
        }
        if !starting_after.is_empty() {
            query_args.push(("starting_after".to_string(), starting_after.to_string()));
        }
        if uncaptured_funds {
            query_args.push(("uncaptured_funds".to_string(), uncaptured_funds.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!("/v1/bitcoin/receivers?{}", query_);

        let resp: crate::types::GetBitcoinReceiversResponse = self.client.get(&url, None).await?;

        // Return our response data.
        Ok(resp.data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/receivers` endpoint.
    *
    * As opposed to `get_receivers`, this function returns all the pages of the request at once.
    *
    * <p>Returns a list of your receivers. Receivers are returned sorted by creation date, with the most recently created receivers appearing first.</p>
    */
    pub async fn get_all_receivers(
        &self,
        active: bool,
        filled: bool,
        uncaptured_funds: bool,
    ) -> Result<Vec<crate::types::BitcoinReceiver>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if active {
            query_args.push(("active".to_string(), active.to_string()));
        }
        if filled {
            query_args.push(("filled".to_string(), filled.to_string()));
        }
        if uncaptured_funds {
            query_args.push(("uncaptured_funds".to_string(), uncaptured_funds.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!("/v1/bitcoin/receivers?{}", query_);

        let mut resp: crate::types::GetBitcoinReceiversResponse =
            self.client.get(&url, None).await?;

        let mut data = resp.data;
        let mut has_more = resp.has_more;
        let mut page = "".to_string();

        // Paginate if we should.
        while has_more {
            if !data.is_empty() {
                let last = data.last().unwrap();
                let j = serde_json::json!(last);
                if let serde_json::Value::Object(o) = j {
                    if let Some(p) = o.get("id") {
                        if let serde_json::Value::String(s) = p {
                            page = s.to_string();
                        }
                    }
                }
            }

            if !url.contains('?') {
                resp = self
                    .client
                    .get(&format!("{}?startng_after={}", url, page), None)
                    .await?;
            } else {
                resp = self
                    .client
                    .get(&format!("{}&starting_after={}", url, page), None)
                    .await?;
            }

            data.append(&mut resp.data);

            has_more = resp.has_more;
        }

        // Return our response data.
        Ok(data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/receivers/{id}` endpoint.
    *
    * <p>Retrieves the Bitcoin receiver with the given ID.</p>
    *
    * **Parameters:**
    *
    * * `expand: &[String]` -- Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    * * `id: &str` -- The account's country.
    */
    pub async fn get_receiver(&self, id: &str) -> Result<crate::types::BitcoinReceiver> {
        let url = format!(
            "/v1/bitcoin/receivers/{}",
            crate::progenitor_support::encode_path(&id.to_string()),
        );

        self.client.get(&url, None).await
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/receivers/{receiver}/transactions` endpoint.
    *
    * <p>List bitcoin transacitons for a given receiver.</p>
    *
    * **Parameters:**
    *
    * * `customer: &str` -- Only return transactions for the customer specified by this customer ID.
    * * `ending_before: &str` -- A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
    * * `expand: &[String]` -- Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    * * `limit: i64` -- A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
    * * `receiver: &str` -- The account's country.
    * * `starting_after: &str` -- A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
    */
    pub async fn get_receivers_receiver_transactions(
        &self,
        customer: &str,
        ending_before: &str,
        limit: i64,
        receiver: &str,
        starting_after: &str,
    ) -> Result<Vec<crate::types::BitcoinTransaction>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if !customer.is_empty() {
            query_args.push(("customer".to_string(), customer.to_string()));
        }
        if !ending_before.is_empty() {
            query_args.push(("ending_before".to_string(), ending_before.to_string()));
        }
        if limit > 0 {
            query_args.push(("limit".to_string(), limit.to_string()));
        }
        if !starting_after.is_empty() {
            query_args.push(("starting_after".to_string(), starting_after.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!(
            "/v1/bitcoin/receivers/{}/transactions?{}",
            crate::progenitor_support::encode_path(&receiver.to_string()),
            query_
        );

        let resp: crate::types::Transactions = self.client.get(&url, None).await?;

        // Return our response data.
        Ok(resp.data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/receivers/{receiver}/transactions` endpoint.
    *
    * As opposed to `get_receivers_receiver_transactions`, this function returns all the pages of the request at once.
    *
    * <p>List bitcoin transacitons for a given receiver.</p>
    */
    pub async fn get_all_receivers_receiver_transactions(
        &self,
        customer: &str,
        receiver: &str,
    ) -> Result<Vec<crate::types::BitcoinTransaction>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if !customer.is_empty() {
            query_args.push(("customer".to_string(), customer.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!(
            "/v1/bitcoin/receivers/{}/transactions?{}",
            crate::progenitor_support::encode_path(&receiver.to_string()),
            query_
        );

        let mut resp: crate::types::Transactions = self.client.get(&url, None).await?;

        let mut data = resp.data;
        let mut has_more = resp.has_more;
        let mut page = "".to_string();

        // Paginate if we should.
        while has_more {
            if !data.is_empty() {
                let last = data.last().unwrap();
                let j = serde_json::json!(last);
                if let serde_json::Value::Object(o) = j {
                    if let Some(p) = o.get("id") {
                        if let serde_json::Value::String(s) = p {
                            page = s.to_string();
                        }
                    }
                }
            }

            if !url.contains('?') {
                resp = self
                    .client
                    .get(&format!("{}?startng_after={}", url, page), None)
                    .await?;
            } else {
                resp = self
                    .client
                    .get(&format!("{}&starting_after={}", url, page), None)
                    .await?;
            }

            data.append(&mut resp.data);

            has_more = resp.has_more;
        }

        // Return our response data.
        Ok(data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/transactions` endpoint.
    *
    * <p>List bitcoin transacitons for a given receiver.</p>
    *
    * **Parameters:**
    *
    * * `customer: &str` -- Only return transactions for the customer specified by this customer ID.
    * * `ending_before: &str` -- A cursor for use in pagination. `ending_before` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, starting with `obj_bar`, your subsequent call can include `ending_before=obj_bar` in order to fetch the previous page of the list.
    * * `expand: &[String]` -- Fields that need to be collected to keep the capability enabled. If not collected by `future_requirements[current_deadline]`, these fields will transition to the main `requirements` hash.
    * * `limit: i64` -- A limit on the number of objects to be returned. Limit can range between 1 and 100, and the default is 10.
    * * `receiver: &str` -- The account's country.
    * * `starting_after: &str` -- A cursor for use in pagination. `starting_after` is an object ID that defines your place in the list. For instance, if you make a list request and receive 100 objects, ending with `obj_foo`, your subsequent call can include `starting_after=obj_foo` in order to fetch the next page of the list.
    */
    pub async fn get_transactions(
        &self,
        customer: &str,
        ending_before: &str,
        limit: i64,
        receiver: &str,
        starting_after: &str,
    ) -> Result<Vec<crate::types::BitcoinTransaction>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if !customer.is_empty() {
            query_args.push(("customer".to_string(), customer.to_string()));
        }
        if !ending_before.is_empty() {
            query_args.push(("ending_before".to_string(), ending_before.to_string()));
        }
        if limit > 0 {
            query_args.push(("limit".to_string(), limit.to_string()));
        }
        if !receiver.is_empty() {
            query_args.push(("receiver".to_string(), receiver.to_string()));
        }
        if !starting_after.is_empty() {
            query_args.push(("starting_after".to_string(), starting_after.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!("/v1/bitcoin/transactions?{}", query_);

        let resp: crate::types::Transactions = self.client.get(&url, None).await?;

        // Return our response data.
        Ok(resp.data.to_vec())
    }

    /**
    * This function performs a `GET` to the `/v1/bitcoin/transactions` endpoint.
    *
    * As opposed to `get_transactions`, this function returns all the pages of the request at once.
    *
    * <p>List bitcoin transacitons for a given receiver.</p>
    */
    pub async fn get_all_transactions(
        &self,
        customer: &str,
        receiver: &str,
    ) -> Result<Vec<crate::types::BitcoinTransaction>> {
        let mut query_args: Vec<(String, String)> = Default::default();
        if !customer.is_empty() {
            query_args.push(("customer".to_string(), customer.to_string()));
        }
        if !receiver.is_empty() {
            query_args.push(("receiver".to_string(), receiver.to_string()));
        }
        let query_ = serde_urlencoded::to_string(&query_args).unwrap();
        let url = format!("/v1/bitcoin/transactions?{}", query_);

        let mut resp: crate::types::Transactions = self.client.get(&url, None).await?;

        let mut data = resp.data;
        let mut has_more = resp.has_more;
        let mut page = "".to_string();

        // Paginate if we should.
        while has_more {
            if !data.is_empty() {
                let last = data.last().unwrap();
                let j = serde_json::json!(last);
                if let serde_json::Value::Object(o) = j {
                    if let Some(p) = o.get("id") {
                        if let serde_json::Value::String(s) = p {
                            page = s.to_string();
                        }
                    }
                }
            }

            if !url.contains('?') {
                resp = self
                    .client
                    .get(&format!("{}?startng_after={}", url, page), None)
                    .await?;
            } else {
                resp = self
                    .client
                    .get(&format!("{}&starting_after={}", url, page), None)
                    .await?;
            }

            data.append(&mut resp.data);

            has_more = resp.has_more;
        }

        // Return our response data.
        Ok(data.to_vec())
    }
}
