use std::io::{stdin, stdout, Write};

use crate::profile::ProfileName;
use crate::rule::{RuleData, RuleName};

const DOFI_DIR: &str = ".dofi/";
const DOFI_EXT: &str = ".toml";

/// Main functions of DoFi
pub trait DoFi {
    fn add(&mut self, name: RuleName, data: RuleData) -> anyhow::Result<()>;
    fn apply(&self, name: &String) -> anyhow::Result<()>;
    fn del(&mut self, name: RuleName) -> anyhow::Result<()>;
    fn show(&self) -> anyhow::Result<()>;
}

/// Create the dofi config directory, if necessary
pub fn create_dofi_dir() {
    let _ = std::fs::create_dir(DOFI_DIR);
}

/// Return the profile path from the profile name
pub fn get_profile_path(name: &ProfileName) -> String {
    let mut path = String::from(DOFI_DIR);
    path.push_str(name);
    path.push_str(DOFI_EXT);

    path
}

/// Shrink home path to ~ and return
pub fn shrink_home(path: String) -> anyhow::Result<String> {
    if let Some(home) = dirs::home_dir() {
        if let Some(home) = home.to_str() {
            if path.starts_with(home) {
                let mut p = path.clone();
                p.replace_range(0..home.len(), "~");

                return Ok(p);
            }
        }
    }

    Ok(path)
}

/// Expand ~ to home path and return
pub fn expand_home(name: &RuleName, path: String) -> anyhow::Result<String> {
    if path.starts_with("~") {
        if let Some(home) = dirs::home_dir() {
            if let Some(home) = home.to_str() {
                let mut p = path.clone();
                p.remove(0);
                p.insert_str(0, home);

                return Ok(p);
            }
        }
        // ERROR: Failed to expand ~
        anyhow::bail!("Skiped: [{}] Failed to expand ~ to home", name);
    }

    Ok(path)
}

/// Ask user a question and get the answer
pub fn ask() -> bool {
    let _ = stdout().lock().flush();

    let stdin = stdin();
    let mut input = String::new();
    let _ = stdin.read_line(&mut input);
    let input = input.to_lowercase();
    let input = input.trim();

    match input {
        "y" | "Y" => true,
        "n" | "N" => false,
        _ => false,
    }
}
