use anyhow::{Context, Result};
use argh::FromArgValue;
use serde::{Deserialize, Serialize};

use std::fs::File;
use std::path::PathBuf;

use crate::util::DoFi;

pub type RuleName = String;

#[derive(Debug, Deserialize, Serialize)]
pub struct RuleData {
    pub src: String,
    pub dst: String,
    pub mode: RuleMode,
}

impl DoFi for RuleData {
    /// Apply the rule from source to target
    fn apply(&self, name: &RuleName) -> Result<()> {
        let source = crate::util::expand_home(name, self.src.clone())?;
        let target = crate::util::expand_home(name, self.dst.clone())?;

        let src = std::fs::canonicalize(&source).unwrap_or(PathBuf::from(&source));
        let dst = PathBuf::from(&target);

        if let Err(error) = match self.mode {
            RuleMode::Symlink => apply_link(name, src, dst),
            RuleMode::Copy => apply_copy(name, src, dst),
        } {
            eprintln!("{}", error);
            error
                .chain()
                .skip(1)
                .for_each(|cause| eprintln!("{:indent$}{}", "", cause, indent = 8));
        }

        Ok(())
    }

    /// This function won't implement
    fn add(&mut self, _name: RuleName, _data: RuleData) -> Result<()> {
        todo!()
    }

    /// This function won't implement
    fn del(&mut self, _name: RuleName) -> Result<()> {
        todo!()
    }

    /// Print informations of the rule
    fn show(&self) -> Result<()> {
        println!("Source: {}", self.src);
        println!("Target: {}", self.dst);
        println!("Mode  : {}", self.mode);

        Ok(())
    }
}

#[derive(Debug, Deserialize, Serialize)]
#[serde(rename_all = "lowercase")]
pub enum RuleMode {
    Symlink,
    Copy,
}

impl std::fmt::Display for RuleMode {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            RuleMode::Symlink => write!(f, "symlink"),
            RuleMode::Copy => write!(f, "copy"),
        }
    }
}

impl FromArgValue for RuleMode {
    fn from_arg_value(value: &str) -> Result<Self, String> {
        match value {
            "symlink" => Ok(RuleMode::Symlink),
            "copy" => Ok(RuleMode::Copy),
            _ => Err("Wrong rule mode".to_string()),
        }
    }
}

/// Apply rule with method symlink
fn apply_link(name: &RuleName, src: PathBuf, dst: PathBuf) -> Result<()> {
    if !src.exists() {
        // ERROR: Source doesn't exist
        anyhow::bail!("Skiped: [{}] {} Source doesn't exist", name, src.display());
    }

    if dst.exists() {
        if is_linked(&src, &dst) {
            return Ok(println!("Linked: [{}]", name));
        }
        // ERROR: Target already exists
        anyhow::bail!("Skiped: [{}] {} Target already exists", name, dst.display())
    }

    create_parent(&name, &dst)?;
    std::os::unix::fs::symlink(&src, &dst)
        // ERROR: IO error
        .with_context(|| format!("Skiped: [{}] {}", name, dst.display()))?;

    Ok(println!("Linkto: [{}] > {}", name, dst.display()))
}

/// Apply rule with method copy
fn apply_copy(name: &RuleName, src: PathBuf, dst: PathBuf) -> Result<()> {
    if !src.exists() {
        // ERROR: Source doesn't exist
        anyhow::bail!("Skiped: [{}] {} Source  doesn't exist", name, src.display());
    }

    if dst.exists() {
        if is_copied(&src, &dst) {
            return Ok(println!("Copied: [{}]", name));
        }
        // ERROR: Target already exists
        anyhow::bail!("Skiped: [{}] {} Target already exists", name, dst.display())
    }

    create_parent(&name, &dst)?;
    std::fs::copy(&src, &dst)
        // ERROR: IO error
        .with_context(|| format!("Skiped: [{}] {}", name, dst.display()))?;

    Ok(println!("Copyto: [{}] > {}", name, dst.display()))
}

/// Check target is linked from source or not
fn is_linked(src: &PathBuf, dst: &PathBuf) -> bool {
    if let Ok(dst) = std::fs::read_link(dst) {
        return &dst == src;
    }

    false
}

/// Check target is copied from source or not
fn is_copied(src: &PathBuf, dst: &PathBuf) -> bool {
    if let Err(_) = std::fs::read_link(&dst) {
        if let Ok(src) = std::fs::File::open(src) {
            if let Ok(dst) = std::fs::File::open(dst) {
                return is_same_file(src, dst);
            }
        }
    }

    false
}

/// Check source and target are same or not
fn is_same_file(mut src: File, mut dst: File) -> bool {
    const BUF_SIZE: usize = 4096;

    let mut buff1 = [0u8; BUF_SIZE];
    let mut buff2 = [0u8; BUF_SIZE];

    loop {
        if let Ok(src_len) = std::io::Read::read(&mut src, &mut buff1) {
            if let Ok(dst_len) = std::io::Read::read(&mut dst, &mut buff2) {
                if src_len != dst_len {
                    return false;
                }
                if src_len == 0 {
                    return true;
                }
                if &buff1[0..src_len] != &buff2[0..dst_len] {
                    return false;
                }
            };
        }
    }
}

/// Create parent directory of target if necessary
fn create_parent(name: &RuleName, dst: &PathBuf) -> Result<()> {
    if let Some(parent) = dst.parent() {
        std::fs::create_dir_all(parent).with_context(|| {
            // ERROR: IO error
            format!("Skiped: [{}] {}", name, dst.display())
        })?
    }

    Ok(())
}
