use anyhow::Context;
use serde::{Deserialize, Serialize};
use std::collections::BTreeMap;

use crate::rule::{RuleData, RuleName};
use crate::util::DoFi;

pub type ProfileName = String;

#[derive(Debug, Deserialize, Serialize)]
pub struct ProfileData {
    pub name: ProfileName,
    pub rules: BTreeMap<RuleName, RuleData>,
}

impl ProfileData {
    /// Return a `ProfileData` from the `ProfileName`
    pub fn from(name: ProfileName) -> anyhow::Result<ProfileData> {
        let path = crate::util::get_profile_path(&name);
        let data = std::fs::read(path)
            // ERROR: Failed to load profile
            .with_context(|| format!("Faild to load profile [{}]", &name))?;
        let rules = toml::from_slice(&data)
            // ERROR: Failed to deserialize profile
            .with_context(|| format!("Failed to deserialize profile [{}] toml data", &name,))?;

        Ok(ProfileData { name, rules })
    }
}

impl DoFi for ProfileData {
    /// Apply rules and print formatted error
    fn apply(&self, name: &ProfileName) -> anyhow::Result<()> {
        println!("Applying: {}\n", name);

        for (k, v) in &self.rules {
            if let Err(error) = v.apply(&k) {
                eprintln!("{}", error);
                error
                    .chain()
                    .skip(1)
                    .for_each(|cause| eprintln!("{:indent$}{}", "", cause, indent = 8));
            }
        }

        Ok(())
    }

    /// Add a rule to the profile
    fn add(&mut self, name: RuleName, data: RuleData) -> anyhow::Result<()> {
        self.rules.insert(name, data);
        crate::util::create_dofi_dir();

        let path = crate::util::get_profile_path(&self.name);
        let data = toml::to_string(&self.rules)?;

        std::fs::write(path, data)?;

        Ok(())
    }

    /// Delete a rule from the profile
    fn del(&mut self, name: RuleName) -> anyhow::Result<()> {
        self.rules.remove(&name);
        crate::util::create_dofi_dir();

        let path = crate::util::get_profile_path(&self.name);
        let data = toml::to_string(&self.rules)?;

        std::fs::write(path, data)?;

        Ok(())
    }

    /// Print all rules from the profile
    fn show(&self) -> anyhow::Result<()> {
        for (k, v) in &self.rules {
            println!("[{}]", k);
            v.show()?;
            println!("");
        }

        Ok(())
    }
}
