use argh::FromArgs;
use dofi::profile::ProfileData;
use dofi::rule::{RuleData, RuleMode};
use dofi::util::DoFi;

/// Add a rule
#[derive(Debug, FromArgs)]
#[argh(subcommand, name = "add")]
pub struct Options {
    /// the rule name
    #[argh(positional)]
    rule: String,
    /// the source path
    #[argh(positional, arg_name = "source")]
    source: String,
    /// the target path
    #[argh(positional, arg_name = "target")]
    target: String,
    /// the rule mode [symlink, copy] (default: symlink)
    #[argh(option, short = 'm', default = "RuleMode::Symlink")]
    mode: RuleMode,
    /// the profile name (default: default)
    #[argh(option, short = 'p', default = "String::from(\"default\")")]
    profile: String,
}

impl Options {
    pub fn run(self) -> anyhow::Result<()> {
        let mut p = ProfileData::from(self.profile)?;

        if let Some(v) = p.rules.get(&self.rule) {
            v.show()?;
            print!("The rule [{}] already exists, overwrite? Y/n: ", &self.rule);

            if !dofi::util::ask() {
                return Ok(());
            }
        }

        let src = dofi::util::shrink_home(self.source)?;
        let dst = dofi::util::shrink_home(self.target)?;
        let rule = RuleData {
            src,
            dst,
            mode: self.mode,
        };

        p.add(self.rule, rule)
    }
}
