//! Yaml Serializer (using [Serde](https://docs.rs/serde_yaml/0.8.11/serde_yaml/)).

use std::error::Error;
use std::io;

use serde::{de::DeserializeOwned, Serialize};

use super::{Result, Serializer, SerializerError};

/// Yaml serializer (using [Serde](https://docs.rs/serde_yaml/0.8.11/serde_yaml/)).
///
/// # Examples
///
/// ```
/// use dodo::serializer::YamlSerializer;
/// use dodo::prelude::*;
/// # use serde::{Deserialize, Serialize};
/// # use uuid::Uuid;
/// #
/// # #[derive(Debug, Entity, Serialize, Deserialize, Eq, PartialEq)]
/// # #[serde(rename_all = "camelCase")]
/// # struct Person { id: Option<Uuid>, name: String, age: u64 }
/// #
/// # impl Person {
/// #    fn with_age(age : u64) -> Self { Self { id : None, name : "John Smith".into(), age }}
/// # }
///
/// type PersonCollection = Collection<Person, Directory, YamlSerializer>;
///
/// fn main() -> Result<(), Box<dyn std::error::Error>> {
/// #   let path  = tempfile::tempdir()?;
///     let directory = Directory::new(&path)?;
///     let mut collection = PersonCollection::new(directory);
///     Ok(())
/// }
/// ```
#[derive(Debug, Clone)]
pub struct YamlSerializer;

impl Serializer for YamlSerializer {
    fn serialize<T, W>(mut writer: W, value: &T) -> Result<()>
        where T: Serialize + DeserializeOwned,
              W: io::Write {
        serde_yaml::to_writer(&mut writer, value)?;
        writer.flush().map_err(From::from)
    }

    fn deserialize<T, R>(reader: R) -> Result<T>
        where T: Serialize + DeserializeOwned,
              R: io::Read {
        serde_yaml::from_reader(reader).map_err(From::from)
    }
}

impl From<serde_yaml::Error> for SerializerError {
    fn from(error: serde_yaml::Error) -> Self {
        //Unfortunately, we have to downcast to know if this is a syntax error or a io error.
        match error.source() {
            Some(e) => {
                if e.is::<io::Error>() {
                    Self::io(error)
                } else {
                    Self::syntax(error)
                }
            }
            None => Self::other(error)
        }
    }
}