use std::{process::Output, sync::Arc};

use eyre::Result;
use getset::{Getters, MutGetters, Setters};
use serde::Deserialize;
use tokio::process::Command;
use tracing::{debug, error, info};

use crate::{error::DocktorError, Config};

#[derive(Debug)]
pub struct Weave {
    config: Arc<Config>,
}

#[derive(Debug, Default, Deserialize, Clone, Getters, Setters, MutGetters)]
#[serde(rename_all = "PascalCase")]
pub struct DnsEntry {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    hostname: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    origin: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    container_id: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    address: String,
}

#[derive(Debug, Default, Deserialize, Clone, Getters, Setters, MutGetters)]
#[serde(rename_all = "PascalCase")]
pub struct WeaveDns {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    domain: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    upstream: Vec<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    address: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    ttl: usize,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    entries: Vec<DnsEntry>,
}

#[derive(Debug, Default, Deserialize, Clone, Getters, Setters, MutGetters)]
#[serde(rename_all = "PascalCase")]
pub struct WeaveReport {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    ready: bool,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    origin: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    dns: WeaveDns,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    version: usize,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    tombstone: usize,
}

impl Weave {
    pub fn new(config: Arc<Config>) -> Self {
        Self { config }
    }

    async fn run_command(&self, command: &str) -> Result<Output, DocktorError> {
        Ok(Command::new("weave").args(command.split(' ')).output().await?)
    }

    pub async fn report(&self) -> WeaveReport {
        match self.run_command("report").await {
            Ok(result) => {
                let report: WeaveReport = serde_json::from_slice(&result.stdout).unwrap_or_default();
                debug!("Weave report: {:#?}", report);
                report
            }
            Err(e) => {
                error!("Error getting Weave report: {}", e);
                WeaveReport::default()
            }
        }
    }

    pub async fn dns_add(&self, hostname: &str, ip_address: &str) {
        match self
            .run_command(&format!("dns-add {} -h {}", hostname, ip_address))
            .await
        {
            Ok(result) => {
                if result.status.success() {
                    info!("Added DNS entry '{}' pointing to '{}'", hostname, ip_address);
                } else {
                    error!(
                        "Error running dns-add for {}: {}",
                        hostname,
                        String::from_utf8_lossy(&result.stderr)
                    );
                }
            }
            Err(e) => {
                error!("Error running dns-add for {}: {}", hostname, e);
            }
        }
    }

    pub async fn dns_remove(&self, hostname: &str, ip_address: &str) {
        match self
            .run_command(&format!("dns-remove {} -h {}", hostname, ip_address))
            .await
        {
            Ok(result) => {
                if result.status.success() {
                    info!("Removed DNS entry '{}' pointing to '{}'", hostname, ip_address);
                } else {
                    error!(
                        "Error running dns-remove for {}: {}",
                        hostname,
                        String::from_utf8_lossy(&result.stderr)
                    );
                }
            }
            Err(e) => {
                error!("Error running dns-remove for {}: {}", hostname, e);
            }
        }
    }

    pub async fn dns_lookup(&self, hostname: &str) -> String {
        let local_ip = self.config.ip_address.to_string();
        match self.run_command(&format!("dns-lookup {}", hostname)).await {
            Ok(result) => {
                if result.status.success() {
                    let mut ip_address = String::from_utf8_lossy(&result.stdout).to_string();
                    if ip_address.is_empty() {
                        ip_address = local_ip
                    }
                    info!("DNS entry '{}' points to '{}'", hostname, ip_address);
                    ip_address
                } else {
                    error!(
                        "Error running dns-lookup for {}: {}. Returning local IP {}",
                        hostname,
                        String::from_utf8_lossy(&result.stderr),
                        local_ip
                    );
                    local_ip
                }
            }
            Err(e) => {
                error!(
                    "Error running dns-lookup for {}: {}. Returning local IP {}",
                    hostname, e, local_ip
                );
                local_ip
            }
        }
    }

    pub async fn dns_list(&self) -> String {
        let report = self.report().await;
        let mut result = format!("DNS entries for {}:\n", self.config.hostname);
        for entry in report.dns.entries {
            result.push_str(&format!(" * {}:\t{}\n", entry.hostname, entry.address));
        }
        result
    }
}
