use std::sync::Arc;

use clap::Parser;
use docktor_api::{model, output};
use eyre::Result;
use getset::{Getters, MutGetters, Setters};
use tokio::{process::Command, sync::RwLock};

use crate::{client::ApiClient, error::DocktorError, Config};

#[derive(Parser, Debug, Clone)]
pub struct System {
    /// System subcommand.
    #[clap(subcommand)]
    pub action: Action,
    /// Hostname to run services on.
    #[clap(short = 'H', long = "hostname")]
    pub hostname: Option<String>,
    /// List of services.
    #[clap(short = 's', long = "services")]
    pub services: Vec<String>,
}

#[derive(Parser, Debug, Clone)]
pub enum Action {
    /// Start service.
    Start,
    /// Stop service.
    Stop,
    /// Restart service.
    Restart,
}

impl System {
    pub async fn exec(&self) -> Result<()> {
        let config = Arc::new(Config::load().await?);
        let data = Arc::new(RwLock::new(model::ListOutput::builder().build()));
        let systemd = Systemd::new(config, data);
        match self.action {
            Action::Start => {
                systemd.start(&self.services, self.hostname.as_deref()).await?;
            }
            Action::Stop => {
                systemd.stop(&self.services, self.hostname.as_deref()).await?;
            }
            Action::Restart => {
                systemd.stop(&self.services, self.hostname.as_deref()).await?;
            }
        }
        Ok(())
    }
}

#[derive(Debug, Clone, Getters, Setters, MutGetters)]
pub struct Systemd {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    config: Arc<Config>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    data: Arc<RwLock<model::ListOutput>>,
}

impl Systemd {
    pub fn new(config: Arc<Config>, data: Arc<RwLock<model::ListOutput>>) -> Self {
        Self { config, data }
    }

    pub async fn start(
        &self,
        services: &[String],
        hostname: Option<&str>,
    ) -> Result<output::StartOperationOutput, DocktorError> {
        let mut result = output::StartOperationOutput::builder().build();
        let hostname = hostname.unwrap_or("");
        if self.config().primary() && hostname != self.config().hostname() {
            for peer in self.config().peers().iter() {
                let cli = ApiClient::new(peer.to_string(), self.config().api_key().clone())?;
                let response = cli
                    .0
                    .start_operation()
                    .set_services(Some(services.to_vec()))
                    .set_hostname(Some(hostname.to_string()))
                    .send()
                    .await?;
                let response = unsafe {
                    std::mem::transmute::<docktor_api_client::output::StartOperationOutput, output::StartOperationOutput>(
                        response,
                    )
                };
                result.inner.as_mut().unwrap().append(&mut response.inner.unwrap());
            }
        }
        result
            .inner
            .as_mut()
            .unwrap()
            .append(&mut self.run_systemctl_command("start", services).await?);
        Ok(result)
    }

    pub async fn stop(
        &self,
        services: &[String],
        hostname: Option<&str>,
    ) -> Result<output::StopOperationOutput, DocktorError> {
        let mut result = output::StopOperationOutput::builder().build();
        let hostname = hostname.unwrap_or("");
        if self.config().primary() && hostname != self.config().hostname() {
            for peer in self.config().peers().iter() {
                let cli = ApiClient::new(peer.to_string(), self.config().api_key().clone())?;
                let response = cli
                    .0
                    .stop_operation()
                    .set_services(Some(services.to_vec()))
                    .set_hostname(Some(hostname.to_string()))
                    .send()
                    .await?;
                let response = unsafe {
                    std::mem::transmute::<docktor_api_client::output::StopOperationOutput, output::StopOperationOutput>(
                        response,
                    )
                };
                result.inner.as_mut().unwrap().append(&mut response.inner.unwrap());
            }
        }
        result
            .inner
            .as_mut()
            .unwrap()
            .append(&mut self.run_systemctl_command("stop", services).await?);
        Ok(result)
    }

    pub async fn restart(
        &self,
        services: &[String],
        hostname: Option<&str>,
    ) -> Result<output::RestartOperationOutput, DocktorError> {
        let mut result = output::RestartOperationOutput::builder().build();
        let hostname = hostname.unwrap_or("");
        if self.config().primary() && hostname != self.config().hostname() {
            for peer in self.config().peers().iter() {
                let cli = ApiClient::new(peer.to_string(), self.config().api_key().clone())?;
                let response = cli
                    .0
                    .restart_operation()
                    .set_services(Some(services.to_vec()))
                    .set_hostname(Some(hostname.to_string()))
                    .send()
                    .await?;
                let response = unsafe {
                    std::mem::transmute::<
                        docktor_api_client::output::RestartOperationOutput,
                        output::RestartOperationOutput,
                    >(response)
                };
                result.inner.as_mut().unwrap().append(&mut response.inner.unwrap());
            }
        }
        result
            .inner
            .as_mut()
            .unwrap()
            .append(&mut self.run_systemctl_command("restart", services).await?);
        Ok(result)
    }

    async fn run_systemctl_command(
        &self,
        action: &str,
        services: &[String],
    ) -> Result<Vec<model::SystemdOutputInner>, DocktorError> {
        let mut results = Vec::new();
        for service in services {
            let command = Command::new("systemctl").arg(action).args(services).output().await?;
            let result = if command.status.success() {
                model::SystemdOutputInner::builder()
                    .set_hostname(Some(self.config().hostname().clone()))
                    .set_service(Some(service.to_string()))
                    .set_output(Some(String::from_utf8_lossy(&command.stdout).to_string()))
                    .set_exit_code(command.status.code().map(|x| x as i64))
                    .build()
            } else {
                model::SystemdOutputInner::builder()
                    .set_hostname(Some(self.config().hostname().clone()))
                    .set_service(Some(service.to_string()))
                    .set_error(Some(String::from_utf8_lossy(&command.stderr).to_string()))
                    .set_exit_code(command.status.code().map(|x| x as i64))
                    .build()
            };
            results.push(result);
        }
        Ok(results)
    }
}
