use std::{
    collections::HashMap,
    fmt,
    net::{TcpStream, ToSocketAddrs},
    path::Path,
    str::FromStr,
    sync::Arc,
};

use async_io::Async;
use async_ssh2_lite::AsyncSession;
use bollard::{
    container::ListContainersOptions,
    models::{ContainerSummaryInner, HealthcheckResult},
    Docker,
};
use docktor_api::model;
use eyre::Result;
use futures::AsyncReadExt;
use getset::{Getters, MutGetters, Setters};
use rand::{distributions::Alphanumeric, thread_rng, Rng};
use systemstat::{saturating_sub_bytes, Platform, System};
use tokio::{process::Command, sync::RwLock};
use tracing::{debug, error, warn};

use super::prometheus::PrometheusTargets;
use crate::{
    client::ApiClient,
    config::{Config, Process, SshProcess},
    systemd::Systemd,
    weave::Weave,
};

#[derive(Getters, Setters, MutGetters)]
pub struct State {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub primary: bool,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub config: Arc<Config>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub docker: Option<Arc<Docker>>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub data: Arc<RwLock<model::ListOutput>>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub systemd: Arc<Systemd>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub weave: Arc<Weave>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub prometheus_targets: Arc<PrometheusTargets>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    ssh_sessions: RwLock<HashMap<String, AsyncSession<TcpStream>>>,
}

impl fmt::Debug for State {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("State")
            .field("config", &self.config)
            .field("docker", &self.docker)
            .field("data", &self.data)
            .field("system", &self.systemd)
            .field("weave", &self.weave)
            .field("prometheus_targets", &self.prometheus_targets)
            .finish()
    }
}

impl State {
    pub fn new(config: Arc<Config>, primary: bool) -> Result<Self> {
        let docker = match Docker::connect_with_socket_defaults() {
            Ok(docker) => Some(Arc::new(docker)),
            Err(e) => {
                warn!("Unable to connect to Docker, containers visit disabled: {}", e);
                None
            }
        };
        let data = Arc::new(RwLock::new(model::ListOutput::builder().build()));
        let systemd = Arc::new(Systemd::new(config.clone(), data.clone()));
        let prometheus_targets = Arc::new(PrometheusTargets::new(config.clone(), data.clone()));
        let weave = Arc::new(Weave::new(config.clone()));
        Ok(Self {
            primary,
            config,
            docker,
            data,
            systemd,
            weave,
            ssh_sessions: Default::default(),
            prometheus_targets,
        })
    }

    async fn host_info(&self) -> model::Host {
        model::Host::builder()
            .hostname(self.config().hostname().clone())
            .domain(self.config().domain().clone())
            .build()
    }

    async fn container_env_variable(&self, container_id: &str, variable: &str) -> Result<Option<String>> {
        if let Some(docker) = self.docker() {
            return Ok(docker.inspect_container(container_id, None).await.map(|config| {
                if let Some(config) = config.config {
                    if let Some(mut envs) = config.env {
                        envs.retain(|x| x.starts_with(variable));
                        return envs.iter().map(String::from).next();
                    }
                }
                None
            })?);
        }
        Ok(None)
    }

    async fn container_https(&self, container_id: &str) -> Result<Option<Vec<model::Http>>> {
        Ok(self
            .container_env_variable(container_id, "DOCKTOR_HTTP")
            .await
            .map(|https| {
                let mut http_list = Vec::new();
                if let Some(https) = https {
                    let split = https.split(',');
                    for token in split {
                        let inner: Vec<&str> = token.split(':').collect();
                        let auth: bool = FromStr::from_str(inner[2]).ok()?;
                        http_list.push(
                            model::Http::builder()
                                .port(inner[0].parse::<i16>().ok()?)
                                .verb(inner[1].to_string())
                                .auth(auth)
                                .build(),
                        );
                    }
                }
                Some(http_list)
            })?)
    }

    async fn container_tcps(&self, container_id: &str) -> Result<Option<Vec<model::Tcp>>> {
        Ok(self
            .container_env_variable(container_id, "DOCKTOR_TCP")
            .await
            .map(|tcps| {
                let mut tcp_list = Vec::new();
                if let Some(tcps) = tcps {
                    let split = tcps.split(',');
                    for token in split {
                        let inner: Vec<&str> = token.split(':').collect();
                        tcp_list.push(
                            model::Tcp::builder()
                                .internal(inner[0].parse::<i16>().ok()?)
                                .external(inner[1].parse::<i16>().ok()?)
                                .build(),
                        );
                    }
                }
                Some(tcp_list)
            })?)
    }

    async fn container_monitorings(&self, container_id: &str) -> Result<Option<Vec<model::Monitoring>>> {
        Ok(self
            .container_env_variable(container_id, "DOCKTOR_MONITOR")
            .await
            .map(|monitors| {
                let mut monitor_list = Vec::new();
                if let Some(monitors) = monitors {
                    let split = monitors.split(',');
                    for token in split {
                        let inner: Vec<&str> = token.split(':').collect();
                        monitor_list.push(
                            model::Monitoring::builder()
                                .url(inner[0].to_string())
                                .port(inner[1].parse::<i16>().ok()?)
                                .build(),
                        );
                    }
                }
                Some(monitor_list)
            })?)
    }

    async fn container_healthcheck(&self, container_id: &str) -> Result<Option<model::Healthcheck>> {
        if let Some(docker) = self.docker() {
            return Ok(docker.inspect_container(container_id, None).await.map(|config| {
                if let Some(state) = config.state {
                    if let Some(health) = state.health {
                        let mut exit_code = -1;
                        let mut message = None;
                        if let Some(logs) = health.log {
                            let entry = logs.get(0).unwrap_or(&HealthcheckResult {
                                start: None,
                                end: None,
                                exit_code: None,
                                output: None,
                            });
                            exit_code = entry.exit_code.unwrap_or(-1);
                            message = entry.output.clone()
                        }
                        return Some(
                            model::Healthcheck::builder()
                                .status(
                                    health
                                        .status
                                        .map(|x| x.to_string())
                                        .unwrap_or_else(|| "unknown".to_string()),
                                )
                                .failing_streak(health.failing_streak.unwrap_or(-1))
                                .exit_code(exit_code)
                                .set_message(message)
                                .build(),
                        );
                    }
                }
                None
            })?);
        }
        Ok(None)
    }

    async fn run_containser_visit(&self, containers: Vec<ContainerSummaryInner>) -> Result<Vec<model::Service>> {
        let mut services = Vec::new();
        if containers.is_empty() {
            error!("No running containers found on this host");
        } else {
            let default_weave_domain = "weave.local".to_string();
            debug!("Scanning {} containers", containers.len());
            for container in containers {
                let unknown = "unknown".to_string();
                let container_id = container.id.as_ref().unwrap_or(&unknown);
                let container_name = if let Some(names) = container.names.as_ref() {
                    names.get(0).unwrap_or(&unknown)
                } else {
                    &unknown
                };
                let service_name = self.container_env_variable(container_id, "DOCKTOR_SERVICE").await?;
                let pod = self
                    .container_env_variable(container_id, "DOCKTOR_POD")
                    .await?
                    .unwrap_or_else(|| container_name.to_string());
                let https = self.container_https(container_id).await?;
                let tcps = self.container_tcps(container_id).await?;
                let monitorings = self.container_monitorings(container_id).await?;
                let healthcheck = self.container_healthcheck(container_id).await?;
                let weave_domain = self
                    .config
                    .weave_domain
                    .as_ref()
                    .unwrap_or(&default_weave_domain)
                    .to_string();
                let hostname = format!("{}.{}", container_name, weave_domain);
                let weave = model::WeaveInfo::builder()
                    .ip(self.weave().dns_lookup(&hostname).await)
                    .domain(weave_domain)
                    .build();
                services.push(
                    model::Service::builder()
                        .id(container_id.to_string())
                        .pod(pod)
                        .name(container_name.replacen("/", "", 1))
                        .set_service(service_name)
                        .host(self.host_info().await)
                        .weave(weave)
                        .set_tcps(tcps)
                        .set_https(https)
                        .set_monitorings(monitorings)
                        .set_healthcheck(healthcheck)
                        .build(),
                );
            }
        }
        Ok(services)
    }

    async fn containers(&self, data: &mut model::ListOutput) {
        match self.docker() {
            Some(docker) => {
                let mut filter = HashMap::new();
                filter.insert(String::from("status"), vec![String::from("running")]);
                match docker
                    .list_containers(Some(ListContainersOptions {
                        all: true,
                        filters: filter,
                        ..Default::default()
                    }))
                    .await
                {
                    Ok(containers) => match self.run_containser_visit(containers).await {
                        Ok(services) => {
                            debug!("Injecting containers services into data hold: {:#?}", services);
                            data.services = Some(services);
                        }
                        Err(e) => error!("Error fetching docker containers: {}", e),
                    },
                    Err(e) => error!("Error fetching docker containers: {}", e),
                }
            }
            None => {}
        }
    }

    async fn run_local_command(&self, name: &str, process: Process) -> Result<model::Service> {
        let default_weave_domain = "weave.local".to_string();
        let command = Command::new("sh").arg("-c").arg(&process.command).output().await?;
        let container_id: String = thread_rng()
            .sample_iter(&Alphanumeric)
            .take(64)
            .map(char::from)
            .collect();
        let tcps = vec![model::Tcp::builder()
            .set_external(process.tcp_external)
            .set_internal(process.tcp_internal)
            .build()];
        let https = vec![model::Http::builder()
            .set_port(process.http_port)
            .set_verb(process.http_verb)
            .set_auth(process.http_auth)
            .build()];
        let monitorings = vec![model::Monitoring::builder()
            .hostname(self.config().hostname().clone())
            .set_port(process.monitoring_port)
            .set_url(process.monitoring_url)
            .build()];
        let healthy = if command.status.success() {
            "healthy"
        } else {
            "unhealthy"
        };
        let message = if command.status.success() {
            std::str::from_utf8(&command.stdout)?
        } else {
            std::str::from_utf8(&command.stderr)?
        };
        let healthcheck = model::Healthcheck::builder()
            .exit_code(command.status.code().unwrap_or(-1) as i64)
            .message(message.to_string())
            .status(healthy.to_string())
            .build();
        let weave_domain = self
            .config
            .weave_domain
            .as_ref()
            .unwrap_or(&default_weave_domain)
            .to_string();
        let hostname = format!("{}.{}", name, weave_domain);
        let weave = model::WeaveInfo::builder()
            .ip(self.weave().dns_lookup(&hostname).await)
            .domain(weave_domain)
            .build();
        Ok(model::Service::builder()
            .id(container_id)
            .pod(process.pod.unwrap_or_else(|| "system".to_string()))
            .name(name.to_string())
            .set_service(process.service)
            .host(self.host_info().await)
            .weave(weave)
            .set_tcps(Some(tcps))
            .set_https(Some(https))
            .set_monitorings(Some(monitorings))
            .set_healthcheck(Some(healthcheck))
            .build())
    }

    async fn processes(&self, data: &mut model::ListOutput) {
        let mut local_processes = match self.config().processes().get("global") {
            Some(global) => global.clone(),
            None => HashMap::new(),
        };
        for (host, proc) in self.config().processes().iter() {
            if host == self.config().hostname() {
                for (proc, command) in proc.iter() {
                    local_processes.insert(proc.clone(), command.clone());
                }
            }
        }
        debug!("Running {} processes: {:#?}", local_processes.len(), local_processes);
        let mut services = Vec::new();
        for (name, process) in local_processes {
            match self.run_local_command(&name, process).await {
                Ok(service) => services.push(service),
                Err(e) => error!("Error executing local command {}: {}", name, e),
            }
        }
        data.services.as_mut().unwrap().append(&mut services);
    }

    async fn open_ssh_session(
        &self,
        addr: &str,
        user: &str,
        key: &str,
    ) -> Result<async_ssh2_lite::AsyncSession<TcpStream>> {
        let addr = addr.to_socket_addrs()?.next().unwrap();
        let stream = Async::<TcpStream>::connect(addr).await?;
        let mut session = AsyncSession::new(stream, None)?;
        session.handshake().await?;
        session.userauth_pubkey_file(user, None, Path::new(key), None).await?;
        Ok(session)
    }

    async fn run_ssh_command(&self, name: &str, host: &str, process: &SshProcess) -> Result<model::Service> {
        let default_weave_domain = "weave.local".to_string();
        let mut sessions = self.ssh_sessions().write().await;
        let session = match sessions.get(host) {
            Some(session) => {
                debug!("Found SSH session in cache for {}", host);
                session
            }
            None => {
                let user = process
                    .user
                    .as_ref()
                    .unwrap_or_else(|| self.config().ssh_settings().user());
                let key = process
                    .user
                    .as_ref()
                    .unwrap_or_else(|| self.config().ssh_settings().key());
                debug!(
                    "SSH session cache for {} not found, generating new session for {}@{}, key: {}",
                    host, user, host, key
                );
                sessions.insert(host.to_string(), self.open_ssh_session(host, user, key).await?);
                sessions.get(host).unwrap()
            }
        };

        let mut channel = session.channel_session().await?;
        channel.exec(&process.command).await?;
        let mut message = String::new();
        channel.read_to_string(&mut message).await?;
        channel.close().await?;
        let container_id: String = thread_rng()
            .sample_iter(&Alphanumeric)
            .take(64)
            .map(char::from)
            .collect();
        let tcps = vec![model::Tcp::builder()
            .set_external(process.tcp_external)
            .set_internal(process.tcp_internal)
            .build()];
        let https = vec![model::Http::builder()
            .set_port(process.http_port)
            .set_verb(process.http_verb.clone())
            .set_auth(process.http_auth)
            .build()];
        let monitorings = vec![model::Monitoring::builder()
            .set_port(process.monitoring_port)
            .set_url(process.monitoring_url.clone())
            .build()];
        let healthy = if channel.exit_status()? == 0 {
            "healthy"
        } else {
            "unhealthy"
        };
        let healthcheck = model::Healthcheck::builder()
            .exit_code(channel.exit_status().unwrap_or(-1) as i64)
            .message(message.to_string())
            .status(healthy.to_string())
            .build();
        let weave_domain = self
            .config
            .weave_domain
            .as_ref()
            .unwrap_or(&default_weave_domain)
            .to_string();
        let hostname = format!("{}.{}", name, weave_domain);
        let weave = model::WeaveInfo::builder()
            .ip(self.weave().dns_lookup(&hostname).await)
            .domain(weave_domain)
            .build();
        Ok(model::Service::builder()
            .id(container_id)
            .pod(process.pod.clone().unwrap_or_else(|| "ssh".to_string()))
            .name(name.to_string())
            .set_service(process.service.clone())
            .host(self.host_info().await)
            .weave(weave)
            .set_tcps(Some(tcps))
            .set_https(Some(https))
            .set_monitorings(Some(monitorings))
            .set_healthcheck(Some(healthcheck))
            .build())
    }

    async fn ssh(&self, data: &mut model::ListOutput) {
        for (src_host, commands) in self.config().ssh_processes() {
            if src_host == self.config().hostname() {
                for (dst_host, processes) in commands {
                    debug!("SSH {} processes on {}: {:#?}", processes.len(), dst_host, processes);
                    let mut services = Vec::new();
                    for (name, process) in processes {
                        match self.run_ssh_command(name, dst_host, process).await {
                            Ok(service) => services.push(service),
                            Err(e) => error!("Error executing ssh command {}: {}", name, e),
                        }
                    }
                    data.services.as_mut().unwrap().append(&mut services);
                }
            }
        }
    }

    async fn run_peer_fetch(&self, peer: &str) -> Result<model::ListOutput> {
        let cli = ApiClient::new(peer.to_string(), self.config().api_key().clone())?;
        let response = cli.0.list_operation().send().await?;
        let response = unsafe {
            std::mem::transmute::<docktor_api_client::output::ListOperationOutput, model::ListOutput>(response)
        };
        Ok(response)
    }

    async fn peers(&self, data: &mut model::ListOutput) {
        if *self.primary() || self.config().primary() {
            for peer in self.config().peers() {
                debug!("Peer list: {:#?}", data.peers());
                match self.run_peer_fetch(peer).await {
                    Ok(response) => {
                        debug!("Injecting peer {} into active peers list", peer);
                        data.peers.as_mut().unwrap().insert(peer.to_string(), response);
                    }
                    Err(e) => error!("Error executing peer {} fetch: {}", peer, e),
                }
            }
        }
    }

    async fn resources(&self, data: &mut model::ListOutput) -> Result<()> {
        let sys = System::new();
        let mem = sys.memory()?;
        let loadavg = sys.load_average()?;
        let resources = model::Resources::builder()
            .set_load_avg(Some(
                model::LoadAvg::builder()
                    .set_one(Some(loadavg.one as f32))
                    .set_five(Some(loadavg.five as f32))
                    .set_fifteen(Some(loadavg.fifteen))
                    .build(),
            ))
            .set_cpus(Some(num_cpus::get() as i16))
            .set_memory(Some(
                model::Memory::builder()
                    .set_free(Some(mem.free.as_u64() as f32))
                    .set_total(Some(mem.total.as_u64() as f32))
                    .set_used(Some(saturating_sub_bytes(mem.total, mem.free).as_u64() as f32))
                    .build(),
            ))
            .build();
        data.resources = Some(resources);
        Ok(())
    }

    pub async fn visit(&self) {
        let mut data = model::ListOutput::builder().build();
        let host = Some(
            model::Host::builder()
                .set_hostname(Some(self.config().hostname().clone()))
                .set_domain(Some(self.config().domain().clone()))
                .build(),
        );
        debug!("Injecting host info into data hold: {:#?}", host);
        data.host = host;
        debug!("Injecting host primary flag: {}", self.config().primary);
        data.primary = Some(self.config().primary.clone());
        data.services = Some(Vec::new());
        data.peers = Some(HashMap::new());
        self.containers(&mut data).await;
        self.processes(&mut data).await;
        self.ssh(&mut data).await;
        self.peers(&mut data).await;
        self.resources(&mut data)
            .await
            .unwrap_or_else(|e| error!("Error running resource collection: {}", e));
        debug!("Storing data inside state storage");
        let mut store_data = self.data().write().await;
        *store_data = data;
    }
}
