use std::{
    collections::HashMap,
    future::Future,
    pin::Pin,
    sync::Arc,
    task::{Context, Poll},
};

use aws_smithy_http_server::{to_boxed, Body, BoxBody};
use docktor_api::{model, output};
use futures_util::ready;
use getset::{Getters, MutGetters, Setters};
use http::{Request, Response};
use pin_project_lite::pin_project;
use tokio::sync::RwLock;
use tower::{Layer, Service};
use tracing::debug;

use crate::{error::DocktorError, Config};

pin_project! {
    /// [`crate::QueryLogLayer`] response future.
    pub struct ResponseFuture<F> {
        #[pin]
        future: F,
        path: String,
    }
}

impl<F> ResponseFuture<F> {
    /// Create a new [ResponseFuture].
    pub(crate) fn new(future: F, path: String) -> Self {
        ResponseFuture { future, path }
    }
}

impl<F, E> Future for ResponseFuture<F>
where
    F: Future<Output = Result<Response<BoxBody>, E>>,
{
    type Output = F::Output;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        let this = self.project();
        let response = ready!(this.future.poll(cx)?);
        if this.path == "/prometheus/targets" {
            let mut f = hyper::body::to_bytes(response.into_body());
            let fut = unsafe { Pin::new_unchecked(&mut f) };
            let response = ready!(fut.poll(cx));
            debug!("Prometheus layer enabled for route {}, removing JSON outer dict and returning only the list of targets", this.path);
            match response {
                Ok(r) => {
                    // TODO: I am removing the JSON I know it shouldn't be there because smithy doesn't
                    // allow to have a list as return structure.
                    // THIS IS HORRIFYING.
                    let mut s = String::from_utf8_lossy(&r).replacen("{\"targets\":", "", 1);
                    s.pop();
                    let body = Response::builder().body(to_boxed(Body::from(s))).unwrap();
                    Poll::Ready(Ok(body))
                }
                Err(_) => panic!("Fuck it"),
            }
        } else {
            debug!(
                "Prometheus layer disabled for route {}, returning output as it is",
                this.path
            );
            Poll::Ready(Ok(response))
        }
    }
}

#[derive(Debug, Default, Clone)]
pub struct PrometheusService<S> {
    inner: S,
}

impl<S> PrometheusService<S> {
    pub fn new(inner: S) -> Self {
        Self { inner }
    }
}

impl<ReqBody, S> Service<Request<ReqBody>> for PrometheusService<S>
where
    S: Service<Request<ReqBody>, Response = Response<BoxBody>>,
{
    type Response = S::Response;
    type Error = S::Error;
    type Future = ResponseFuture<S::Future>;

    #[inline]
    fn poll_ready(&mut self, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        self.inner.poll_ready(cx)
    }

    fn call(&mut self, request: Request<ReqBody>) -> Self::Future {
        let path = request.uri().to_string();
        let response = self.inner.call(request);
        ResponseFuture::new(response, path)
    }
}

#[derive(Debug, Default, Clone)]
pub struct PrometheusLayer {}

impl<S> Layer<S> for PrometheusLayer {
    type Service = PrometheusService<S>;

    #[inline]
    fn layer(&self, service: S) -> Self::Service {
        PrometheusService::new(service)
    }
}

#[derive(Debug, Clone, Getters, Setters, MutGetters)]
pub struct PrometheusTargets {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    config: Arc<Config>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    data: Arc<RwLock<model::ListOutput>>,
}

impl PrometheusTargets {
    pub fn new(config: Arc<Config>, data: Arc<RwLock<model::ListOutput>>) -> Self {
        Self { config, data }
    }

    pub async fn targets(&self) -> Result<output::PrometheusTargetOperationOutput, DocktorError> {
        let data = self.data().read().await;
        let services = data.services().unwrap_or_default();
        let default_url = "metrics".to_string();
        let mut targets = Vec::new();
        for service in services {
            if let Some(monitorings) = service.monitorings.as_ref() {
                for monitoring in monitorings {
                    if let Some(port) = monitoring.port {
                        let target = format!(
                            "{}-{}.{}:{}",
                            service.name.as_ref().unwrap(),
                            monitoring.hostname.as_ref().unwrap(),
                            service.weave.as_ref().unwrap().domain.as_ref().unwrap(),
                            port,
                        );
                        let mut labels = HashMap::new();
                        labels.insert("primary".to_string(), self.config().primary.clone());
                        labels.insert("domain".to_string(), self.config().domain.clone());
                        labels.insert(
                            "hostname".to_string(),
                            service.host.as_ref().unwrap().hostname.as_ref().unwrap().to_string(),
                        );
                        labels.insert("pod".to_string(), service.pod.as_ref().unwrap().to_string());
                        let result = model::PrometheusTarget::builder()
                            .set_targets(Some(vec![target]))
                            .set_labels(Some(labels))
                            .build();
                        targets.push(result);
                    }
                }
            }
        }
        Ok(output::PrometheusTargetOperationOutput::builder()
            .set_targets(Some(targets))
            .build())
    }
}
