use std::sync::Arc;

use aws_smithy_http_server::Extension;
use docktor_api::{error, input, model, output};

use crate::server::State;

#[tracing::instrument(name = "list")]
pub async fn list(_: input::ListOperationInput, state: Extension<Arc<State>>) -> output::ListOperationOutput {
    let response = state.0.data.read().await;
    unsafe { std::mem::transmute::<model::ListOutput, output::ListOperationOutput>(response.clone()) }
}

#[tracing::instrument(name = "healthcheck")]
pub async fn healthcheck(
    _: input::HealthcheckOperationInput,
    _: Extension<Arc<State>>,
) -> Result<output::HealthcheckOperationOutput, error::HealthcheckOperationError> {
    Ok(output::HealthcheckOperationOutput::builder()
        .message("ok".to_string())
        .build())
}

#[tracing::instrument(name = "start")]
pub async fn start(
    input: input::StartOperationInput,
    state: Extension<Arc<State>>,
) -> Result<output::StartOperationOutput, error::StartOperationError> {
    state
        .0
        .systemd
        .start(input.services().unwrap_or_default(), input.hostname())
        .await
        .map_err(|e| {
            error::StartOperationError::SystemdError(
                error::SystemdError::builder().set_message(Some(e.to_string())).build(),
            )
        })
}

#[tracing::instrument(name = "stop")]
pub async fn stop(
    input: input::StopOperationInput,
    state: Extension<Arc<State>>,
) -> Result<output::StopOperationOutput, error::StopOperationError> {
    state
        .0
        .systemd
        .stop(input.services().unwrap_or_default(), input.hostname())
        .await
        .map_err(|e| {
            error::StopOperationError::SystemdError(
                error::SystemdError::builder().set_message(Some(e.to_string())).build(),
            )
        })
}

#[tracing::instrument(name = "restart")]
pub async fn restart(
    input: input::RestartOperationInput,
    state: Extension<Arc<State>>,
) -> Result<output::RestartOperationOutput, error::RestartOperationError> {
    state
        .0
        .systemd
        .restart(input.services().unwrap_or_default(), input.hostname())
        .await
        .map_err(|e| {
            error::RestartOperationError::SystemdError(
                error::SystemdError::builder().set_message(Some(e.to_string())).build(),
            )
        })
}

#[tracing::instrument(name = "prometheus_target")]
pub async fn prometheus_target(
    _: input::PrometheusTargetOperationInput,
    state: Extension<Arc<State>>,
) -> Result<output::PrometheusTargetOperationOutput, error::PrometheusTargetOperationError> {
    state.0.prometheus_targets.targets().await.map_err(|e| {
        error::PrometheusTargetOperationError::PrometheusError(
            error::PrometheusError::builder()
                .set_message(Some(e.to_string()))
                .build(),
        )
    })
}
