use std::path::{Path, PathBuf};
use std::sync::Arc;
use std::time::Duration;
use std::{env, net::SocketAddr};

use aws_smithy_http_server::{AddExtensionLayer, Router};
use clap::Parser;
use docktor_api::operation_registry::OperationRegistryBuilder;
use eyre::{bail, Result};
use tower::ServiceBuilder;
use tower_http::{
    auth::RequireAuthorizationLayer,
    trace::{DefaultMakeSpan, DefaultOnRequest, DefaultOnResponse, TraceLayer},
    LatencyUnit,
};
use tracing::{debug, info, Level};

use self::operations::{healthcheck, list, prometheus_target, restart, start, stop};
pub use self::state::State;
use crate::config::Config;
use crate::server::prometheus::PrometheusLayer;

mod operations;
mod prometheus;
mod state;

#[derive(Parser, Debug, Clone)]
pub struct Server {
    #[clap(short = 'b', long = "bind")]
    pub bind: Option<String>,
    /// This is a primary worker.
    #[clap(short = 'p', long = "primary")]
    pub primary: bool,
}

fn find_in_path<P>(exe_name: P) -> Option<PathBuf>
where
    P: AsRef<Path>,
{
    env::var_os("PATH").and_then(|paths| {
        env::split_paths(&paths).find_map(|dir| {
            let full_path = dir.join(&exe_name);
            if full_path.is_file() {
                Some(full_path)
            } else {
                None
            }
        })
    })
}

impl Server {
    fn validate_dependencies(&self) -> Result<()> {
        if find_in_path("docker").is_none() {
            bail!("Docker executable not found, please follow installation instruction at https://github.com/crisidev/docktor")
        } else if find_in_path("systemctl").is_none() {
            bail!("Systemd executable not found, please follow installation instruction at https://github.com/crisidev/docktor")
        } else if find_in_path("weave").is_none() {
            bail!("Weave executable not found, please follow installation instruction at https://github.com/crisidev/docktor")
        } else {
            Ok(())
        }
    }
    pub async fn exec(&self) -> Result<()> {
        self.validate_dependencies()?;
        let format = tracing_subscriber::fmt::format()
            .with_ansi(true)
            .with_line_number(true)
            .with_file(true)
            .with_level(true)
            .with_source_location(true)
            .compact();
        tracing_subscriber::fmt().event_format(format).init();
        let config = Config::load().await?;
        if config.debug {
            env::set_var("RUST_LOG", "debug");
        } else {
            env::set_var("RUST_LOG", ":info");
        }
        let app: Router = OperationRegistryBuilder::default()
            // Build a registry containing implementations to all the operations in the service. These
            // are async functions or async closures that take as input the operation's input and
            // return the operation's output.
            .list_operation(list)
            .healthcheck_operation(healthcheck)
            .start_operation(start)
            .stop_operation(stop)
            .restart_operation(restart)
            .prometheus_target_operation(prometheus_target)
            .build()
            .expect("Unable to build operation registry")
            // Convert it into a router that will route requests to the matching operation
            // implementation.
            .into();

        // Create a new state management structure, wrapped inside an Arc and build the application
        // middleware using [`ServiceBuilder`].
        //
        // [`ServiceBuilder`]: [`tower::ServiceBuilder`]
        let config = Arc::new(config);
        let shared_state = Arc::new(State::new(config.clone(), self.primary)?);
        let task_state = shared_state.clone();
        let task_config = config.clone();
        tokio::task::spawn(async move {
            loop {
                task_state.visit().await;
                debug!(
                    "Sleeping {} seconds before next visit",
                    task_config.background_fetch_interval
                );
                tokio::time::sleep(Duration::from_secs(task_config.background_fetch_interval)).await;
            }
        });
        let level = if config.debug { Level::DEBUG } else { Level::INFO };
        let app = app.layer(
            ServiceBuilder::new()
                .layer(
                    TraceLayer::new_for_http()
                        .make_span_with(DefaultMakeSpan::new().include_headers(true))
                        .on_request(DefaultOnRequest::new().level(level))
                        .on_response(DefaultOnResponse::new().level(level).latency_unit(LatencyUnit::Millis)),
                )
                .layer(AddExtensionLayer::new(shared_state))
                .layer(RequireAuthorizationLayer::bearer(&config.api_key))
                .layer(PrometheusLayer::default()),
        );

        // Start the [`axum server`]. Note that also an [`hyper` server`] can be used.
        //
        // [`axum server`]: [`axum::Server`]
        // [`hyper server`]: [`hyper::Server`]
        let bind: SocketAddr = match self.bind.as_ref() {
            Some(bind) => bind.parse()?,
            None => config.bind()?,
        };
        info!("Starting Docktor server at http://{}", &bind);
        let server = axum_server::bind(bind).serve(app.into_make_service());

        // Run forever-ish...
        Ok(server.await?)
    }
}
