use clap::Parser;
use docktor::{Client, Dns, Server, System};
use eyre::Result;

/// Docktor Container Orchestration System.
#[derive(Parser, Debug, Clone)]
#[clap(author, about, version)]
pub struct Docktor {
    /// Docktor subcommand.
    #[clap(subcommand)]
    pub action: Action,
    /// Use debug.
    #[clap(short = 'D', long = "debug")]
    pub debug: bool,
}

impl Docktor {
    pub async fn exec(&self) -> Result<()> {
        color_eyre::install()?;
        match self.action.clone() {
            Action::Serve(ref action) => Ok(action.exec().await?),
            Action::Cli(ref action) => Ok(action.exec().await?),
            Action::Start(ref action) => Ok(action.exec().await?),
            Action::Stop(ref action) => Ok(action.exec().await?),
            Action::Restart(ref action) => Ok(action.exec().await?),
            Action::Logs(ref action) => Ok(action.exec().await?),
            Action::Dns(ref action) => Ok(action.exec().await?),
        }
    }
}

/// Docktor subcommand.
#[derive(Parser, Debug, Clone)]
pub enum Action {
    /// Run the server.
    Serve(Server),
    /// Call the client.
    Cli(Client),
    /// Start a service.
    Start(System),
    /// Stop a service.
    Stop(System),
    /// Restart a service.
    Restart(System),
    /// Service logs.
    Logs(System),
    /// Dns management.
    Dns(Dns),
}

#[tokio::main]
pub async fn main() -> Result<()> {
    Docktor::parse().exec().await
}
