use aws_smithy_client::SdkError;
use aws_smithy_http_server::to_boxed;
use axum_core::response::{IntoResponse, Response};
use docktor_api_client::error::{RestartOperationError, StartOperationError, StopOperationError};
use http::StatusCode;
use thiserror::Error;

#[derive(Error, Debug)]
pub enum DocktorError {
    #[error("Docker error")]
    Docker(#[from] bollard::errors::Error),
    #[error("I/O error")]
    Io(#[from] std::io::Error),
    #[error("Systemd error: {0}")]
    Systemd(String),
    #[error("SDK start operation error")]
    SdkStart(#[from] SdkError<StartOperationError>),
    #[error("SDK stop operation error")]
    SdkStop(#[from] SdkError<StopOperationError>),
    #[error("SDK restart operation error")]
    SdkRestart(#[from] SdkError<RestartOperationError>),
    #[error("Eyre")]
    Eyre(#[from] eyre::ErrReport),
    #[error("Weave error: {0}")]
    Weave(String),
    #[error("Parse address error")]
    AddrParse(#[from] std::net::AddrParseError),
}

impl IntoResponse for DocktorError {
    fn into_response(self) -> Response {
        let body = match self {
            DocktorError::Docker(ref e) => to_boxed(e.to_string()),
            DocktorError::Io(ref e) => to_boxed(e.to_string()),
            DocktorError::Systemd(ref e) => to_boxed(e.to_string()),
            DocktorError::SdkStart(ref e) => to_boxed(e.to_string()),
            DocktorError::SdkStop(ref e) => to_boxed(e.to_string()),
            DocktorError::SdkRestart(ref e) => to_boxed(e.to_string()),
            DocktorError::Eyre(ref e) => to_boxed(e.to_string()),
            DocktorError::Weave(ref e) => to_boxed(e.to_string()),
            DocktorError::AddrParse(ref e) => to_boxed(e.to_string()),
        };
        Response::builder()
            .status(StatusCode::INTERNAL_SERVER_ERROR)
            .body(body)
            .unwrap()
    }
}
