use std::sync::Arc;

use clap::Parser;
use eyre::{bail, Result};

use crate::{Config, Weave};

#[derive(Parser, Debug, Clone)]
pub struct Dns {
    /// Dns subcommand.
    #[clap(subcommand)]
    pub action: Action,
    /// Hostname.(mandatory for add and remove).
    #[clap(short = 'H', long = "hostname")]
    pub hostname: Option<String>,
    /// IP address (mandatory for add and remove)..
    #[clap(short = 'i', long = "ip-address")]
    pub ip_address: Option<String>,
}

#[derive(Parser, Debug, Clone)]
pub enum Action {
    /// Add DNS entry.
    Add,
    /// Remove DNS entry.
    Remove,
    /// Lookup DNS name.
    Lookup,
    /// List available DNS entries.
    List,
}

impl Dns {
    pub async fn exec(&self) -> Result<()> {
        let config = Arc::new(Config::load().await?);
        let weave = Weave::new(config);
        match self.action {
            Action::Add | Action::Remove => {
                if self.hostname.is_none() || self.ip_address.is_none() {
                    bail!("--hostname and/or --ip-address missing")
                }
            }
            _ => {}
        }
        let result = match self.action {
            Action::Add => {
                if self.hostname.is_none() || self.ip_address.is_none() {
                    bail!("--hostname and/or --ip-address missing")
                }
                let hostname = self.hostname.as_ref().unwrap();
                let ip_address = self.ip_address.as_ref().unwrap();
                weave.dns_add(hostname, ip_address).await;
                format!("Added DNS name {} pointing to {}", hostname, ip_address)
            }
            Action::Remove => {
                if self.hostname.is_none() || self.ip_address.is_none() {
                    bail!("--hostname and/or --ip-address missing")
                }
                let hostname = self.hostname.as_ref().unwrap();
                let ip_address = self.ip_address.as_ref().unwrap();
                weave.dns_remove(hostname, ip_address).await;
                format!("Removed DNS name {} pointing to {}", hostname, ip_address)
            }
            Action::Lookup => {
                if self.hostname.is_none() {
                    bail!("--hostname missing")
                }
                let hostname = self.hostname.as_ref().unwrap();
                let ip_address = weave.dns_lookup(hostname).await;
                format!("DNS name {} points to {}", hostname, ip_address)
            }
            Action::List => weave.dns_list().await,
        };
        println!("{}", result);
        Ok(())
    }
}
