use std::path::Path;
use std::{collections::HashMap, net::SocketAddr};

use eyre::Result;
use getset::{Getters, MutGetters, Setters};
use pnet::datalink;
use serde::Deserialize;
use tokio::fs;
use tracing::{debug, info};

use crate::error::DocktorError;

#[derive(Debug, Clone, Deserialize, Default, Getters, Setters, MutGetters)]
pub struct Config {
    #[serde(skip)]
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub hostname: String,
    #[serde(skip)]
    pub ip_address: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub debug: bool,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub api_key: String,
    pub bind: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub domain: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub weave_domain: Option<String>,
    pub primary: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub data_dir: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub apps_dir: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub peers: Vec<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub services: HashMap<String, Vec<String>>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub processes: HashMap<String, HashMap<String, Process>>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub ssh_processes: HashMap<String, HashMap<String, HashMap<String, SshProcess>>>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub ssh_settings: SshSettings,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub background_fetch_interval: u64,
}

#[derive(Debug, Clone, Deserialize, Default, Getters, Setters, MutGetters)]
pub struct SshSettings {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub user: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub key: String,
}

#[derive(Debug, Clone, Deserialize, Default, Getters, Setters, MutGetters)]
pub struct Process {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub command: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub tcp_external: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub tcp_internal: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub http_port: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub http_auth: Option<bool>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub http_verb: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub monitoring_port: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub monitoring_url: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub pod: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub service: Option<String>,
}

#[derive(Debug, Clone, Deserialize, Default, Getters, Setters, MutGetters)]
pub struct SshProcess {
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub command: String,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub user: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub key: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub tcp_external: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub tcp_internal: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub http_port: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub http_auth: Option<bool>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub http_verb: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub monitoring_port: Option<i16>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub monitoring_url: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub pod: Option<String>,
    #[getset(get = "pub", set = "pub", get_mut = "pub")]
    pub service: Option<String>,
}

impl Config {
    pub async fn load() -> Result<Self> {
        let config_dir = match dirs::home_dir() {
            Some(dir) => Path::new(&dir).join(".config/docktor"),
            None => Path::new("/etc").join("docktor"),
        };
        let config_path = config_dir.join("config.toml");
        info!("Loading configuration from {}", config_path.display());
        let config_string = fs::read_to_string(&config_path).await?;
        let mut config: Config = toml::from_str(&config_string)?;
        config.hostname = hostname::get()?
            .into_string()
            .unwrap_or_default()
            .split('.')
            .next()
            .unwrap_or_default()
            .to_string();
        config.ip_address = config.find_public_ip()?;
        debug!("Configuration loaded from {}:\n{:#?}", config_path.display(), config);
        Ok(config)
    }

    /// Find the public IP by looping over all the available interfaces and finding a public
    /// routable interface with an IP address which can be reached from the outside.
    /// This method currently works only on *nix.
    fn find_public_ip(&self) -> Result<String, DocktorError> {
        let all_interfaces = datalink::interfaces();
        let default_interface = all_interfaces
            .iter()
            .find(|e| e.is_up() && !e.is_loopback() && !e.ips.is_empty());

        let mut ip_address = "127.0.0.1".parse()?;
        match default_interface {
            Some(interface) => {
                for ip in interface.ips.iter() {
                    // TODO: support IPv6
                    if ip.is_ipv4() {
                        ip_address = ip.ip();
                        break;
                    }
                }
                if !ip_address.is_loopback() {
                    debug!("Found IP address {} for interface {}", ip_address, interface.name);
                    Ok(ip_address.to_string())
                } else {
                    Err(DocktorError::Weave(
                        "Unable to find a valid global IP address".to_string(),
                    ))
                }
            }
            None => Err(DocktorError::Weave(
                "Unable to find default network interface".to_string(),
            )),
        }
    }

    pub fn bind(&self) -> Result<SocketAddr> {
        Ok(self.bind.parse()?)
    }

    pub fn primary(&self) -> bool {
        self.hostname == self.primary
    }

    pub fn ip_address(&self) -> &str {
        &self.ip_address
    }
}
