use proc_macro::{Group, Ident, Literal, Punct, Span, TokenStream, TokenTree};
use quote::{quote, ToTokens};
use syn::parse_macro_input;

#[proc_macro_attribute]
pub fn doc_since(args: TokenStream, input: TokenStream) -> TokenStream {
    let mut out = TokenStream::new();

    let args = parse_macro_input!(args as syn::AttributeArgs);
    let since_version = match args.first() {
        Some(syn::NestedMeta::Lit(syn::Lit::Str(lit))) => lit.value(),
        _ => todo!("macro input not legal"),
    };

    let mut ast = match syn::parse::<syn::ItemFn>(input.clone()) {
        Ok(ast) => ast,
        // on parse error, make IDEs happy; see fn docs
        Err(err) => return input_and_compile_error(input, err),
    };

    let mut insert_doc_at = None;

    for (i, attr) in ast.attrs.iter().enumerate() {
        if attr_is_empty_doc(attr) {
            insert_doc_at = Some(i);
        }
    }

    if let Some(i) = insert_doc_at {
        out.extend(
            ast.attrs
                .drain(..i)
                .map(ToTokens::into_token_stream)
                .map(TokenStream::from),
        );

        out.extend(doc_since_token_stream(&since_version))
    }

    out.extend([TokenStream::from(quote! { #ast })]);

    out
}

fn attr_is_empty_doc(attr: &syn::Attribute) -> bool {
    let path_is_doc = {
        if let Some(seg) = attr.path.segments.first() {
            seg.ident == "doc"
        } else {
            false
        }
    };

    if path_is_doc {
        let tt_iter = attr.tokens.clone().into_iter();
        let tt = tt_iter.last().unwrap();
        if let proc_macro2::TokenTree::Literal(lit) = tt {
            // literal is empty doc comment string
            if lit.to_string() == "\"\"" {
                return true;
            }
        }
    }

    false
}

fn doc_since_token_stream(since: &str) -> impl IntoIterator<Item = TokenTree> {
    [
        TokenTree::Punct(Punct::new('#', proc_macro::Spacing::Alone)),
        TokenTree::Group(Group::new(
            proc_macro::Delimiter::Bracket,
            TokenStream::from_iter([
                TokenTree::Ident(Ident::new("doc", Span::call_site())),
                TokenTree::Punct(Punct::new('=', proc_macro::Spacing::Alone)),
                TokenTree::Literal(Literal::string("")),
            ]),
        )),
        TokenTree::Punct(Punct::new('#', proc_macro::Spacing::Alone)),
        TokenTree::Group(Group::new(
            proc_macro::Delimiter::Bracket,
            TokenStream::from_iter([
                TokenTree::Ident(Ident::new("doc", Span::call_site())),
                TokenTree::Punct(Punct::new('=', proc_macro::Spacing::Alone)),
                TokenTree::Literal(Literal::string(&format!(
                    " Available since crate version: **{}**",
                    since
                ))),
            ]),
        )),
    ]
}

/// Converts the error to a token stream and appends it to the original input.
///
/// Returning the original input in addition to the error is good for IDEs which can gracefully
/// recover and show more precise errors within the macro body.
///
/// See <https://github.com/rust-analyzer/rust-analyzer/issues/10468> for more info.
fn input_and_compile_error(mut item: TokenStream, err: syn::Error) -> TokenStream {
    let compile_err = TokenStream::from(err.to_compile_error());
    item.extend(compile_err);
    item
}
