#![cfg_attr(not(doctest), doc = include_str!("../README.md") )]
#![warn(missing_docs)]
#![deny(rustdoc::broken_intra_doc_links)]
#![warn(clippy::pedantic)]

use proc_macro::TokenStream;
use quote::quote;

/// Generates a new `Scope` and `Id` types for the item that was derived upon.
///
/// # Example
///
/// ```
/// #[derive(Dmv)]
/// pub(super) struct Foo {
///     id: FooId,
/// }
/// ```
///
/// The code snippet above will be expanded by the derive macro into
///
/// ```
/// #[derive(Hash)]
/// pub(super) struct FooScope;
/// pub(super) type FooId = ::dmv::Id<FooScope>;
/// pub(super) struct Foo {
///     id: FooId,
/// }
/// ```
#[proc_macro_derive(Dmv)]
pub fn derive(input: TokenStream) -> TokenStream {
    let ast = syn::parse_macro_input!(input as syn::DeriveInput);

    let vis = ast.vis;
    let orig_ident = ast.ident;
    let orig_span = orig_ident.span();
    let scope_ident = syn::Ident::new(&format!("{}Scope", orig_ident), orig_span);
    let id_ident = syn::Ident::new(&format!("{}Id", orig_ident), orig_span);

    let out_stream = quote! {
        #[derive(Hash)]
        #vis struct #scope_ident;
        impl ::dmv::Scope for #scope_ident {}
        #vis type #id_ident = ::dmv::Id<#scope_ident>;
    };
    out_stream.into()
}
