/// Trait describing any type that can be used to restrict the scope of
/// uniqueness of a [`crate::Id`].
pub trait Scope: std::any::Any {}

/// Simple macro filling out the boiler-plate code of creating a new scope type
/// called `$ident` with optional visibility `$vis`, and implementing the
/// [`Scope`] trait.
///
/// # Examples
///
/// A macro call of
///
/// ```
/// # macro_rules! scope { ($($t:tt)*) => {}; }
/// scope!{ pub FooScope };
/// ```
///
/// expands into
///
/// ```
/// pub struct FooScope;
/// impl ::dmv::scope::Scope for FooScope {}
/// ```
#[macro_export]
macro_rules! scope {
    ($vis:vis $ident:ident) => {
        scope_details!{$vis $ident ::dmv::scope::Scope };
    };
    ($($t:tt)*) => {
        compile_error!(concat!(
            "encountered unexpected tokens: ",
            stringify!($($t)*),
            "\nhelp: this macro accepts input in the form `$vis $ident`, for ",
            "example: `scope!{pub Foo}` or `scope!{Bar}`."
        ));
    };
}

#[macro_export]
#[doc(hidden)]
macro_rules! scope_details {
    ($vis:vis $ident:ident $scope_path:path) => {
        $vis struct $ident;
        impl $scope_path for $ident {}
    };
    ($($t:tt)*) => {
        compile_error!(concat!(
            "encountered unexpected tokens: ",
            stringify!($($t)*),
            "\nhelp: this macro accepts input in the form `$vis $ident $path`, ",
            "for example: `scope_details!{pub Foo dmv::Scope}`."
        ));
    };
}

#[macro_export]
#[cfg(test)]
#[doc(hidden)]
macro_rules! local_scope {
    ($vis:vis $ident:ident) => {
        scope_details! {$vis $ident crate::Scope}
    };
    ($($t:tt)*) => {
        compile_error!(concat!(
            "encountered unexpected tokens: ",
            stringify!($($t)*),
            "\nhelp: this macro accepts input in the form `$vis $ident`, for ",
            "example: `local_scope!{pub Foo}` or `local_scope!{Bar}`."
        ));
    };
}

/// Globally public [`Scope`].
#[allow(clippy::module_name_repetitions)]
pub struct GlobalScope;
impl Scope for GlobalScope {}
