use core::fmt;
use std::{fmt::Debug, hash::Hash, marker::PhantomData, ops::Deref};

use crate::{id::RawRepr, DmvId, Scope};

#[allow(clippy::module_name_repetitions)]
pub struct DmvHandle<S: Scope, T: RawRepr>(T, PhantomData<*const S>);

impl<S: Scope, T: RawRepr> DmvHandle<S, T> {
    pub(super) fn new(id: &DmvId<S, T>) -> Self {
        Self(id.0, PhantomData)
    }
}

impl<S: Scope, T: RawRepr> Copy for DmvHandle<S, T> {}

impl<S: Scope, T: RawRepr> Clone for DmvHandle<S, T> {
    fn clone(&self) -> Self {
        Self(self.0, PhantomData)
    }
}

impl<S: Scope, T: RawRepr> Deref for DmvHandle<S, T> {
    type Target = T;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl<S: Scope, T: Debug + RawRepr> Debug for DmvHandle<S, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Debug::fmt(&self.0, f)
    }
}

impl<S: Scope, T: RawRepr> PartialEq for DmvHandle<S, T> {
    fn eq(&self, other: &Self) -> bool {
        PartialEq::eq(&self.0, &other.0)
    }
}

impl<S: Scope, T: RawRepr> Eq for DmvHandle<S, T> {}

impl<S: Scope, T: RawRepr> PartialOrd for DmvHandle<S, T> {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        PartialOrd::partial_cmp(&self.0, &other.0)
    }
}

impl<S: Scope, T: RawRepr> Ord for DmvHandle<S, T> {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        Ord::cmp(&self.0, &other.0)
    }
}

impl<S: Scope, T: RawRepr + Hash> Hash for DmvHandle<S, T> {
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        Hash::hash(&self.0, state);
    }
}
