use core::{
    fmt::{self, Debug, Display},
    hash::Hash,
};
use std::{marker::PhantomData, pin::Pin};

use crate::{Comparable, DmvId, Id as IdT, RawRepr, Scope};

#[cfg(doc)]
use crate::Dmv;

use self::handle::DmvHandle;

pub mod handle;

/// The [`Id`] part of the [`Dmv`] system.
///
/// See the [`Dmv`] and [`Id`] documentation for more details.
pub struct Id<S: Scope, T: Comparable>(T, PhantomData<*const S>);

impl<S: Scope, T: Comparable> Id<S, T> {
    pub(crate) fn new(val: T) -> Pin<Box<Self>> {
        Box::pin(Self(val, PhantomData))
    }
}

impl<S: Scope, T: RawRepr + Comparable> IdT<S, T> for DmvId<S, T> {
    type Handle = DmvHandle<S, T>;

    fn handle(&self) -> Self::Handle {
        DmvHandle::new(self)
    }
}

impl<S: Scope, T: Display + Comparable> Display for DmvId<S, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Display::fmt(&self.0, f)
    }
}

impl<S: Scope, T: Debug + Comparable> Debug for DmvId<S, T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Debug::fmt(&self.0, f)
    }
}

impl<S: Scope, T: Comparable> PartialEq for Id<S, T> {
    fn eq(&self, other: &Self) -> bool {
        PartialEq::eq(&self.0, &other.0)
    }
}

impl<S: Scope, T: Comparable> Eq for Id<S, T> {}

impl<S: Scope, T: Comparable> PartialOrd for Id<S, T> {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        PartialOrd::partial_cmp(&self.0, &other.0)
    }
}

impl<S: Scope, T: Comparable> Ord for Id<S, T> {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        Ord::cmp(&self.0, &other.0)
    }
}

impl<S: Scope, T: Comparable + Hash> Hash for Id<S, T> {
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        Hash::hash(&self.0, state);
    }
}
