use std::{any::Any, fmt::{Debug, Display}, hash::Hash, ops::{Add, Deref}};

use crate::Scope;

/// Useful trait aliases of [`Id`] with common types.
pub mod aliases;

pub trait Comparable = Debug + PartialEq + Eq + PartialOrd + Ord + Hash;
pub trait RawRepr = Any + Add<Output = Self> + Copy + Display + Comparable;

/// Trait describing unique identifiers.
///
/// IDs are values which can be used as a method of differentiating one
/// object from another. As such there should not be a way to *safely* create an
/// ID object which would compare equal to another separate ID object, therefore
/// this trait intentionally does not require [`Copy`] nor [`Clone`], and types
/// implementing this trait should not implement them.
pub trait Id<S: Scope, T: RawRepr>: Comparable {
    /// The type that can be used as a way of accessing the ID's value without
    /// taking ownership of the ID itself.
    type Handle<'handle>: Deref<Target = T> = &'handle T;

    /// Returns a handle to this ID.
    ///
    /// See [`Self::Handle`] documentation for more details.
    fn handle<'id, 'handle>(&'id self) -> Self::Handle<'handle> where 'id: 'handle;
}
