/// Trait describing any type that can be used to restrict the scope of
/// uniqueness of a [`crate::Id`].
pub trait Scope: std::any::Any {}

/// Simple macro filling out the boiler-plate code of creating a new scope type
/// called `$ident` with optional visibility `$vis`, and implementing the
/// [`Scope`] trait and its dependent traits for the new type.
///
/// # Examples
///
/// A macro call of
///
/// ```
/// # use dmv::scope;
/// scope!{ pub FooScope };
/// ```
///
/// expands into
///
/// ```
/// #[derive(Hash)]
/// pub struct FooScope;
/// impl ::dmv::scope::Scope for FooScope {}
/// ```
#[macro_export]
macro_rules! scope {
    ($vis:vis $ident:ident) => {
        #[derive(Hash)]
        $vis struct $ident;
        impl ::dmv::scope::Scope for $ident {}
    };
    ($($t:tt)*) => {
        compile_error!(concat!(
            "encountered unexpected tokens: ",
            stringify!($($t)*),
            "\nhelp: this macro accepts input in the form `$vis $ident`, for ",
            "example: `scope!{pub Foo}` or `scope!{Bar}`."
        ));
    };
}

/// Globally public [`Scope`] for use with [`crate::Id`].
#[allow(clippy::module_name_repetitions)]
pub struct GlobalScope;
impl Scope for GlobalScope {}
