use core::{
    fmt::{self, Debug, Display},
    hash::Hash,
};
use std::{any::TypeId, collections::HashMap};

use num_traits::Num;

use crate::{id::Comparable, Id};
use crate::{RawRepr, Scope};

#[derive(Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub(super) struct DmvId<T: Comparable>(T);

impl<T: Comparable> DmvId<T> {
    pub fn new(val: T) -> Self {
        Self(val)
    }
}

impl<S: Scope, T: RawRepr + Comparable> Id<S, T> for DmvId<T> {
    fn handle<'id, 'handle>(&'id self) -> Self::Handle<'handle>
    where
        'id: 'handle,
    {
        &self.0
    }
}

impl<T: Display + Comparable> Display for DmvId<T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Display::fmt(&self.0, f)
    }
}
pub struct ScopeMap<T: Copy> {
    map: HashMap<TypeId, T>,
}

impl<T: Copy + Num> ScopeMap<T> {
    pub fn new() -> Self {
        Self {
            map: HashMap::new(),
        }
    }

    pub fn get(&mut self, scope: TypeId) -> Option<T> {
        let val_ref = self.map.get_mut(&scope)?;
        let out_val = *val_ref;
        *val_ref = *val_ref + T::one();

        Some(out_val)
    }

    pub fn set(&mut self, scope: TypeId, val: T) {
        self.map.insert(scope, val);
    }

    pub fn has(&self, scope: TypeId) -> bool {
        self.map.contains_key(&scope)
    }
}
