/// Trait describing any type that can be used to restrict the scope of
/// uniqueness of a [`crate::Id`].

pub trait Scope: std::any::Any {}

/// Simple macro filling out the boiler-plate code of creating a new scope type
/// called `$ident` with optional visibility `$vis`, and implementing the
/// [`Scope`] trait and its dependent traits for the new type.
///
/// Using this macro directly is unneeded in most use cases as there is a derive
/// macro available now that takes care of even more boiler-plate. For usage of
/// the derive macro, see the [`crate::Dmv`] docs.
///
/// # Examples
///
/// A macro call of
///
/// ```
/// # use dmv::scope;
/// scope!{ pub FooScope };
/// ```
///
/// expands into
///
/// ```
/// #[derive(Hash)]
/// pub struct FooScope;
/// impl ::dmv::scope::Scope for FooScope {}
/// ```
#[macro_export]
macro_rules! scope {
    ($vis:vis $ident:ident) => {
        #[derive(Hash)]
        $vis struct $ident;
        impl ::dmv::scope::Scope for $ident {}
    };
    ($($t:tt)*) => {
        compile_error!(concat!("This macro accepts input in the form `$vis $ident`. For example: `scope!{pub Foo}` or `scope!{Bar}`"))
    };
}

/// Globally public [`Scope`] for use with [`crate::Id`].
///
/// ## See Also
///
/// - [`crate::GlobalId`]
/// - [`crate::register_global`]
#[allow(clippy::module_name_repetitions)]
pub struct GlobalScope;
impl Scope for GlobalScope {}
