use core::{
    any::TypeId,
    fmt::{self, Debug, Display},
    hash::Hash,
    marker::PhantomData,
    sync::atomic::{AtomicUsize, Ordering},
};
use std::{collections::HashMap, lazy::SyncLazy, sync::Mutex};

use crate::{scope::Scope, GlobalScope};

/// The inner type used within [`Id`] representing its raw value.
///
/// This value may be accessed from an [`Id`] through the [`IdHandle`] type:
///
/// ```
/// # use dmv::{GlobalId, register, identity::InnerRepr};
/// let id: GlobalId = register();
/// let handle = id.handle();
/// let raw: &InnerRepr = handle.value();
/// ```
pub type InnerRepr = usize;

/// A value that serves as an identifier guaranteed to be unique within a given
/// [`Scope`] as represented by the generic parameter `S`. As such, this type
/// does not implement [`Copy`] or [`Clone`].
///
/// As no value of type `S` is stored or used within the struct itself, the
/// member `scope` is type [`PhantomData`], with the generic parameter being
/// `*const S`. As the documentation for [`PhantomData`] explains, this ensures
/// that the compiler understands that while this type takes a generic parameter
/// `S`, it does not *own* a value of type `S`.
pub struct Id<S: Scope> {
    id: InnerRepr,
    scope: PhantomData<*const S>,
}

impl<S: Scope> Debug for Id<S> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Debug::fmt(&self.id, f)
    }
}

impl<S: Scope> PartialEq for Id<S> {
    fn eq(&self, other: &Self) -> bool {
        self.id.eq(&other.id)
    }
}

impl<S: Scope> Eq for Id<S> {}

impl<S: Scope> PartialOrd for Id<S> {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        self.id.partial_cmp(&other.id)
    }
}

impl<S: Scope> Ord for Id<S> {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        self.id.cmp(&other.id)
    }
}

impl<S: Scope> Hash for Id<S> {
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.id.hash(state);
    }
}

impl<S: Scope> Id<S> {
    pub(crate) fn new() -> Self {
        static ID_MAP: SyncLazy<Mutex<HashMap<TypeId, AtomicUsize>>> = SyncLazy::new(|| {
            Mutex::new(
                core::iter::once((TypeId::of::<GlobalScope>(), AtomicUsize::new(0))).collect(),
            )
        });

        Self {
            id: {
                let mut map = ID_MAP.lock().unwrap();
                let scope_id = TypeId::of::<S>();

                map.entry(scope_id)
                    .or_insert_with(|| AtomicUsize::new(1))
                    .fetch_add(1, Ordering::Relaxed)
            },
            scope: PhantomData,
        }
    }

    /// Returns a handle to this identity's value.
    #[must_use]
    pub fn handle<'id, 'handle>(&'id self) -> IdHandle<'handle, S> where 'id: 'handle {
        (&self.id).into()
    }
}

impl<S: Scope> Display for Id<S> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Display::fmt(&self.id, f)
    }
}

/// Instantiation of an [`Id`] with the scope [`GlobalScope`].
///
/// ## See Also
///
/// - [`crate::register_global`]
pub type GlobalId = Id<GlobalScope>;

/// A handle to an [`Id`]'s value.
///
/// [`Id`] intentionally does not implement [`Copy`] or [`Clone`] to guarantee
/// that if you have an [`Id`] object it will be unique. [`IdHandle`] allows
/// the ability to share what the specific value [`Id`] is without breaking the
/// uniqueness of the original [`Id`].
///
/// As no value of type `S` is stored or used within the struct itself, the
/// member `scope` is type [`PhantomData`], with the generic parameter being
/// `*const S`. As the documentation for [`PhantomData`] explains, this ensures
/// that the compiler understands that while this type takes a generic parameter
/// `S`, it does not *own* a value of type `S`.
pub struct IdHandle<'handle, S: Scope>(
    &'handle InnerRepr,
    PhantomData<* const S>,
);

impl<'handle, S: Scope> IdHandle<'handle, S> {
    /// Returns the raw value of the [`Id`] this [`IdHandle`] refers too.
    pub fn value(&self) -> &'handle InnerRepr {
        self.0
    }
}

impl<'handle, S: Scope> From<&'handle InnerRepr> for IdHandle<'handle, S> {
    fn from(v: &'handle InnerRepr) -> Self {
        Self(v, PhantomData)
    }
}

impl<'handle, S: Scope> Copy for IdHandle<'handle, S> {}

impl<'handle, S: Scope> Clone for IdHandle<'handle, S> {
    fn clone(&self) -> Self {
        Self(self.0, PhantomData)
    }
}

impl<'handle, S: Scope> Debug for IdHandle<'handle, S> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Debug::fmt(&self.0, f)
    }
}

impl<'handle, S: Scope> PartialEq for IdHandle<'handle, S> {
    fn eq(&self, other: &Self) -> bool {
        self.0.eq(other.0)
    }
}

impl<'handle, S: Scope> Eq for IdHandle<'handle, S> {}

impl<'handle, S: Scope> PartialOrd for IdHandle<'handle, S> {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        self.0.partial_cmp(other.0)
    }
}

impl<'handle, S: Scope> Ord for IdHandle<'handle, S> {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        self.0.cmp(other.0)
    }
}

impl<'handle, S: Scope> Hash for IdHandle<'handle, S> {
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.0.hash(state);
    }
}
