/// Trait describing any type that can be used to restrict the scope of
/// uniqueness of a [`crate::Id`].
///
/// In the current state it is recommended to create a new scope by using the
/// [`scope!`] macro. Soon the [`scope!`] macro will be deprecated in favor of a
/// derive macro that can be called on any type to generate a scope and id type
/// specifically for that type.
pub trait Scope: std::any::Any {}

/// Simple macro filling out the boilerplate code of creating a new scope type
/// called `$ident` with optional visibility `$vis`, and implementing the
/// [`Scope`] trait and its dependent traits for the new type.
#[macro_export]
macro_rules! scope {
    ($vis:vis $ident:ident) => {
        #[derive(Hash)]
        $vis struct $ident;
        impl ::dmv::scope::Scope for $ident {}
    };
    ($($t:tt)*) => {
        compile_error!(concat!("This macro accepts input in the form `$vis $ident`. For example: `scope!{pub Foo}` or `scope!{Bar}`"))
    };
}

/// Globally public [`Scope`] for use with [`crate::Id`].
///
/// ## See Also
///
/// - [`crate::GlobalId`]
/// - [`crate::register_global`]
#[allow(clippy::module_name_repetitions)]
pub struct GlobalScope;
impl Scope for GlobalScope {}
