use core::{
    any::TypeId,
    fmt::{self, Debug, Display},
    hash::Hash,
    marker::PhantomData,
    sync::atomic::{AtomicUsize, Ordering},
};
use std::{collections::HashMap, lazy::SyncLazy, sync::Mutex};

use crate::{scope::Scope, GlobalScope};

/// A value that serves as an identifier guaranteed to be unique within a given
/// [`Scope`] as represented by the generic parameter `S`. As such, this type
/// does not implement [`Copy`] or [`Clone`].
///
/// As no value of type `S` is stored or used within the struct itself, the
/// member `scope` is type [`PhantomData`], with the generic parameter being
/// `*const S`. As the documentation for [`PhantomData`] explains, this ensures
/// that the compiler understands that while this type takes a generic parameter
/// `S`, it does not *own* a value of type `S`.
pub struct Id<S: Scope> {
    id: usize,
    scope: PhantomData<*const S>,
}

impl<S: Scope> Debug for Id<S> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Debug::fmt(&self.id, f)
    }
}

impl<S: Scope> PartialEq for Id<S> {
    fn eq(&self, other: &Self) -> bool {
        self.id.eq(&other.id)
    }
}

impl<S: Scope> Eq for Id<S> {}

impl<S: Scope> PartialOrd for Id<S> {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        self.id.partial_cmp(&other.id)
    }
}

impl<S: Scope> Ord for Id<S> {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        self.id.cmp(&other.id)
    }
}

impl<S: Scope> Hash for Id<S> {
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.id.hash(state);
    }
}

impl<S: Scope> Id<S> {
    pub(crate) fn new() -> Self {
        static ID_MAP: SyncLazy<Mutex<HashMap<TypeId, AtomicUsize>>> = SyncLazy::new(|| {
            Mutex::new(
                core::iter::once((TypeId::of::<GlobalScope>(), AtomicUsize::new(0))).collect(),
            )
        });

        Self {
            id: {
                let mut map = ID_MAP.lock().unwrap();
                let scope_id = TypeId::of::<S>();

                map.entry(scope_id)
                    .or_insert_with(|| AtomicUsize::new(1))
                    .fetch_add(1, Ordering::Relaxed)
            },
            scope: PhantomData,
        }
    }

    /// Returns this identity's value as a [`usize`].
    #[must_use]
    pub fn value(&self) -> usize {
        self.id
    }
}

impl<S: Scope> Display for Id<S> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        Display::fmt(&self.id, f)
    }
}

/// Instantiation of an [`Id`] with the scope [`GlobalScope`].
///
/// ## See Also
///
/// - [`crate::register_global`]
pub type GlobalId = Id<GlobalScope>;
