use std::collections::HashMap;

use anyhow::{Error, Result};
use petgraph::{algo::toposort, graphmap::DiGraphMap};

use crate::fql::{Csv, Fql};

#[derive(Debug)]
pub struct Model {
    pub id: String,
    pub fql: Fql,
    pub is_refed: bool,
}

#[derive(Debug)]
pub struct Project(pub Vec<Model>);

impl Project {
    pub fn try_new(fqls: HashMap<String, Fql>) -> Result<Self> {
        let mut edges = vec![];
        let mut refs = vec![];
        for (id, fql) in &fqls {
            for ref_id in &fql.refs {
                if !fqls.contains_key(ref_id) {
                    return Err(Error::msg(format!(
                        "Could not resolve ref {} in model {}",
                        ref_id, id
                    )));
                }
                edges.push((ref_id.as_str(), id.as_str()));
                refs.push(ref_id.to_owned());
            }
        }

        let order = get_order(edges)?;
        let mut ordered = vec![];
        let mut flex = vec![];
        for (id, fql) in fqls {
            let is_refed = refs.contains(&id);
            let model = Model { id, fql, is_refed };
            if order.contains(&model.id) {
                ordered.push(model);
            } else {
                flex.push(model);
            }
        }

        ordered.sort_by(|a, b| {
            let a_idx = order.iter().position(|x| x == &a.id).unwrap();
            let b_idx = order.iter().position(|x| x == &b.id).unwrap();
            a_idx.partial_cmp(&b_idx).unwrap()
        });
        ordered.append(&mut flex);
        Ok(Self(ordered))
    }

    pub fn sources(&self) -> Vec<&Csv> {
        self.0.iter().flat_map(|x| &x.fql.sources).collect()
    }
}

fn get_order(edges: Vec<(&str, &str)>) -> Result<Vec<String>> {
    let graph: DiGraphMap<_, ()> = DiGraphMap::from_edges(edges);
    let sorted = toposort(&graph, None);
    match sorted {
        Ok(sorted) => Ok(sorted.into_iter().map(|x| x.to_owned()).collect()),
        Err(cycle) => Err(Error::msg(format!("Found cycle: {:?}", cycle))),
    }
}
