mod fql;
mod io;
mod project;

use std::collections::HashMap;
use std::sync::Arc;

use anyhow::Result;
use arrow::datatypes::Schema;
use datafusion::{datasource::MemTable, prelude::*};

use fql::Fql;
use project::Project;

#[tokio::main]
async fn main() -> Result<()> {
    let cwd = std::env::current_dir()?;
    let files = io::parse_dir(cwd)?;
    let fqls = files
        .into_iter()
        .map(|(id, text)| -> Result<_> { Ok((id, Fql::try_new(text)?)) })
        .collect::<Result<HashMap<String, Fql>>>()?;
    let proj = Project::try_new(fqls)?;

    let mut ctx = ExecutionContext::new();
    for csv in proj.sources() {
        ctx.register_csv(&csv.id, &csv.path, csv.into())?;
    }

    for model in proj.0 {
        let fql = model.fql;
        let df = ctx.sql(&fql.sql)?;
        let results = df.collect().await?;

        for export in fql.exports {
            io::write_csv(export, &results)?;
        }

        if model.is_refed {
            let schema: Schema = df.schema().to_owned().into();
            let mem_table = MemTable::try_new(Arc::new(schema), vec![results])?;
            ctx.register_table(model.id.as_str(), Arc::new(mem_table))?;
        }
    }

    Ok(())
}
