use std::{
    collections::HashMap,
    fs::{read_dir, read_to_string, File},
    path::Path,
};

use anyhow::Result;
use arrow::{csv, record_batch::RecordBatch};

pub fn parse_dir(dir: impl AsRef<Path>) -> Result<HashMap<String, String>> {
    read_dir(dir)?
        .filter_map(std::result::Result::ok)
        .filter(|f| match f.path().extension() {
            Some(suffix) => suffix == "sql",
            None => false,
        })
        .map(|f| -> Result<_> {
            Ok((
                f.path()
                    .file_stem()
                    .ok_or(anyhow::Error::msg("Missing filename"))?
                    .to_string_lossy()
                    .into_owned(),
                read_to_string(f.path())?,
            ))
        })
        .collect()
}

pub fn write_csv(file: impl AsRef<Path>, data: &Vec<RecordBatch>) -> Result<()> {
    let file = File::create(file)?;
    let mut writer = csv::Writer::new(file);
    for batch in data {
        writer.write(batch)?;
    }
    Ok(())
}
