use anyhow::{Error, Result};
use datafusion::prelude::CsvReadOptions;
use lazy_static::lazy_static;
use rand::{distributions::Alphanumeric, thread_rng, Rng};
use regex::Regex;

const ID_LENGTH: usize = 7;
const DEFAULT_DELIM: char = ',';

fn generate_id() -> String {
    let mut rng = thread_rng();
    let mut id = String::with_capacity(ID_LENGTH);
    while id.len() < ID_LENGTH {
        let letter = char::from(rng.sample(Alphanumeric));
        if letter.is_alphabetic() {
            id.push(letter);
        }
    }
    id.to_lowercase()
}

#[derive(Debug)]
pub struct Csv {
    pub id: String,
    pub path: String,
    delim: char,
}

impl Csv {
    pub fn new(path: &str, delim: char) -> Self {
        Self {
            id: generate_id(),
            path: path.to_owned(),
            delim,
        }
    }
}

impl From<&Csv> for CsvReadOptions<'_> {
    fn from(csv: &Csv) -> Self {
        Self::new().delimiter(csv.delim as u8)
    }
}

#[derive(Debug)]
pub struct Fql {
    pub sources: Vec<Csv>,
    pub refs: Vec<String>,
    pub exports: Vec<String>,
    pub sql: String,
}

impl Fql {
    pub fn try_new(text: String) -> Result<Self> {
        lazy_static! {
            static ref EXPR_RE: Regex = Regex::new(r"\{(.*?)\}").unwrap();
        };

        let mut sql = text.clone();
        let mut num_matches = 0usize;
        let mut sources = vec![];
        let mut refs = vec![];
        let mut exports = vec![];
        for expr in EXPR_RE.find_iter(&text) {
            let raw_expr = expr.as_str();
            let expr_text = remove_padding(raw_expr).trim();
            let args: Vec<&str> = find_args(expr_text)
                .iter()
                .map(|x| remove_padding(x))
                .collect();
            match expr_text.to_lowercase() {
                x if x.starts_with("csv") => {
                    let num_args = args.len();
                    if num_args == 0 {
                        return Err(Error::msg("CSV source must include a path".to_string()));
                    }

                    let path = args[0];
                    let delim = if num_args >= 2 {
                        let delim_text = args[1];
                        if delim_text.len() != 1 {
                            return Err(Error::msg(
                                "Delimiter has to be single character".to_string(),
                            ));
                        }
                        delim_text.chars().next().unwrap()
                    } else {
                        DEFAULT_DELIM
                    };

                    let source = Csv::new(path, delim);
                    sql = sql.replace(raw_expr, &source.id);
                    sources.push(source);
                }
                x if x.starts_with("ref") => {
                    if args.len() == 0 {
                        return Err(Error::msg("Reference must have an identifier".to_string()));
                    }
                    let ref_id = args[0];
                    sql = sql.replace(raw_expr, ref_id);
                    refs.push(ref_id.to_owned());
                }
                x if x.starts_with("to_csv") => {
                    if args.len() == 0 {
                        return Err(Error::msg("CSV Export must include a path".to_string()));
                    }
                    sql = sql.replace(raw_expr, "");
                    exports.push(args[0].to_owned());
                }
                x => {
                    return Err(Error::msg(format!("Could not parse expression: {}", x)));
                }
            }
            num_matches += 1;
        }

        if num_matches == 0 {
            return Err(Error::msg("Must have at least one expression".to_string()));
        }
        Ok(Fql {
            sources,
            exports,
            refs,
            sql,
        })
    }
}

fn find_args(text: &str) -> Vec<&str> {
    lazy_static! {
        static ref ARG_RE: Regex = Regex::new(r#""(.*?)""#).unwrap();
    };

    ARG_RE.find_iter(text).map(|x| x.as_str()).collect()
}

fn remove_padding(text: &str) -> &str {
    &text[1..text.len() - 1]
}

#[cfg(test)]
mod tests {
    use super::*;

    fn kitchen_sink() -> Fql {
        let text = r#"
        { to_csv("costs_per_region.csv") }
        { to_csv("reports/costs_per_region.csv") }
        select *
        from { csv("costs.csv") }
        inner join { ref("sales") }
        left join { csv("taxes.csv", ";") }
        outer join { ref("salaries") }
        "#
        .to_string();
        Fql::try_new(text).unwrap()
    }

    #[test]
    fn sources() {
        let model = kitchen_sink();
        assert_eq!(model.sources.len(), 2);
        let csv_1 = &model.sources[0];
        assert_eq!(csv_1.delim, ',');
        assert_eq!(csv_1.path, "costs.csv");
        let csv_2 = &model.sources[1];
        assert_eq!(csv_2.delim, ';');
    }

    #[test]
    fn refs() {
        let model = kitchen_sink();
        assert_eq!(model.refs.len(), 2);
        assert_eq!(model.refs[0], "sales".to_string());
        assert!(model.sql.contains(&model.refs[1]));
    }

    #[test]
    fn exports() {
        let model = kitchen_sink();
        assert_eq!(model.exports.len(), 2);
        assert_eq!(model.exports[1], "reports/costs_per_region.csv".to_string());
        assert!(!model.sql.contains(&model.exports[0]));
    }

    #[test]
    #[should_panic]
    fn no_exprs() {
        let text = "Hello World".to_string();
        let _ = Fql::try_new(text).unwrap();
    }

    #[test]
    #[should_panic]
    fn lone_csv() {
        let text = "select * from { csv }".to_string();
        let _ = Fql::try_new(text).unwrap();
    }

    #[test]
    #[should_panic]
    fn long_delim() {
        let text = r#"select * from { csv("sales.csv", "xxx") }"#.to_string();
        let _ = Fql::try_new(text).unwrap();
    }

    #[test]
    #[should_panic]
    fn lone_ref() {
        let text = "select * from { ref }".to_string();
        let _ = Fql::try_new(text).unwrap();
    }

    #[test]
    #[should_panic]
    fn lone_export() {
        let text = "select * from { to_csv }".to_string();
        let _ = Fql::try_new(text).unwrap();
    }

    #[test]
    #[should_panic]
    fn unkown_expr() {
        let text = r#"select * from { excel("sales.xlsx") }"#.to_string();
        let _ = Fql::try_new(text).unwrap();
    }
}
