use petgraph::{algo::toposort, graphmap::DiGraphMap};
use rand::{distributions::Alphanumeric, thread_rng, Rng};

use crate::parser::{Func, SqlFile};

fn create_id(length: usize) -> String {
    let mut rng = thread_rng();
    let mut id = String::with_capacity(length);
    while id.len() < length {
        let letter = char::from(rng.sample(Alphanumeric));
        if letter.is_alphabetic() {
            id.push(letter);
        }
    }
    id.to_lowercase()
}

#[derive(Debug)]
pub struct Source {
    pub id: String,
    pub path: String,
}

impl Source {
    fn new(path: String) -> Self {
        Self {
            id: create_id(7),
            path,
        }
    }
}

#[derive(Debug)]
pub struct Model {
    pub id: String,
    pub sql: String,
    pub exports: Vec<String>,
    refs: Vec<String>,
}

#[derive(Debug)]
pub struct Pipeline {
    pub sources: Vec<Source>,
    pub models: Vec<Model>,
}

impl Pipeline {
    pub fn new(sql_files: Vec<SqlFile>) -> Self {
        let mut sources = Vec::new();
        let mut models = Vec::new();
        for sql_file in sql_files {
            let mut sql = sql_file.text;
            let mut exports = Vec::new();
            let mut refs = Vec::new();
            for expr in sql_file.exprs {
                match expr.func {
                    Func::Csv(path) => {
                        let source = Source::new(path);
                        sql = sql
                            .as_str()
                            .replace(expr.raw_text.as_str(), source.id.as_str());
                        sources.push(source);
                    }
                    Func::Ref(name) => {
                        sql = sql.as_str().replace(expr.raw_text.as_str(), name.as_str());
                        refs.push(name);
                    }
                    Func::ToCsv(path) => {
                        sql = sql.as_str().replace(expr.raw_text.as_str(), "");
                        exports.push(path);
                    }
                }
            }
            models.push(Model {
                id: sql_file.name,
                sql: sql.to_owned(),
                exports,
                refs,
            });
        }
        Self { sources, models }
    }

    pub fn exec_order(&self) -> Vec<&str> {
        let graph: DiGraphMap<_, ()> = self
            .models
            .iter()
            .flat_map(|model| {
                model
                    .refs
                    .iter()
                    .map(move |mref| (mref.as_str(), model.id.as_str()))
            })
            .collect();
        let mut ordered: Vec<&str> = toposort(&graph, None).unwrap();
        let mut flex: Vec<&str> = self
            .models
            .iter()
            .filter_map(|x| {
                if ordered.contains(&x.id.as_str()) {
                    None
                } else {
                    Some(x.id.as_str())
                }
            })
            .collect();
        ordered.append(&mut flex);
        ordered
    }
}
