use std::fs::read_to_string;

use glob::glob;

#[derive(Debug, PartialEq)]
pub enum Func {
    Csv(String),
    ToCsv(String),
    Ref(String),
}

impl Func {
    fn from_str(text: &str) -> Self {
        let len = text.len() - 2;
        if text.starts_with("csv(") {
            return Self::Csv(text[5..len].to_owned());
        }
        if text.starts_with("ref(") {
            return Self::Ref(text[5..len].to_owned());
        }
        if text.starts_with("to_csv(") {
            return Self::ToCsv(text[8..len].to_owned());
        }
        panic!("Could not parse string to function");
    }
}

#[derive(Debug, PartialEq)]
pub struct Expr {
    pub raw_text: String,
    pub func: Func,
}

fn parse(text: &str) -> Vec<Expr> {
    let starts: Vec<usize> = text.match_indices("{").into_iter().map(|x| x.0).collect();
    let ends: Vec<usize> = text.match_indices("}").into_iter().map(|x| x.0).collect();
    if starts.len() != ends.len() {
        panic!("Brackets do not match up.");
    }
    if starts.len() == 0 {
        panic!("Must have at least one expression");
    }
    starts
        .into_iter()
        .zip(ends.into_iter())
        .map(|(start, end)| Expr {
            raw_text: text[start..=end].to_owned(),
            func: Func::from_str(text[start + 1..end].trim()),
        })
        .collect()
}

#[derive(Debug)]
pub struct SqlFile {
    pub name: String,
    pub text: String,
    pub exprs: Vec<Expr>,
}

pub fn parse_dir() -> Vec<SqlFile> {
    glob("./*.sql")
        .unwrap()
        .map(|x| {
            let path = x.as_ref().unwrap().as_path();
            let text = read_to_string(path).unwrap();
            SqlFile {
                name: path.file_stem().unwrap().to_str().unwrap().to_owned(),
                text,
                exprs: parse(&read_to_string(path).unwrap()),
            }
        })
        .collect()
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_parser_full() {
        let text = r#"{ to_csv("profits.csv") }  select * from { csv("sales.csv") } inner join { ref("costs") }"#;
        assert_eq!(
            parse(text),
            vec![
                Expr {
                    raw_text: r#"{ to_csv("profits.csv") }"#.to_string(),
                    func: Func::ToCsv("profits.csv".to_string()),
                },
                Expr {
                    raw_text: r#"{ csv("sales.csv") }"#.to_string(),
                    func: Func::Csv("sales.csv".to_string()),
                },
                Expr {
                    raw_text: r#"{ ref("costs") }"#.to_string(),
                    func: Func::Ref("costs".to_string()),
                },
            ]
        );
    }

    #[test]
    #[should_panic]
    fn test_parser_unequal() {
        let text = r#"{ select * from { csv("sales.csv") } inner join { ref("costs") }"#;
        let _ = parse(text);
    }

    #[test]
    #[should_panic]
    fn test_parser_empty() {
        let text = "select * from sales inner join costs";
        let _ = parse(text);
    }
}
