/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * Workspace
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Errors reported by the workspace.

use dmntk_common::DmntkError;

/// Errors related with operating on workspaces.
#[derive(Error, Debug)]
pub enum WorkspaceError {
  #[error("artifact '{0}' with name '{1}' was not found")]
  ArtifactNotFound(String, String),
  #[error("'{0}' is not a valid name of invoked artifact")]
  InvalidInvokedArtifactName(String),
  #[error("definitions with name '{0}' already exist in workspace")]
  DefinitionsWithNameAlreadyExist(String),
  #[error("definitions with identifier '{0}' already exist in workspace")]
  DefinitionsWithIdAlreadyExist(String),
  #[error("definitions with tag '{0}' already exist in workspace")]
  DefinitionsWithTagAlreadyExist(String),
  #[error("can not lock workspace")]
  CanNotLockWorkspace,
  #[error("workspace for user '{0}' was not found")]
  WorkspaceNotFound(String),
}

impl From<WorkspaceError> for DmntkError {
  fn from(e: WorkspaceError) -> Self {
    DmntkError::new("WorkspaceError", &e.to_string())
  }
}

pub fn err_artifact_not_found(artifact: String, name: String) -> DmntkError {
  WorkspaceError::ArtifactNotFound(artifact, name).into()
}

pub fn err_invalid_invoked_artifact_name(name: String) -> DmntkError {
  WorkspaceError::InvalidInvokedArtifactName(name).into()
}

pub fn err_definitions_with_name_already_exists(name: String) -> DmntkError {
  WorkspaceError::DefinitionsWithNameAlreadyExist(name).into()
}

pub fn err_definitions_with_id_already_exists(name: String) -> DmntkError {
  WorkspaceError::DefinitionsWithIdAlreadyExist(name).into()
}

pub fn err_definitions_with_tag_already_exists(name: String) -> DmntkError {
  WorkspaceError::DefinitionsWithTagAlreadyExist(name).into()
}

pub fn err_can_not_lock_workspace() -> DmntkError {
  WorkspaceError::CanNotLockWorkspace.into()
}

pub fn err_workspace_not_found(s: &str) -> DmntkError {
  WorkspaceError::WorkspaceNotFound(s.to_string()).into()
}
