/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * Workspace
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! A `workspace` is a place where user's DMN™ models are stored.
//! Currently all models are stored in the same user's space (single-user mode).
//! In the future, the workspaces should be adjusted to store models
//! for multiple users (multi-user mode).

use self::errors::*;
use dmntk_common::{DmntkError, Result};
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::Value;
use dmntk_model::model::{Definitions, DmnElement, NamedElement};
use std::collections::HashMap;
use std::convert::TryFrom;
use std::sync::{Arc, Mutex};

lazy_static! {
  static ref WORKSPACES: Arc<Mutex<HashMap<String, Workspace>>> = Arc::new(Mutex::new(HashMap::new()));
}

#[derive(Default)]
pub struct Workspace {
  /// Collection of definitions deployed in workspace.
  definitions: Vec<Arc<Definitions>>,
  /// Mapping of definitions by definition's identifier.
  definitions_by_id: HashMap<String, Arc<Definitions>>,
  /// Mapping of definitions by definition's name.
  definitions_by_name: HashMap<String, Arc<Definitions>>,
  /// Mapping of definitions by definition's tag.
  definitions_by_tag: HashMap<String, Arc<Definitions>>,
}

impl Workspace {
  ///
  pub fn append_definitions(&mut self, tag: &str, definitions: Definitions) -> Result<(String, Option<String>, String)> {
    let definitions_name = definitions.name().to_string();
    let definitions_id = definitions.id().clone();
    let definitions_arc = Arc::new(definitions);
    if self.definitions_by_name.contains_key(&definitions_name) {
      return Err(definitions_with_name_already_exists(definitions_name));
    }
    self.definitions_by_name.insert(definitions_name.to_string(), Arc::clone(&definitions_arc));
    if let Some(id) = &definitions_id {
      if self.definitions_by_id.contains_key(id) {
        return Err(definitions_with_id_already_exists(id.clone()));
      }
      self.definitions_by_id.insert(id.clone(), Arc::clone(&definitions_arc));
    }
    if self.definitions_by_tag.contains_key(tag) {
      return Err(definitions_with_tag_already_exists(tag.to_string()));
    }
    self.definitions_by_tag.insert(tag.to_string(), Arc::clone(&definitions_arc));
    self.definitions.push(definitions_arc);
    Ok((definitions_name, definitions_id, tag.to_string()))
  }
  ///
  pub fn replace_definitions(&mut self, tag: &str, definitions: Definitions) -> Result<(String, Option<String>, String)> {
    if let Some(index) = self.definitions.iter().position(|d| d.name() == definitions.name()) {
      let removed = self.definitions.remove(index);
      self.definitions_by_name.remove(&removed.name().to_string());
      if let Some(id) = removed.id() {
        self.definitions_by_id.remove(id);
      }
      self.definitions_by_tag.remove(&tag.to_string());
    }
    self.append_definitions(tag, definitions)
  }
  ///
  pub fn get_by_tag(&self, tag: &str) -> Option<&Definitions> {
    if let Some(definitions) = self.definitions_by_tag.get(tag) {
      Some(definitions)
    } else {
      None
    }
  }
}

pub enum InvokedArtifact {
  Decision,
  BusinessKnowledgeModel,
  DecisionService,
}

impl TryFrom<&str> for InvokedArtifact {
  type Error = DmntkError;
  fn try_from(value: &str) -> Result<Self, Self::Error> {
    match value {
      "decision" => Ok(InvokedArtifact::Decision),
      "bkm" => Ok(InvokedArtifact::BusinessKnowledgeModel),
      "decisionService" => Ok(InvokedArtifact::DecisionService),
      _ => Err(invalid_invoked_artifact_name(value.to_string())),
    }
  }
}

/// Saves the definitions in workspace for specified user.
/// Currently all definitions are stored like for single user.
/// After successful deployment, the following tuple is returned `(name, id, tag)`.
pub fn deploy_definitions(tag: &str, definitions: Definitions, replace_existing: bool) -> Result<(String, Option<String>, String)> {
  let mut workspaces = WORKSPACES.lock().unwrap();
  if !workspaces.contains_key("1") {
    let workspace: Workspace = Default::default();
    workspaces.insert("1".into(), workspace);
  }
  let workspace = workspaces.get_mut("1").unwrap();
  if replace_existing {
    workspace.replace_definitions(tag, definitions)
  } else {
    workspace.append_definitions(tag, definitions)
  }
}

/// Evaluates an artifact.
pub fn evaluate_artifact(ctx: &FeelContext, model_tag: &str, artifact_type: &str, artifact_name: &str) -> Result<Value> {
  let workspaces = WORKSPACES.lock().unwrap();
  if let Some(workspace) = workspaces.get("1") {
    if let Some(definitions) = workspace.get_by_tag(model_tag) {
      return match InvokedArtifact::try_from(artifact_type)? {
        InvokedArtifact::Decision => dmntk_evaluator::evaluate_decision_by_name(definitions, artifact_name, ctx),
        InvokedArtifact::BusinessKnowledgeModel => dmntk_evaluator::evaluate_business_knowledge_model_by_name(definitions, artifact_name, ctx),
        InvokedArtifact::DecisionService => dmntk_evaluator::eval_decision_service_by_name(definitions, artifact_name, ctx),
      };
    }
  }
  Err(artifact_not_found(artifact_type.to_owned(), artifact_name.to_owned()))
}

/// Errors reported by the workspace.
mod errors {
  use dmntk_common::DmntkError;

  /// Errors related with operating on workspaces.
  #[derive(Debug, PartialEq)]
  pub enum WorkspaceError {
    ArtifactNotFound(String, String),
    InvalidInvokedArtifactName(String),
    DefinitionsWithNameAlreadyExist(String),
    DefinitionsWithIdAlreadyExist(String),
    DefinitionsWithTagAlreadyExist(String),
  }

  impl From<WorkspaceError> for DmntkError {
    fn from(e: WorkspaceError) -> Self {
      DmntkError::new("WorkspaceError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for WorkspaceError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        WorkspaceError::ArtifactNotFound(artifact, name) => {
          write!(f, "artifact '{}' with name '{}' was not found", artifact, name)
        }
        WorkspaceError::InvalidInvokedArtifactName(name) => {
          write!(f, "'{}' is not a valid name of invoked artifact", name)
        }
        WorkspaceError::DefinitionsWithNameAlreadyExist(name) => {
          write!(f, "definitions with name '{}' already exist in workspace", name)
        }
        WorkspaceError::DefinitionsWithIdAlreadyExist(name) => {
          write!(f, "definitions with identifier '{}' already exist in workspace", name)
        }
        WorkspaceError::DefinitionsWithTagAlreadyExist(name) => {
          write!(f, "definitions with tag '{}' already exist in workspace", name)
        }
      }
    }
  }

  pub fn artifact_not_found(artifact: String, name: String) -> DmntkError {
    WorkspaceError::ArtifactNotFound(artifact, name).into()
  }

  pub fn invalid_invoked_artifact_name(name: String) -> DmntkError {
    WorkspaceError::InvalidInvokedArtifactName(name).into()
  }

  pub fn definitions_with_name_already_exists(name: String) -> DmntkError {
    WorkspaceError::DefinitionsWithNameAlreadyExist(name).into()
  }

  pub fn definitions_with_id_already_exists(name: String) -> DmntkError {
    WorkspaceError::DefinitionsWithIdAlreadyExist(name).into()
  }

  pub fn definitions_with_tag_already_exists(name: String) -> DmntkError {
    WorkspaceError::DefinitionsWithTagAlreadyExist(name).into()
  }
}
