/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model and parser
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Parser for loading a model from the XML file containing DMN interchange format.

use {
  self::{errors::*, xml_utils::*},
  super::*,
  dmntk_common::Result,
  roxmltree::Node,
};

const NODE_ALLOWED_ANSWERS: &str = "allowedAnswers";
const NODE_ALLOWED_VALUES: &str = "allowedValues";
const NODE_BINDING: &str = "binding";
const NODE_BUSINESS_KNOWLEDGE_MODEL: &str = "businessKnowledgeModel";
const NODE_COLUMN: &str = "column";
const NODE_CONTEXT: &str = "context";
const NODE_CONTEXT_ENTRY: &str = "contextEntry";
const NODE_DEFAULT_OUTPUT_ENTRY: &str = "defaultOutputEntry";
const NODE_DEFINITIONS: &str = "definitions";
const NODE_DECISION: &str = "decision";
const NODE_DECISION_TABLE: &str = "decisionTable";
const NODE_DECISION_SERVICE: &str = "decisionService";
const NODE_DMNDI: &str = "DMNDI";
const NODE_OUTPUT_DECISION: &str = "outputDecision";
const NODE_DESCRIPTION: &str = "description";
const NODE_ENCAPSULATED_LOGIC: &str = "encapsulatedLogic";
const NODE_FUNCTION_DEFINITION: &str = "functionDefinition";
const NODE_FORMAL_PARAMETER: &str = "formalParameter";
const NODE_FUNCTION_ITEM: &str = "functionItem";
const NODE_INFORMATION_REQUIREMENT: &str = "informationRequirement";
const NODE_INPUT_DATA: &str = "inputData";
const NODE_INPUT: &str = "input";
const NODE_INPUT_ENTRY: &str = "inputEntry";
const NODE_INPUT_EXPRESSION: &str = "inputExpression";
const NODE_INPUT_VALUES: &str = "inputValues";
const NODE_INVOCATION: &str = "invocation";
const NODE_ITEM_DEFINITION: &str = "itemDefinition";
const NODE_ITEM_COMPONENT: &str = "itemComponent";
const NODE_KNOWLEDGE_REQUIREMENT: &str = "knowledgeRequirement";
const NODE_LITERAL_EXPRESSION: &str = "literalExpression";
const NODE_OUTPUT: &str = "output";
const NODE_OUTPUT_ENTRY: &str = "outputEntry";
const NODE_OUTPUT_VALUES: &str = "outputValues";
const NODE_PARAMETER: &str = "parameter";
const NODE_QUESTION: &str = "question";
const NODE_RELATION: &str = "relation";
const NODE_REQUIRED_DECISION: &str = "requiredDecision";
const NODE_REQUIRED_KNOWLEDGE: &str = "requiredKnowledge";
const NODE_REQUIRED_INPUT: &str = "requiredInput";
const NODE_ROW: &str = "row";
const NODE_RULE: &str = "rule";
const NODE_TEXT: &str = "text";
const NODE_TYPE_REF: &str = "typeRef";
const NODE_VARIABLE: &str = "variable";

const ATTR_EXPORTER: &str = "exporter";
const ATTR_EXPORTER_VERSION: &str = "exporter_version";
const ATTR_EXPRESSION_LANGUAGE: &str = "expressionLanguage";
const ATTR_HIT_POLICY: &str = "hitPolicy";
const ATTR_AGGREGATION: &str = "aggregation";
const ATTR_PREFERRED_ORIENTATION: &str = "preferredOrientation";
const ATTR_HREF: &str = "href";
const ATTR_ID: &str = "id";
const ATTR_IS_COLLECTION: &str = "isCollection";
const ATTR_KIND: &str = "kind";
const ATTR_LABEL: &str = "label";
const ATTR_NAME: &str = "name";
const ATTR_NAMESPACE: &str = "namespace";
const ATTR_OUTPUT_LABEL: &str = "outputLabel";
const ATTR_OUTPUT_TYPE_REF: &str = "outputTypeRef";
const ATTR_TYPE_LANGUAGE: &str = "typeLanguage";
const ATTR_TYPE_REF: &str = "typeRef";

/// Parses the XML document containing definitions.
pub fn parse(xml: &str, source: &str) -> Result<Definitions> {
  ModelParser::default().parse(xml, source)
}

struct ModelParser {
  /// Collection of parsed item definitions.
  item_definitions: Vec<ItemDefinition>,
}

impl Default for ModelParser {
  fn default() -> Self {
    Self { item_definitions: vec![] }
  }
}

impl ModelParser {
  /// Parses the XML document containing definitions.
  pub fn parse(&mut self, xml: &str, source: &str) -> Result<Definitions> {
    match roxmltree::Document::parse(&xml) {
      Ok(document) => {
        let definitions_node = document.root_element();
        if definitions_node.tag_name().name() != NODE_DEFINITIONS {
          return Err(xml_unexpected_node(NODE_DEFINITIONS, definitions_node.tag_name().name().to_string()));
        }
        self.parse_definitions(&definitions_node, source)
      }
      Err(reason) => Err(xml_parsing_model_failed(format!("{}", reason))),
    }
  }

  /// Parses model definitions.
  ///
  /// # Arguments
  ///
  /// - mode   - definitions node.
  /// - source - name of the model source file.
  ///
  fn parse_definitions(&mut self, node: &Node, source: &str) -> Result<Definitions> {
    let name = self.required_name(node)?;
    let item_definitions = self.parse_item_definitions(node, NODE_ITEM_DEFINITION)?;
    self.item_definitions.append(item_definitions.clone().as_mut());
    let definitions = Definitions {
      name,
      id: optional_attribute(node, ATTR_ID),
      description: optional_child_optional_content(node, NODE_DESCRIPTION),
      label: optional_attribute(node, ATTR_LABEL),
      extension_elements: self.parse_extension_elements(node),
      extension_attributes: self.parse_extension_attributes(node),
      namespace: required_attribute(node, ATTR_NAMESPACE)?,
      expression_language: optional_attribute(node, ATTR_EXPRESSION_LANGUAGE),
      type_language: optional_attribute(node, ATTR_TYPE_LANGUAGE),
      exporter: optional_attribute(node, ATTR_EXPORTER),
      exporter_version: optional_attribute(node, ATTR_EXPORTER_VERSION),
      item_definitions,
      drg_elements: self.parse_drg_elements(node)?,
      business_context_elements: self.parse_business_context_elements(node)?,
      source: source.to_string(),
    };
    self.parse_dmndi(node); //FIXME move up
    Ok(definitions)
  }

  fn parse_item_definitions(&mut self, node: &Node, child_name: &str) -> Result<Vec<ItemDefinition>> {
    let mut items = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == child_name) {
      let name = self.required_name(child_node)?;
      let type_ref = optional_child_required_content(child_node, NODE_TYPE_REF)?;
      let type_language = optional_attribute(child_node, ATTR_TYPE_LANGUAGE);
      let allowed_values = self.parse_unary_tests(child_node, NODE_ALLOWED_VALUES)?;
      let item_components_definitions = self.parse_item_definitions(child_node, NODE_ITEM_COMPONENT)?;
      let item_definition = ItemDefinition {
        name,
        id: optional_attribute(child_node, ATTR_ID),
        description: optional_child_optional_content(child_node, NODE_DESCRIPTION),
        label: optional_attribute(child_node, ATTR_LABEL),
        extension_elements: self.parse_extension_elements(child_node),
        extension_attributes: self.parse_extension_attributes(child_node),
        type_ref,
        type_language,
        allowed_values,
        item_components: item_components_definitions,
        is_collection: self.parse_boolean_attribute(child_node, ATTR_IS_COLLECTION, false),
        function_item: self.parse_function_item(child_node),
      };
      items.push(item_definition);
    }
    Ok(items)
  }

  ///
  fn parse_function_item(&self, node: &Node) -> Option<FunctionItem> {
    node
      .children()
      .find(|n| n.tag_name().name() == NODE_FUNCTION_ITEM)
      .as_ref()
      .map(|n| FunctionItem {
        output_type_ref: optional_attribute(n, ATTR_OUTPUT_TYPE_REF),
        parameters: vec![],
      })
  }

  fn parse_unary_tests(&self, node: &Node, child_name: &str) -> Result<Option<UnaryTests>> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      Ok(Some(UnaryTests {
        text: optional_child_required_content(&child_node, NODE_TEXT)?,
        expression_language: optional_attribute(&child_node, ATTR_EXPRESSION_LANGUAGE),
      }))
    } else {
      Ok(None)
    }
  }

  fn parse_drg_elements(&self, definitions_node: &Node) -> Result<Vec<DrgElement>> {
    let mut drg_elements = vec![];
    drg_elements.append(&mut self.parse_input_data(&definitions_node)?);
    drg_elements.append(&mut self.parse_decisions(&definitions_node)?);
    drg_elements.append(&mut self.parse_business_knowledge_models(&definitions_node)?);
    drg_elements.append(&mut self.parse_decision_services(&definitions_node)?);
    Ok(drg_elements)
  }

  fn parse_input_data(&self, node: &Node) -> Result<Vec<DrgElement>> {
    let mut input_data_items = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_INPUT_DATA) {
      let name = self.required_name(child_node)?;
      let input_data = InputData {
        id: optional_attribute(child_node, ATTR_ID),
        description: optional_child_optional_content(child_node, NODE_DESCRIPTION),
        label: optional_attribute(child_node, ATTR_LABEL),
        extension_elements: self.parse_extension_elements(child_node),
        extension_attributes: self.parse_extension_attributes(child_node),
        name,
        variable: self.parse_information_item_child(child_node, NODE_VARIABLE)?,
      };
      input_data_items.push(DrgElement::InputData(input_data));
    }
    Ok(input_data_items)
  }

  fn parse_decisions(&self, definitions_node: &Node) -> Result<Vec<DrgElement>> {
    let mut decision_items = vec![];
    for child_node in definitions_node.children().filter(|n| n.tag_name().name() == NODE_DECISION) {
      let name = self.required_name(&child_node)?;
      let decision = Decision {
        name,
        id: optional_attribute(&child_node, ATTR_ID),
        description: optional_child_optional_content(&child_node, NODE_DESCRIPTION),
        label: optional_attribute(&child_node, ATTR_LABEL),
        extension_elements: None,
        extension_attributes: vec![],
        question: optional_child_optional_content(&child_node, NODE_QUESTION),
        allowed_answers: optional_child_optional_content(&child_node, NODE_ALLOWED_ANSWERS),
        variable: self.parse_information_item_child(&child_node, NODE_VARIABLE)?,
        decision_logic: self.parse_optional_expression_instance(&child_node)?,
        information_requirements: self.parse_information_requirements(&child_node, NODE_INFORMATION_REQUIREMENT)?,
        knowledge_requirements: self.parse_knowledge_requirements(&child_node, NODE_KNOWLEDGE_REQUIREMENT)?,
      };
      decision_items.push(DrgElement::Decision(decision));
    }
    Ok(decision_items)
  }

  fn parse_business_knowledge_models(&self, definitions_node: &Node) -> Result<Vec<DrgElement>> {
    let mut parsed_items = vec![];
    for ref child_node in definitions_node.children().filter(|n| n.tag_name().name() == NODE_BUSINESS_KNOWLEDGE_MODEL) {
      let name = self.required_name(&child_node)?;
      let business_knowledge_model = BusinessKnowledgeModel {
        name,
        id: optional_attribute(child_node, ATTR_ID),
        description: optional_child_optional_content(child_node, NODE_DESCRIPTION),
        label: optional_attribute(child_node, ATTR_LABEL),
        extension_elements: None,
        extension_attributes: vec![],
        variable: self.parse_information_item_child(child_node, NODE_VARIABLE)?,
        encapsulated_logic: self.parse_function_definition_child(child_node, NODE_ENCAPSULATED_LOGIC)?,
        knowledge_requirements: self.parse_knowledge_requirements(&child_node, NODE_KNOWLEDGE_REQUIREMENT)?,
        authority_requirements: vec![],
      };
      parsed_items.push(DrgElement::BusinessKnowledgeModel(business_knowledge_model));
    }
    Ok(parsed_items)
  }

  fn parse_decision_services(&self, definitions_node: &Node) -> Result<Vec<DrgElement>> {
    let mut drg_elements = vec![];
    for ref decision_service_node in definitions_node.children().filter(|n| n.tag_name().name() == NODE_DECISION_SERVICE) {
      let name = self.required_name(decision_service_node)?;
      let decision_service = DecisionService {
        name,
        id: optional_attribute(decision_service_node, ATTR_ID),
        description: optional_child_optional_content(decision_service_node, NODE_DESCRIPTION),
        label: optional_attribute(decision_service_node, ATTR_LABEL),
        extension_elements: None,
        extension_attributes: vec![],
        variable: self.parse_information_item_child(decision_service_node, NODE_VARIABLE)?,
        output_decisions: self.required_hrefs_in_child_nodes(decision_service_node, NODE_OUTPUT_DECISION)?,
        encapsulated_decisions: self.required_hrefs_in_child_nodes(decision_service_node, NODE_OUTPUT_DECISION)?,
        input_decisions: self.required_hrefs_in_child_nodes(decision_service_node, NODE_OUTPUT_DECISION)?,
        input_data: self.required_hrefs_in_child_nodes(decision_service_node, NODE_OUTPUT_DECISION)?,
      };
      drg_elements.push(DrgElement::DecisionService(decision_service));
    }
    Ok(drg_elements)
  }

  fn required_hrefs_in_child_nodes(&self, node: &Node, child_name: &str) -> Result<Vec<HRef>> {
    let mut hrefs = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == child_name) {
      let text = required_attribute(child_node, ATTR_HREF)?;
      hrefs.push(HRef::from(text));
    }
    Ok(hrefs)
  }

  fn parse_function_definition_child(&self, node: &Node, child_name: &str) -> Result<Option<FunctionDefinition>> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      Ok(Some(self.parse_function_definition(&child_node)?))
    } else {
      Ok(None)
    }
  }

  fn parse_optional_function_definition(&self, node: &Node) -> Result<Option<FunctionDefinition>> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == NODE_FUNCTION_DEFINITION) {
      Ok(Some(self.parse_function_definition(&child_node)?))
    } else {
      Ok(None)
    }
  }

  fn parse_function_definition(&self, node: &Node) -> Result<FunctionDefinition> {
    Ok(FunctionDefinition {
      id: optional_attribute(node, ATTR_ID),
      description: optional_child_optional_content(node, NODE_DESCRIPTION),
      label: optional_attribute(node, ATTR_LABEL),
      extension_elements: self.parse_extension_elements(node),
      extension_attributes: self.parse_extension_attributes(node),
      type_ref: optional_attribute(node, ATTR_TYPE_REF),
      formal_parameters: self.parse_information_items_child(node, NODE_FORMAL_PARAMETER)?,
      body: self.parse_optional_expression_instance(node)?,
      kind: self.parse_function_kind(node)?,
    })
  }

  fn parse_function_kind(&self, node: &Node) -> Result<FunctionKind> {
    if let Some(function_kind_text) = optional_attribute(node, ATTR_KIND) {
      match function_kind_text.trim() {
        "FEEL" => Ok(FunctionKind::Feel),
        "Java" => Ok(FunctionKind::Java),
        "PMML" => Ok(FunctionKind::Pmml),
        other => Err(invalid_function_kind(other)),
      }
    } else {
      Ok(FunctionKind::Feel)
    }
  }

  #[allow(clippy::unnecessary_wraps)]
  fn parse_business_context_elements(&self, _node: &Node) -> Result<Vec<BusinessContextElementInstance>> {
    Ok(vec![])
  }

  fn parse_information_item_child(&self, node: &Node, child_name: &str) -> Result<InformationItem> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      self.parse_information_item(&child_node)
    } else {
      Err(xml_expected_mandatory_child_node(node_name_pos(node), child_name.to_string()))
    }
  }

  fn parse_optional_information_item_child(&self, node: &Node, child_name: &str) -> Result<Option<InformationItem>> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      return Ok(Some(self.parse_information_item(&child_node)?));
    }
    Ok(None)
  }

  fn parse_information_items_child(&self, node: &Node, child_name: &str) -> Result<Vec<InformationItem>> {
    let mut information_items = vec![];
    for child_node in node.children().filter(|n| n.tag_name().name() == child_name) {
      information_items.push(self.parse_information_item(&child_node)?);
    }
    Ok(information_items)
  }

  fn parse_information_item(&self, node: &Node) -> Result<InformationItem> {
    let type_ref = optional_attribute(node, ATTR_TYPE_REF);
    let name = self.required_name(&node)?;
    Ok(InformationItem {
      id: optional_attribute(node, ATTR_ID),
      description: optional_child_optional_content(node, NODE_DESCRIPTION),
      label: optional_attribute(node, ATTR_LABEL),
      extension_elements: self.parse_extension_elements(node),
      extension_attributes: self.parse_extension_attributes(node),
      name,
      value_expression: self.parse_optional_expression_instance(node)?,
      type_ref: type_ref.unwrap_or_else(|| "".to_string()),
    })
  }

  fn parse_information_requirements(&self, node: &Node, child_name: &str) -> Result<Vec<InformationRequirement>> {
    let mut information_requirement_items = vec![];
    for child_node in node.children().filter(|n| n.tag_name().name() == child_name) {
      information_requirement_items.push(self.parse_information_requirement(&child_node)?);
    }
    Ok(information_requirement_items)
  }

  fn parse_information_requirement(&self, node: &Node) -> Result<InformationRequirement> {
    Ok(InformationRequirement {
      id: optional_attribute(node, ATTR_ID),
      description: optional_child_optional_content(&node, NODE_DESCRIPTION),
      label: optional_attribute(node, ATTR_LABEL),
      extension_elements: self.parse_extension_elements(node),
      extension_attributes: self.parse_extension_attributes(node),
      required_decision: optional_child_required_href(node, NODE_REQUIRED_DECISION)?,
      required_input: optional_child_required_href(node, NODE_REQUIRED_INPUT)?,
    })
  }

  fn parse_knowledge_requirements(&self, node: &Node, child_name: &str) -> Result<Vec<KnowledgeRequirement>> {
    let mut knowledge_requirement_items = vec![];
    for child_node in node.children().filter(|n| n.tag_name().name() == child_name) {
      knowledge_requirement_items.push(self.parse_knowledge_requirement(&child_node)?);
    }
    Ok(knowledge_requirement_items)
  }

  fn parse_knowledge_requirement(&self, node: &Node) -> Result<KnowledgeRequirement> {
    Ok(KnowledgeRequirement {
      id: optional_attribute(node, ATTR_ID),
      description: optional_child_optional_content(&node, NODE_DESCRIPTION),
      label: optional_attribute(node, ATTR_LABEL),
      extension_elements: self.parse_extension_elements(node),
      extension_attributes: self.parse_extension_attributes(node),
      required_knowledge: optional_child_required_href(node, NODE_REQUIRED_KNOWLEDGE)?,
    })
  }

  fn parse_required_expression_instance(&self, node: &Node) -> Result<ExpressionInstance> {
    self
      .parse_optional_expression_instance(node)?
      .ok_or_else(required_expression_instance_is_missing)
  }

  fn parse_optional_expression_instance(&self, node: &Node) -> Result<Option<ExpressionInstance>> {
    if let Some(context) = self.parse_optional_context(node)? {
      return Ok(Some(ExpressionInstance::Context(context)));
    }
    if let Some(decision_table) = self.parse_decision_table(node)? {
      return Ok(Some(ExpressionInstance::DecisionTable(decision_table)));
    }
    if let Some(function_definition) = self.parse_optional_function_definition(node)? {
      return Ok(Some(ExpressionInstance::FunctionDefinition(Box::new(function_definition))));
    }
    if let Some(invocation) = self.parse_optional_invocation(node)? {
      return Ok(Some(ExpressionInstance::Invocation(Box::new(invocation))));
    }
    if let Some(literal_expression) = self.parse_optional_literal_expression(node) {
      return Ok(Some(ExpressionInstance::LiteralExpression(literal_expression)));
    }
    //TODO add unary tests
    if let Some(relation) = self.parse_optional_relation(node)? {
      return Ok(Some(ExpressionInstance::Relation(relation)));
    }

    Ok(None)
  }

  fn parse_decision_table(&self, node: &Node) -> Result<Option<DecisionTable>> {
    if let Some(ref child_node) = node.children().find(|n| n.tag_name().name() == NODE_DECISION_TABLE) {
      return Ok(Some(DecisionTable {
        information_item_name: None,
        input_clauses: self.parse_decision_table_inputs(child_node)?,
        output_clauses: self.parse_decision_table_outputs(child_node)?,
        annotations: vec![],
        rules: self.parse_decision_table_rules(child_node)?,
        hit_policy: self.parse_hit_policy_attribute(child_node)?,
        aggregation: None,
        preferred_orientation: self.parse_preferred_orientation_attribute(child_node)?,
        output_label: optional_attribute(child_node, ATTR_OUTPUT_LABEL),
      }));
    }
    Ok(None)
  }

  fn parse_decision_table_inputs(&self, node: &Node) -> Result<Vec<InputClause>> {
    let mut input_clauses = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_INPUT) {
      input_clauses.push(self.parse_decision_table_input(child_node)?);
    }
    Ok(input_clauses)
  }

  fn parse_decision_table_input(&self, node: &Node) -> Result<InputClause> {
    let input_expression = if let Ok(ref child_node) = required_child(node, NODE_INPUT_EXPRESSION) {
      required_child_required_content(child_node, NODE_TEXT)?
    } else {
      return Err(required_input_expression_is_missing());
    };
    let input_values = if let Some(ref child_node) = optional_child(node, NODE_INPUT_VALUES) {
      optional_child_required_content(child_node, NODE_TEXT)?
    } else {
      None
    };
    Ok(InputClause {
      input_expression,
      input_values,
    })
  }

  fn parse_decision_table_outputs(&self, node: &Node) -> Result<Vec<OutputClause>> {
    let mut output_clauses = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_OUTPUT) {
      output_clauses.push(self.parse_decision_table_output(child_node)?);
    }
    Ok(output_clauses)
  }

  fn parse_decision_table_output(&self, node: &Node) -> Result<OutputClause> {
    let output_values = if let Some(ref child_node) = optional_child(node, NODE_OUTPUT_VALUES) {
      optional_child_required_content(child_node, NODE_TEXT)?
    } else {
      None
    };
    let default_output_entry = if let Some(ref child_node) = optional_child(node, NODE_DEFAULT_OUTPUT_ENTRY) {
      optional_child_required_content(child_node, NODE_TEXT)?
    } else {
      None
    };
    Ok(OutputClause {
      type_ref: optional_attribute(node, ATTR_TYPE_REF),
      name: optional_attribute(node, ATTR_NAME),
      output_values,
      default_output_entry,
    })
  }

  fn parse_decision_table_rules(&self, node: &Node) -> Result<Vec<DecisionRule>> {
    let mut rules = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_RULE) {
      rules.push(self.parse_decision_table_rule(child_node)?);
    }
    Ok(rules)
  }

  fn parse_decision_table_rule(&self, node: &Node) -> Result<DecisionRule> {
    Ok(DecisionRule {
      input_entries: self.parse_decision_table_input_entries(node)?,
      output_entries: self.parse_decision_table_output_entries(node)?,
      annotation_entries: vec![],
    })
  }

  fn parse_decision_table_input_entries(&self, node: &Node) -> Result<Vec<InputEntry>> {
    let mut input_entries = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_INPUT_ENTRY) {
      input_entries.push(self.parse_decision_table_input_entry(child_node)?);
    }
    Ok(input_entries)
  }

  fn parse_decision_table_input_entry(&self, node: &Node) -> Result<InputEntry> {
    Ok(InputEntry {
      text: required_child_required_content(node, NODE_TEXT)?,
    })
  }

  fn parse_decision_table_output_entries(&self, node: &Node) -> Result<Vec<OutputEntry>> {
    let mut output_entries = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_OUTPUT_ENTRY) {
      output_entries.push(self.parse_decision_table_output_entry(child_node)?);
    }
    Ok(output_entries)
  }

  fn parse_decision_table_output_entry(&self, node: &Node) -> Result<OutputEntry> {
    Ok(OutputEntry {
      text: required_child_required_content(node, NODE_TEXT)?,
    })
  }

  fn parse_optional_context(&self, node: &Node) -> Result<Option<Context>> {
    if let Some(ref child_node) = node.children().find(|n| n.tag_name().name() == NODE_CONTEXT) {
      return Ok(Some(Context {
        context_entries: self.parse_context_entries(child_node)?,
      }));
    }
    Ok(None)
  }

  fn parse_context_entries(&self, node: &Node) -> Result<Vec<ContextEntry>> {
    let mut context_entries = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_CONTEXT_ENTRY) {
      context_entries.push(ContextEntry {
        variable: self.parse_optional_information_item_child(&child_node, NODE_VARIABLE)?,
        value: self.parse_required_expression_instance(child_node)?,
      });
    }
    Ok(context_entries)
  }

  fn parse_optional_invocation(&self, node: &Node) -> Result<Option<Invocation>> {
    if let Some(ref child_node) = node.children().find(|n| n.tag_name().name() == NODE_INVOCATION) {
      return Ok(Some(Invocation {
        called_function: self.parse_required_expression_instance(child_node)?,
        bindings: self.parse_bindings(child_node)?,
      }));
    }
    Ok(None)
  }

  fn parse_bindings(&self, node: &Node) -> Result<Vec<Binding>> {
    let mut bindings = vec![];
    for ref child_node in node.children().filter(|n| n.tag_name().name() == NODE_BINDING) {
      bindings.push(Binding {
        parameter: self.parse_information_item_child(child_node, NODE_PARAMETER)?,
        binding_formula: self.parse_optional_expression_instance(child_node)?,
      });
    }
    Ok(bindings)
  }

  /// Searches for the first node named 'literalExpression' among children of the specified `node`.
  /// When such node is found, then parses literal expression and returns it; otherwise returns [None].
  fn parse_optional_literal_expression(&self, node: &Node) -> Option<LiteralExpression> {
    if let Some(ref child_node) = node.children().find(|n| n.tag_name().name() == NODE_LITERAL_EXPRESSION) {
      return Some(self.parse_literal_expression(child_node));
    }
    None
  }

  /// Parses [LiteralExpression] directly from the specified node.
  /// The `literal_expression_node` must be a node named `literalExpression`.
  fn parse_literal_expression(&self, literal_expression_node: &Node) -> LiteralExpression {
    LiteralExpression {
      id: optional_attribute(literal_expression_node, ATTR_ID),
      description: optional_child_optional_content(literal_expression_node, NODE_DESCRIPTION),
      label: optional_attribute(literal_expression_node, ATTR_LABEL),
      extension_elements: self.parse_extension_elements(literal_expression_node),
      extension_attributes: self.parse_extension_attributes(literal_expression_node),
      type_ref: optional_attribute(literal_expression_node, ATTR_TYPE_REF),
      text: optional_child_optional_content(literal_expression_node, NODE_TEXT),
      expression_language: optional_attribute(literal_expression_node, ATTR_EXPRESSION_LANGUAGE),
      imported_values: None,
    }
  }

  fn parse_optional_relation(&self, node: &Node) -> Result<Option<Relation>> {
    if let Some(ref relation_node) = node.children().find(|n| n.tag_name().name() == NODE_RELATION) {
      let mut columns = vec![];
      for ref column_node in relation_node.children().filter(|n| n.tag_name().name() == NODE_COLUMN) {
        columns.push(self.parse_information_item(column_node)?);
      }
      let mut rows = vec![];
      for ref row_node in relation_node.children().filter(|n| n.tag_name().name() == NODE_ROW) {
        let mut elements = vec![];
        for ref expression_instance_node in row_node.children() {
          if expression_instance_node.tag_name().name() == NODE_LITERAL_EXPRESSION {
            let literal_expression = self.parse_literal_expression(expression_instance_node);
            elements.push(ExpressionInstance::LiteralExpression(literal_expression));
          }
        }
        if elements.len() != columns.len() {
          return Err(number_of_elements_in_row_differs_from_number_of_columns());
        }
        rows.push(List {
          id: optional_attribute(row_node, ATTR_ID),
          description: optional_child_optional_content(row_node, NODE_DESCRIPTION),
          label: optional_attribute(row_node, ATTR_LABEL),
          extension_elements: self.parse_extension_elements(row_node),
          extension_attributes: self.parse_extension_attributes(row_node),
          type_ref: optional_attribute(row_node, ATTR_TYPE_REF),
          elements,
        });
      }
      return Ok(Some(Relation {
        id: optional_attribute(relation_node, ATTR_ID),
        description: optional_child_optional_content(relation_node, NODE_DESCRIPTION),
        label: optional_attribute(relation_node, ATTR_LABEL),
        extension_elements: self.parse_extension_elements(relation_node),
        extension_attributes: self.parse_extension_attributes(relation_node),
        type_ref: optional_attribute(relation_node, ATTR_TYPE_REF),
        rows,
        columns,
      }));
    }
    Ok(None)
  }

  /// Parses extension elements.
  /// Currently extension elements are ignored and [None] is always returned.
  /// This function is a placeholder for further development.   
  fn parse_extension_elements(&self, _: &Node) -> Option<ExtensionElement> {
    None
  }

  /// Parses extension attributes.
  /// Currently extension elements are omitted and [None] is always returned.
  /// This function is a placeholder for further development.   
  fn parse_extension_attributes(&self, _: &Node) -> Vec<ExtensionAttribute> {
    vec![]
  }

  /// Returns boolean value of the specified attribute.
  fn parse_boolean_attribute(&self, node: &Node, attr_name: &str, default_value: bool) -> bool {
    if let Some(attr_value) = node.attribute(attr_name) {
      attr_value == "true"
    } else {
      default_value
    }
  }

  /// Returns the value of the hit policy attribute.
  fn parse_hit_policy_attribute(&self, node: &Node) -> Result<HitPolicy> {
    if let Some(hit_policy_text) = node.attribute(ATTR_HIT_POLICY) {
      match hit_policy_text.trim() {
        "UNIQUE" => Ok(HitPolicy::Unique),
        "ANY" => Ok(HitPolicy::Any),
        "PRIORITY" => Ok(HitPolicy::Priority),
        "FIRST" => Ok(HitPolicy::First),
        "RULE ORDER" => Ok(HitPolicy::RuleOrder),
        "OUTPUT ORDER" => Ok(HitPolicy::OutputOrder),
        "COLLECT" => Ok(HitPolicy::Collect(self.parse_aggregation_attribute(node)?)),
        other => Err(invalid_hit_policy(other)),
      }
    } else {
      Ok(HitPolicy::Unique)
    }
  }

  /// Returns the value of the aggregation attribute.
  fn parse_aggregation_attribute(&self, node: &Node) -> Result<BuiltinAggregator> {
    if let Some(aggregation_text) = node.attribute(ATTR_AGGREGATION) {
      match aggregation_text.trim() {
        "COUNT" => Ok(BuiltinAggregator::Count),
        "SUM" => Ok(BuiltinAggregator::Sum),
        "MIN" => Ok(BuiltinAggregator::Min),
        "MAX" => Ok(BuiltinAggregator::Max),
        other => Err(invalid_aggregation(other)),
      }
    } else {
      Ok(BuiltinAggregator::List)
    }
  }

  /// Returns the value of the preferred decision table orientation attribute.
  fn parse_preferred_orientation_attribute(&self, node: &Node) -> Result<DecisionTableOrientation> {
    if let Some(attr_value) = node.attribute(ATTR_PREFERRED_ORIENTATION) {
      DecisionTableOrientation::try_from(attr_value)
    } else {
      Ok(DecisionTableOrientation::RuleAsRow)
    }
  }

  /// Returns required name attribute for specified node.
  fn required_name(&self, node: &Node) -> Result<String> {
    let name = required_attribute(node, ATTR_NAME)?;
    Ok(name)
  }

  /// Parse DMNDI part of the diagram definitions.
  ///
  /// # Arguments
  ///
  /// - node - definitions node.
  ///
  fn parse_dmndi(&self, node: &Node) {
    println!("HERE IS THE PLACE TO START PARSING DMNDI");
    for child_node in node.children().filter(|n| n.tag_name().name() == NODE_DMNDI) {
      println!("{:?}", child_node.tag_name())
    }
  }
}

/// Utility helper functions for processing XML structures.
mod xml_utils {
  use super::errors::*;
  use super::ATTR_HREF;
  use dmntk_common::{OptHRef, Result};
  use roxmltree::Node;

  /// Returns the value of the required attribute.
  pub fn required_attribute(node: &Node, attr_name: &str) -> Result<String> {
    if let Some(attr_value) = node.attribute(attr_name) {
      Ok(attr_value.to_string())
    } else {
      Err(xml_expected_mandatory_attribute(node_name_pos(node), attr_name.to_string()))
    }
  }

  /// Returns the value of the optional attribute.
  pub fn optional_attribute(node: &Node, attr_name: &str) -> Option<String> {
    node.attribute(attr_name).map(|attr_value| attr_value.to_string())
  }

  /// Returns required textual content of the node.
  pub fn required_content(node: &Node) -> Result<String> {
    if let Some(text) = node.text() {
      Ok(text.to_string())
    } else {
      Err(xml_expected_mandatory_text_content(node.tag_name().name().to_string()))
    }
  }

  /// Returns optional textual content of the node.
  pub fn optional_content(node: &Node) -> Option<String> {
    node.text().map(|text| text.to_string())
  }

  /// Returns required child node or raises an error when there is no child with given name.
  pub fn required_child<'a>(node: &'a Node, child_name: &str) -> Result<Node<'a, 'a>> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      Ok(child_node)
    } else {
      Err(required_child_node_is_missing(node_name_pos(node), child_name))
    }
  }

  /// Returns child node when there is a child with the given name.
  pub fn optional_child<'a>(node: &'a Node, child_name: &str) -> Option<Node<'a, 'a>> {
    node.children().find(|n| n.tag_name().name() == child_name)
  }

  /// Returns the required text content of the required child node.
  pub fn required_child_required_content(node: &Node, child_name: &str) -> Result<String> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      required_content(&child_node)
    } else {
      Err(xml_expected_mandatory_child_node(node_name_pos(node), child_name.to_string()))
    }
  }

  /// Returns the required content of the optional child node.
  pub fn optional_child_required_content(node: &Node, child_name: &str) -> Result<Option<String>> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      Ok(Some(required_content(&child_node)?))
    } else {
      Ok(None)
    }
  }

  /// Returns the optional content of the optional child node.
  pub fn optional_child_optional_content(node: &Node, child_name: &str) -> Option<String> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      optional_content(&child_node)
    } else {
      None
    }
  }

  /// Returns the required attribute of the optional child node.
  pub fn optional_child_required_href(node: &Node, child_name: &str) -> Result<OptHRef> {
    if let Some(child_node) = node.children().find(|n| n.tag_name().name() == child_name) {
      Ok(Some(required_attribute(&child_node, ATTR_HREF)?.into()))
    } else {
      Ok(None)
    }
  }

  /// XML utility function that returns node's name with node's position in the original document.
  pub fn node_name_pos(n: &Node) -> String {
    format!("`{}` at [{}]", n.tag_name().name().to_string(), n.document().text_pos_at(n.range().start))
  }
}

/// Definitions of errors raised while parsing the XML model.
pub mod errors {
  use dmntk_common::DmntkError;

  const ACCEPTED_FUNCTION_KIND: &[&str] = &["FEEL", "Java", "PMML"];
  const ACCEPTED_HIT_POLICY: &[&str] = &["UNIQUE", "FIRST", "PRIORITY", "ANY", "COLLECT", "RULE ORDER", "OUTPUT ORDER"];
  const ACCEPTED_AGGREGATION: &[&str] = &["COUNT", "SUM", "MIN", "MAX"];

  /// Errors related with parsing the decision model.
  #[derive(Debug, PartialEq)]
  pub enum ModelParserError {
    /// Raised when parsed text is not a valid function kind,
    /// accepted values are: `FEEL`, `Java` or `PMML`.
    InvalidFunctionKind(String),
    /// Raised when parsed text is not a valid hit policy,
    /// accepted values are: `UNIQUE`, `FIRST`, `PRIORITY`,
    /// `ANY`, `COLLECT`, `RULE ORDER`, or `OUTPUT ORDER`.
    InvalidHitPolicy(String),
    /// Raised when parsed text is not a valid aggregation for hit policy,
    /// accepted values are: `COUNT`, `SUM`, `MIN`, or `MAX`.
    InvalidAggregation(String),
    /// Raised when required `inputExpression` node is missing.
    RequiredInputExpressionIsMissing,
    /// Raised when required child node is missing.
    RequiredChildNodeIsMissing(String, String),
    /// Raised when required expression instance is missing.
    RequiredExpressionInstanceIsMissing,
    /// Raised when the number of elements in a row differs from the number of columns in relation.
    NumberOfElementsInRowDiffersFromNumberOfColumns,
    ///
    XmlParsingModelFailed(String),
    ///
    XmlUnexpectedNode(String, String),
    ///
    XmlExpectedMandatoryAttribute(String, String),
    ///
    XmlExpectedMandatoryChildNode(String, String),
    ///
    XmlExpectedMandatoryTextContent(String),
  }

  impl From<ModelParserError> for DmntkError {
    fn from(e: ModelParserError) -> Self {
      DmntkError::new("ModelParserError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for ModelParserError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        ModelParserError::InvalidFunctionKind(s) => {
          write!(f, "'{}' is not a valid function kind, accepted values are: {:?}", s, ACCEPTED_FUNCTION_KIND)
        }
        ModelParserError::InvalidHitPolicy(s) => {
          write!(f, "'{}' is not a valid hit policy, allowed values are: {:?}", s, ACCEPTED_HIT_POLICY)
        }
        ModelParserError::InvalidAggregation(s) => {
          write!(f, "'{}' is not a valid aggregation, allowed values are: {:?}", s, ACCEPTED_AGGREGATION)
        }
        ModelParserError::RequiredInputExpressionIsMissing => {
          write!(f, "required input expression in decision table's input clause is missing")
        }
        ModelParserError::RequiredChildNodeIsMissing(parent_node, child_node) => {
          write!(f, "required child node '{}' in parent node '{}' is missing", child_node, parent_node)
        }
        ModelParserError::RequiredExpressionInstanceIsMissing => {
          write!(f, "required expression instance in context entry is missing")
        }
        ModelParserError::NumberOfElementsInRowDiffersFromNumberOfColumns => {
          write!(f, "number of elements in a row differs from the number of columns defined in a relation")
        }
        ModelParserError::XmlParsingModelFailed(s) => {
          write!(f, "parsing model from XML failed with reason: {}", s)
        }
        ModelParserError::XmlUnexpectedNode(expected, actual) => {
          write!(f, "unexpected XML node, expected: {}, actual: {}", expected, actual)
        }
        ModelParserError::XmlExpectedMandatoryAttribute(node, expected_attribute) => {
          write!(f, "expected value for mandatory attribute `{}` in node `{}`", expected_attribute, node)
        }
        ModelParserError::XmlExpectedMandatoryChildNode(parent_node, expected_child_node) => {
          write!(f, "expected mandatory child node '{}' in parent node '{}'", expected_child_node, parent_node)
        }
        ModelParserError::XmlExpectedMandatoryTextContent(expected) => {
          write!(f, "expected mandatory text content in node: {}", expected)
        }
      }
    }
  }

  pub fn invalid_function_kind(text: &str) -> DmntkError {
    ModelParserError::InvalidFunctionKind(text.to_string()).into()
  }

  pub fn invalid_hit_policy(text: &str) -> DmntkError {
    ModelParserError::InvalidHitPolicy(text.to_string()).into()
  }

  pub fn invalid_aggregation(text: &str) -> DmntkError {
    ModelParserError::InvalidAggregation(text.to_string()).into()
  }

  pub fn required_child_node_is_missing(parent_node: String, child_node: &str) -> DmntkError {
    ModelParserError::RequiredChildNodeIsMissing(parent_node, child_node.to_string()).into()
  }

  pub fn required_input_expression_is_missing() -> DmntkError {
    ModelParserError::RequiredInputExpressionIsMissing.into()
  }

  pub fn required_expression_instance_is_missing() -> DmntkError {
    ModelParserError::RequiredExpressionInstanceIsMissing.into()
  }

  pub fn number_of_elements_in_row_differs_from_number_of_columns() -> DmntkError {
    ModelParserError::NumberOfElementsInRowDiffersFromNumberOfColumns.into()
  }

  pub fn xml_parsing_model_failed(reason: String) -> DmntkError {
    ModelParserError::XmlParsingModelFailed(reason).into()
  }

  pub fn xml_unexpected_node(expected: &str, actual: String) -> DmntkError {
    ModelParserError::XmlUnexpectedNode(expected.to_string(), actual).into()
  }

  pub fn xml_expected_mandatory_attribute(node: String, expected_attribute: String) -> DmntkError {
    ModelParserError::XmlExpectedMandatoryAttribute(node, expected_attribute).into()
  }

  pub fn xml_expected_mandatory_child_node(parent_node: String, expected_child_node: String) -> DmntkError {
    ModelParserError::XmlExpectedMandatoryChildNode(parent_node, expected_child_node).into()
  }

  pub fn xml_expected_mandatory_text_content(expected: String) -> DmntkError {
    ModelParserError::XmlExpectedMandatoryTextContent(expected).into()
  }
}
